﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * nn::spy の開発者が実装作業に利用するアプリケーションです。
 */

#include <nns/nns_Log.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/os.h>
#include <nn/settings.h>
#include <nn/settings/settings_DebugPad.h>

#include <nn/spy/spy_Config.h>

#include "SpySandbox.h"

//-----------------------------------------------------------------------------

namespace SpySandbox {

    const nn::hid::NpadIdType g_NpadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld,
    };

    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

    nn::hid::DebugPadButtonSet g_Button;
    nn::hid::DebugPadButtonSet g_LastButton;

    const nn::hid::DebugPadButtonSet MapNpadButtonSet(const nn::hid::NpadButtonSet& buttons)
    {
        static const struct {
            nn::hid::NpadButtonSet from;
            nn::hid::DebugPadButtonSet to;
        } buttonMapList[] = {
            {
                nn::hid::NpadButton::A::Mask,
                nn::hid::DebugPadButton::A::Mask,
            },
            {
                nn::hid::NpadButton::B::Mask,
                nn::hid::DebugPadButton::B::Mask,
            },
            {
                nn::hid::NpadButton::X::Mask,
                nn::hid::DebugPadButton::X::Mask,
            },
            {
                nn::hid::NpadButton::Y::Mask,
                nn::hid::DebugPadButton::Y::Mask,
            },
            {
                nn::hid::NpadButton::L::Mask,
                nn::hid::DebugPadButton::L::Mask,
            },
            {
                nn::hid::NpadButton::R::Mask,
                nn::hid::DebugPadButton::R::Mask,
            },
            {
                nn::hid::NpadButton::Left::Mask,
                nn::hid::DebugPadButton::Left::Mask,
            },
            {
                nn::hid::NpadButton::Right::Mask,
                nn::hid::DebugPadButton::Right::Mask,
            },
            {
                nn::hid::NpadButton::Up::Mask,
                nn::hid::DebugPadButton::Up::Mask,
            },
            {
                nn::hid::NpadButton::Down::Mask,
                nn::hid::DebugPadButton::Down::Mask,
            },
            {
                nn::hid::NpadButton::Plus::Mask,
                nn::hid::DebugPadButton::Start::Mask,
            },
        };

        nn::hid::DebugPadButtonSet result = {};

        for (const auto& buttonMap : buttonMapList)
        {
            if ((buttons & buttonMap.from) == buttonMap.from)
            {
                result |= buttonMap.to;
            }
        }

        return result;
    }

    void UpdatePadStatus()
    {
        g_LastButton = g_Button;

        {
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            g_Button = state.buttons;
        }

        for (auto id : g_NpadIds)
        {
            const auto styles = nn::hid::GetNpadStyleSet(id);

            if (styles.Test<nn::hid::NpadStyleFullKey>())
            {
                nn::hid::NpadFullKeyState state;
                nn::hid::GetNpadState(&state, id);
                g_Button |= MapNpadButtonSet(state.buttons);
            }

            if (styles.Test<nn::hid::NpadStyleHandheld>())
            {
                nn::hid::NpadHandheldState state;
                nn::hid::GetNpadState(&state, id);
                g_Button |= MapNpadButtonSet(state.buttons);
            }
        }
    }

    const nn::hid::DebugPadButtonSet GetButtonDown()
    {
        return ~g_LastButton & g_Button;
    }

    const nn::hid::DebugPadButtonSet GetButtonState()
    {
        return g_Button;
    }
}

namespace {

    void InitializeHid()
    {
        nn::hid::InitializeDebugPad();

        //キーボードのキーを DebugPad のボタンに割り当てます。
        nn::settings::DebugPadKeyboardMap map;
        nn::settings::GetDebugPadKeyboardMap(&map);
        map.buttonA = nn::hid::KeyboardKey::A::Index;
        map.buttonB = nn::hid::KeyboardKey::B::Index;
        map.buttonX = nn::hid::KeyboardKey::X::Index;
        map.buttonY = nn::hid::KeyboardKey::Y::Index;
        map.buttonL = nn::hid::KeyboardKey::L::Index;
        map.buttonR = nn::hid::KeyboardKey::R::Index;
        map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
        map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
        map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
        map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
        map.buttonStart = nn::hid::KeyboardKey::Space::Index;
        nn::settings::SetDebugPadKeyboardMap(map);

        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::hid::SetSupportedNpadIdType(SpySandbox::g_NpadIds, SpySandbox::NpadIdCountMax);
    }

    void Mainloop()
    {
        NNS_LOG(
            "-----------------------------\n"
            " SpySandbox:\n"
            "\n"
            " [A]              AudioPerformanceMetrics.\n"
            " [B]              PlotModule.\n"
            " [X]              AtkSoundProfile.\n"
            " [Up] + [+/Start][Space]  exit.\n"
            "-----------------------------\n"
            );

        while (NN_STATIC_CONDITION(true))
        {
            SpySandbox::UpdatePadStatus();

            nn::hid::DebugPadButtonSet buttonDown = SpySandbox::GetButtonDown();
            nn::hid::DebugPadButtonSet buttonState = SpySandbox::GetButtonState();

            if (buttonDown.Test<nn::hid::DebugPadButton::A>())
            {
                AudioPerformanceMetrics();
                return;
            }

            if (buttonDown.Test<nn::hid::DebugPadButton::B>())
            {
                PlotModule();
                return;
            }

            if (buttonDown.Test<nn::hid::DebugPadButton::X>())
            {
                AtkSoundProfile();
                return;
            }

            if (buttonState.Test<nn::hid::DebugPadButton::Start>() &&
                buttonState.Test<nn::hid::DebugPadButton::Up>())
            {
                NNS_LOG("[Up]+[Start] Exit.\n");
                return;
            }

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    InitializeHid();

    Mainloop();
}

