﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

#include <nn/atk.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nns/atk/atk_SampleCommon.h>

#include "Atk.h"
#include "Common.fsid"

//-----------------------------------------------------------------------------

namespace {

    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForLabelStringData;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

}

void InitializeAtk() NN_NOEXCEPT
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    // パフォーマンス計測のために、プロファイラを有効にします。
    param.enableProfiler = true;
    // 波形取得のために、CircularBufferSink を有効にします。
    param.enableCircularBufferSink = true;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
    g_pMemoryForSoundSystem = nns::atk::Allocate(memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>(g_pMemoryForSoundSystem),
        memSizeForSoundSystem);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundSystem");

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate(SoundHeapSize);
    isSuccess = g_SoundHeap.Create(g_pMemoryForSoundHeap, SoundHeapSize);
    NN_ABORT_UNLESS(isSuccess, "cannot create SoundHeap");

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS(isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath);

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize);
    isSuccess = g_SoundArchive.LoadHeader(g_pMemoryForInfoBlock, infoBlockSize);
    NN_ABORT_UNLESS(isSuccess, "cannot load InfoBlock");

    // 文字列テーブルの初期化
    std::size_t memSizeForLabelStringData = g_SoundArchive.GetLabelStringDataSize();
    g_pMemoryForLabelStringData = nns::atk::Allocate(memSizeForLabelStringData);
    isSuccess = g_SoundArchive.LoadLabelStringData(g_pMemoryForLabelStringData, memSizeForLabelStringData);
    NN_ABORT_UNLESS(isSuccess, "cannnot load LabelStringData");

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize(&g_SoundArchive);
    g_pMemoryForSoundDataManager = nns::atk::Allocate(memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize);
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundDataManager");

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize(&g_SoundArchive);
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize(&g_SoundArchive);
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate(memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundArchivePlayer");
}

void FinalizeAtk() NN_NOEXCEPT
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForLabelStringData);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadAtkData() NN_NOEXCEPT
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData(SEQ_MARIOKART, &g_SoundHeap);
    NN_ABORT_UNLESS(isSuccess, "LoadData(SEQ_MARIOKART) failed.");

    isSuccess = g_SoundDataManager.LoadData(SE_YOSHI, &g_SoundHeap);
    NN_ABORT_UNLESS(isSuccess, "LoadData(SE_YOSHI) failed.");
}

static bool PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId)
{
    g_SoundHandle.Stop(0);

    return g_SoundArchivePlayer.StartSound(&g_SoundHandle, soundId).IsSuccess();
}

static bool PlayWithHoldSound(nn::atk::SoundArchive::ItemId soundId)
{
    return g_SoundArchivePlayer.HoldSound(&g_SoundHandleHold, soundId).IsSuccess();
}

bool StartAtkSequenceSound() NN_NOEXCEPT
{
    return PlayWithStartSound(SEQ_MARIOKART);
}

bool StartAtkWaveSound() NN_NOEXCEPT
{
    return PlayWithStartSound(SE_YOSHI);
}

bool StartAtkStreamSound() NN_NOEXCEPT
{
    return PlayWithStartSound(STRM_MARIOKART);
}

bool HoldAtkSequenceSound() NN_NOEXCEPT
{
    return PlayWithHoldSound(SEQ_MARIOKART);
}

bool HoldAtkWaveSound() NN_NOEXCEPT
{
    return PlayWithHoldSound(SE_YOSHI);
}

bool HoldAtkStreamSound() NN_NOEXCEPT
{
    return PlayWithHoldSound(STRM_MARIOKART);
}

void StopAtkSound() NN_NOEXCEPT
{
    g_SoundHandle.Stop(0);
}

void UpdateAtkSoundArchivePlayer() NN_NOEXCEPT
{
    g_SoundArchivePlayer.Update();
}

nn::atk::SoundArchivePlayer& GetAtkSoundArchivePlayer() NN_NOEXCEPT
{
    return g_SoundArchivePlayer;
}

size_t GetCircularBufferSinkBufferSize() NN_NOEXCEPT
{
    return nn::atk::SoundSystem::GetCircularBufferSinkBufferSize();
}

size_t ReadCircularBufferSink(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return nn::atk::SoundSystem::ReadCircularBufferSink(buffer, bufferSize);
}
