﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SocketResolver.cpp,PageSampleSocketResolver}
 *
 * @brief
 * Simple communication with the resolver via the socket library
 */

/**
 * @page PageSampleSocketResolver demonstrate using the resolver via the socket library
 * @tableofcontents
 *
 * @brief
 * Documentation about the sample resolver program of the socket library
 *
 * @section PageSampleSocketResolver_SectionBrief Overview
 * First this sample prepares network connection by using the NIFM library. If network can be used,
 * it queries for the provided hostname or IP via
 *
 * - nn::socket::GetHostByName
 * - nn::socket::GetHostByAddr
 * - nn::socket::GetAddrInfo
 * - nn::socket::GetNameInfo
 *
 * Furthermore, GetHostByName & GetHostByAddr were earlier resolver standards whereas
 * and GetAddrInfo & GetNameInfo are from later standards. The earlier standards
 * differ from the latter on matters of semantics and use contracts. For example, you do not
 * free the structure returned by GetHostByName but you do free the structure returned by.
 * GetAddrInfo.
 *
 * Proper use is documented below, namely that
 *
 * Earlier:
 * - The hostent pointer returned by is a global value and may change between calls.
 * - To avoid race conditions, the hostent structure returned by GetHostByName is thread-localized;
 *   simply referring to a global hostent is not supported.
 * - To preserve the hostent the developer must copy it before calling GetHostByName or GetAddrInfo.
 * - The developer must not free the structure returned by GetHostByName, it is managed internally.
 * - errors are provided by the h_errno facility by dereferencing nn::socket::GetHError.
 * - strings provided by nn::socket::HStrError()
 *
 * Later:
 * - GetAddrInfo is inherently re-enterant and therefore does not use thread-localized values.
 * - The addrinfo pointer is is allocated from the socket heap.
 * - The developer must free it back to the heap via FreeAddrInfo
 * - The return value from GetAddrInfo & GetNameInfo is a EAI_* code, with 0 as success (see netdb.h)
 * - Error to string constants are provided by nn::socket::GAIStrError
 *
 * @section PageSampleSocketResolver_SectionFileStoructure File Structure
 * This sample program can be found at
 * @link ../../../Samples/Sources/Applications/SocketResolver
 * Samples/Sources/Applications/SocketResolver @endlink.
 *
 * @section PageSampleSocketResolver_SectionNecessaryEnvironment System Requirements
 * It is required to import network setting by SettingManager prior to running this sample.
 * Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 * By default this sample sends HTTP GET request to example.com on port 443 thus Internet
 * connection is required. Please pass desired host name in the command line argument to run it
 * with other servers. Note that "https://" should not be included when passing the host name in
 * the argument because this sample doesn't parse it.
 *
 * @section PageSampleSocketResolver_SectionHowToOperate Operation Procedure
 * None.
 *
 * @section PageSampleSocketResolver_SectionPrecaution Precautions
 * None.
 *
 * @section PageSampleSocketResolver_SectionHowToExecute Execution Procedure
 * Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleSocketResolver_SectionDetail Detail
 * Here is the sequence of this sample program.
 *
 * - Sends request to use network connection by the NIFM library
 * - Initializes the socket library
 * - Resolves host name or IP address, on error use appropriate error constant and string function, print results
 * - Finalizes the socket library and the SSL library
 * - Finalizes the NIFM library
 *
 * Here is the expected output log of this sample program. Please note that it will not be printed
 * in the Release build.
 *
 * @verbinclude SocketResolver_Output.txt
 */

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nifm.h>
#include <nn/socket.h>

namespace
{
    /** @brief Socket memory pool, initialized in nnMain */
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
    const char* g_NOT_FOUND = "not found";

    /**
     * @brief utility function to print a host entry structure to the console
     * Used by DemonstrateGetHostByName and DemonstrateGetHostByAddr
     * @param pHostEntry a pointer to a hostent structure
     */
    void PrintHostEntry(nn::socket::HostEnt* pHostEntry)
    {
        if (nullptr == pHostEntry)
        {
            NN_LOG("ERROR: pHostEntry is nullptr\n");
            return;
        };

        NN_LOG("Host name: %s\n", pHostEntry->h_name);
        nn::socket::InAddr** pInternetAddressList = reinterpret_cast<nn::socket::InAddr **>(pHostEntry->h_addr_list);
        if (nullptr == pInternetAddressList)
        {
            NN_LOG(g_NOT_FOUND);
        }
        else
        {
            for(int index = 0; pInternetAddressList[index] != nullptr; index++)
            {
                NN_LOG("IP address: %s\n", nn::socket::InetNtoa(*pInternetAddressList[index]));
            };
        };
    };

    /**
     * @brief utility function to print an addrinfo structure entry to the console
     * Used by DemonstrateGetAddrInfo and DemonstrateGetNameInfo
     * @param pAddrInfoIn a pointer to an addrinfo structure
     */
    void PrintAddressInformation(nn::socket::AddrInfo* pAddrInfoIn)
    {
        if (nullptr == pAddrInfoIn)
        {
            NN_LOG("ERROR: pAddrInfoIn is nullptr\n");
            return;
        };

        nn::socket::SockAddrIn* pSin = nullptr;
        for (nn::socket::AddrInfo* pAddrinfo = pAddrInfoIn; pAddrinfo != nullptr; pAddrinfo = pAddrinfo->ai_next)
        {
            /* only nn::socket::Family::Af_Inet (IPv4) addresses are supported */
            if (nn::socket::Family::Af_Inet == pAddrinfo->ai_family)
            {
                /* a canonical name name (or CNAME) is a DNS resource record
                 * that uses a host name as an alias to another host name */
                if (pAddrinfo->ai_canonname != nullptr)
                {
                    NN_LOG("Canonical Name: %s\n", pAddrinfo->ai_canonname);
                };

                pSin = reinterpret_cast<nn::socket::SockAddrIn*>(pAddrinfo->ai_addr);
                NN_LOG("IP address: %s\n", pSin != nullptr ? nn::socket::InetNtoa(pSin->sin_addr) : g_NOT_FOUND);
            }
            else
            {
                NN_LOG("Unsupported address type (%d) returned\n", pAddrinfo->ai_family);
            }
        };
    };

    /**
     * @brief demonstrates basic use of nn::socket::GetHostByName, look up an address for a hostname
     * @param pHostnameString the string that represents a hostname (i.e. www.nintendo.com)
     */
    void DemonstrateGetHostByName(const char* pHostnameString)
    {
        NN_LOG("============ DemonstrateGetHostByName(%s) ============\n", pHostnameString);
        nn::socket::HostEnt* pHostEntry = nullptr;

        /* error case */
        pHostEntry = nn::socket::GetHostEntByName(pHostnameString);
        if (nullptr == pHostEntry)
        {
            /* on errors GetHostByName & GetHostByAddr return host-specific error in the
             * host errno: nn::socket::GetHError */
            nn::socket::HErrno hErrno = *nn::socket::GetHError();
            const char* hErrnoString = nn::socket::HStrError(hErrno);

            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            const char* errnoString = strerror(static_cast<int>(errorNumber));

            /* you can display host errno strings for GetHostByName & GetHostByAddr
             * by using nn::socket::HStrError */
            NN_LOG("Error, nn::socket::GetHostByName(%s) failed; "
                   "resolver h_errno: %s (%d), "
                   "errno: %s (%d) "
                   "\n",
                   pHostnameString == nullptr ? "(null)" : pHostnameString,
                   hErrnoString    == nullptr ? "(null)" : hErrnoString, hErrno,
                   errnoString     == nullptr ? "(null)" : errnoString, errorNumber);
        }
        else
        {
            /* print on success */
            PrintHostEntry(pHostEntry);
        };
    };

    /**
     * @brief demonstrates basic use of nn::socket::GetHostByAddr
     * @param pIpAddressString the ip address used to look up a host name
     */
    void DemonstrateGetHostByAddr(const char* pIpAddressString)
    {
        NN_LOG("============ DemonstrateGetHostByAddr(%s) ============\n", pIpAddressString);
        nn::socket::HostEnt* pHostEntry = nullptr;
        nn::socket::SockAddrIn sa = nn::socket::SockAddrIn();

        /* GetHostByAddr takes nn::socket::SockAddr as the first argument and so we convert the string
         * using nn::socket::InetPton; -1 indicates a malformed address */
        if ( 1 != nn::socket::InetPton(nn::socket::Family::Af_Inet, pIpAddressString, &(sa.sin_addr)) )
        {
            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            NN_LOG("Error, nn::socket::InetPton failed; errno: %d\n", errorNumber);
        }
        /* error case */
        else if (nullptr == (pHostEntry = nn::socket::GetHostEntByAddr(&(sa.sin_addr),
                                                                     sizeof(nn::socket::InAddr),
                                                                     nn::socket::Family::Af_Inet)))
        {
            /* on errors GetHostByName & GetHostByAddr return host-specific error in the
             * host errno: nn::socket::GetHError */

            nn::socket::HErrno hErrno = *nn::socket::GetHError();
            const char* hErrnoString = nn::socket::HStrError(hErrno);

            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            const char* errnoString = strerror(static_cast<int>(errorNumber));

            /* you can display host errno strings for GetHostByName & GetHostByAddr
             * by using nn::socket::HStrError */
            NN_LOG("Error, nn::socket::GetHostByAddr(%s) failed; "
                   "resolver h_errno: %s (%d) "
                   "errno: %s (%d) "
                   "\n",
                   pIpAddressString == nullptr ? "(null)" : pIpAddressString,
                   hErrnoString     == nullptr ? "(null)" : hErrnoString, hErrno,
                   errnoString      == nullptr ? "(null)" : errnoString, errorNumber);
        }
        else
        {
            /* print on success */
            PrintHostEntry(pHostEntry);
        };
    };

    /**
     * @brief demonstrates basic use of nn::socket::GetAddrInfo
     * @param pHostnameOrIPString is a string representation of a hostname or address
     * the most common use-case for GetAddrInfo is hostname-to-IP address lookup, but
     * it is also possible to find information about an IP address using GetAddrInfo
     */
    void DemonstrateGetAddrInfo(const char* pHostnameOrIPString)
    {
        NN_LOG("============ DemonstrateGetAddrInfo(%s) ============\n", pHostnameOrIPString);
        nn::socket::AiErrno rc = nn::socket::AiErrno::EAi_Success; // was -1;
        nn::socket::AddrInfo hints = nn::socket::AddrInfo();
        nn::socket::AddrInfo* pAddressInfoResult = nullptr;
        memset(&hints, 0, sizeof hints);
        hints.ai_family = nn::socket::Family::Af_Inet;
        hints.ai_socktype = nn::socket::Type::Sock_Stream;

        /* On success GetAddrInfo & GetNameInfo return 0, and on error they return a
         *  non-zero EAI error code described in netdb.h */
        if (nn::socket::AiErrno::EAi_Success != (rc = nn::socket::GetAddrInfo(pHostnameOrIPString,
                                                                              nullptr,
                                                                              &hints,
                                                                              &pAddressInfoResult)))
        {
            const char* gaiErrorString = nn::socket::GAIStrError(rc);

            nn::socket::HErrno hErrno = *nn::socket::GetHError();
            const char* hErrnoString = nn::socket::HStrError(hErrno);

            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            const char* errnoString = strerror(static_cast<int>(errorNumber));

            /* the EAI error code can be translated into a string result */
            NN_LOG("Error, nn::socket::GetAddrInfo(%s) failed; "
                   "resolver gai result: %s (%d) "
                   "resolver h_errno: %s (%d) "
                   "errno: %s (%d) "
                   "\n",
                   pHostnameOrIPString == nullptr ? "(null)" : pHostnameOrIPString,
                   gaiErrorString      == nullptr ? "(null)" : gaiErrorString, rc,
                   hErrnoString        == nullptr ? "(null)" : hErrnoString, hErrno,
                   errnoString         == nullptr ? "(null)" : errnoString, errorNumber);
        }
        else if (nullptr == pAddressInfoResult)
        {
            /** GetAddrInfo should always return a result on success */
            NN_LOG("nn::socket::GetAddrInfo pAddressInfoResult was nullptr\n");
            NN_SDK_ASSERT(false);
        }
        else
        {
            /* print on success */
            PrintAddressInformation(pAddressInfoResult);
        };

        /* if resources were allocated then we free them back to the system */
        if (nullptr != pAddressInfoResult)
        {
            nn::socket::FreeAddrInfo(pAddressInfoResult);
            pAddressInfoResult = nullptr;
        }
    };

    /**
     * @brief demonstrates basic use of nn::socket::GetNameInfo
     * @param pIpAddressString a dotted-decimal IP address string
     */
    void DemonstrateGetNameInfo(const char* pIpAddressString)
    {
        NN_LOG("============ DemonstrateGetNameInfo(%s) ============\n", pIpAddressString);
        nn::socket::AiErrno rc = nn::socket::AiErrno::EAi_Success; // was false;
        nn::socket::SockAddrIn sa = nn::socket::SockAddrIn();
        char hostnameBuffer[nn::socket::Ni_MaxHost] = { '\0' };
        char servicenameBuffer[nn::socket::Ni_MaxServ] = { '\0' };
        sa.sin_family = nn::socket::Family::Af_Inet;

        /* hard-code service name for demonstration purposes */
        sa.sin_port = nn::socket::InetHtons(80);

        /* GetNameInfo takes nn::socket::SockAddr as the first argument and so we convert the string
         * using nn::socket::InetPton; -1 indicates a malformed address */
        if (1 != nn::socket::InetPton(nn::socket::Family::Af_Inet, pIpAddressString, &(sa.sin_addr)))
        {
            nn::socket::Errno errorNumber = nn::socket::GetLastError();
            NN_LOG("Error, nn::socket::InetPton failed; errno: %d\n", errorNumber);
        }
        /* error case */
        else if (nn::socket::AiErrno::EAi_Success != (rc = nn::socket::GetNameInfo(reinterpret_cast<const nn::socket::SockAddr*>(&sa),
                                                                              static_cast<nn::socket::SockLenT>(sizeof(nn::socket::SockAddr)),
                                                                              hostnameBuffer, static_cast<nn::socket::SockLenT>(sizeof(hostnameBuffer)),
                                                                              servicenameBuffer, static_cast<nn::socket::SockLenT>(sizeof(servicenameBuffer)),
                                                                              static_cast<nn::socket::NameInfoFlag>(0))))
        {
            /* the EAI error code can be translated into a string result */
            NN_LOG("nn::socket::GetNameInfo error: %s (%d)\n", nn::socket::GAIStrError(rc), rc);
        }
        /* print host & service on success */
        else
        {
            NN_LOG("  Host: %s\n", hostnameBuffer);
            NN_LOG("  Service: %s\n", servicenameBuffer);
        };
    };

} // END Anonymous namespace


//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------

/**
 * @brief nnMain
 * This function runs the program.
 *
 * First: the sample acquires pHostname and pIpAddress from nn::os::GetHostArgv();
 * the hostname argument is used for address record lookups and the ip address
 * argument is used for reverse lookups. Please note that these can be any
 * arbitrary hostname and ip address; for example: www.nintendo.com 205.166.76.26.
 * Also, please refer to SocketResolver_Output.txt for an example of what output
 * might look like as results may vary between different DNS configurations.
 *
 * Second: the program acquires network access via NIFM and socket layers. If this
 * is successful then the program continues. On the other hand, if it fails then
 * we proceed to clean up and exit.
 *
 * Third: We demonstrate a common use of the resolver API's using the arguments; please note that
 * if a canonical name (a hostname to hostname alias) is available then
 * PrintAddressInformation will display this information as well (see AI_CANONNAME flag)
 *
 * - DemonstrateGetHostByName(host): find the ip address from the provided host name
 * - DemonstrateGetHostByAddr(ip): find the host name for the provided IP address
 * - DemonstrateGetAddrInfo(host): find the host name for a given ip address
 * - DemonstrateGetAddrInfo(ip): find the IP address for a given host name
 * - DemonstrateGetNameInfo(ip) find the host and service name for a given ip address
 * -- Please note that the port is hard-coded to 80 (http) for service name demonstration purposes.
 *
 * Each of these functions demonstrate a basic use case, specifically:
 * - how to populate the function call arguments and call the function
 * - how to check for errors or success
 * - on success, how to interact with the returned API data structure
 * - if necessary, how to free resources back to the system
 *
 * Finally, we clean up and exit the program.
 */
extern "C" void nnMain()
{
    /* result for socket initialize / error description */
    nn::Result ret = nn::Result();

    /* flag value for cleanup: did we acquire access to NIFM?*/
    bool shouldFinalizeNifm = false;

    /* flag value for cleanup: did we acquire access to nn::socket */
    bool shouldFinalizeSocket = false;

    /* the host name argument used for Address-record lookups:
     * if your DNS server has access to the public internet then you may use any hostname (i.e. www.nintendo.com)
     * otherwise please contact your network administrator for a hostname valid on your private network */
    const char* pHostname = nullptr;

    /* the ip address argument used for reverse (ip-to-host or PTR) record lookups:
     * if your DNS server has access to the public internet then you may use any ip address (i.e. 205.166.76.26)
     * otherwise please contact your network administrator for an IP valid on your private network */
    const char* pIpAddress = nullptr;

    /* the argument vector is simply populated with two strings, hostName and ipForReverseLookups
     * an example string passed to the "argument" field in target manager would be:
     * www.nintendo.com 205.166.76.26  */
    if( nn::os::GetHostArgc() < 3 )
    {
        NN_LOG("\nError: Must pass parameter <hostName> <ipForReverseLookups>\n");
        goto Exit;
    }
    pHostname = nn::os::GetHostArgv()[1];
    pIpAddress = nn::os::GetHostArgv()[2];

    /* since the following will utilize the system network interface, we must initialize
     * network interface manager (NIFM) */
    nn::nifm::Initialize();

    /* this application is now requesting to use network interface */
    nn::nifm::SubmitNetworkRequest();

    /* wait for network interface availability, while providing status */
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    /* if the network is not available after waiting then clean up and exit */
    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\nError: network is not available.\n");
        goto Exit;
    }
    shouldFinalizeNifm = true;

    /* Initialize socket library, while supplying configuration and memory pool */
    ret = nn::socket::Initialize(g_SocketConfigWithMemory);

    if( ret.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed. Err Desc: %d\n", ret.GetDescription());
        goto Exit;
    }
    shouldFinalizeSocket = true;

    /* demonstrate forward hostname-to-ip address lookups with nn::socket::GetHostByName */
    DemonstrateGetHostByName(pHostname);

    /* demonstrate reverse ip-to-host name lookups with nn::socket::GetHostByAddr */
    DemonstrateGetHostByAddr(pIpAddress);

    /* demonstrate forward hostname-to-ip address lookups with nn::socket::GetAddrInfo */
    DemonstrateGetAddrInfo(pHostname);

    /* demonstrate reverse ip-to-host name lookups with nn::socket::GetAddrInfo */
    DemonstrateGetAddrInfo(pIpAddress);

    /* demonstrate reverse ip-to-host name lookups with nn::socket::GetNameInfo */
    DemonstrateGetNameInfo(pIpAddress);

Exit:
    /* cleanup socket library */
    if ( true == shouldFinalizeSocket )
    {
        nn::socket::Finalize();
    }

    /* this application no longer requires use of network interface */
    if ( true == shouldFinalizeNifm )
    {
        nn::nifm::CancelNetworkRequest();
    }

    return;
}

