﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PlIlluminance.cpp,PageSamplePlIlluminance}
 *
 * @brief
 *  照度の取得のサンプルプログラム
 */

/**
 * @page PageSamplePlIlluminance 照度の取得
 * @tableofcontents
 *
 * @brief
 *  現在の照度を取得する方法の解説です。
 *
 * @section PageSamplePlIlluminance_SectionBrief 概要
 *  現在の照度を取得する方法を解説します。
 *
 * @section PageSamplePlIlluminance_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/PlIlluminance
 *  Samples/Sources/Applications/PlIlluminance @endlink 以下にあります。
 *
 * @section PageSamplePlIlluminance_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSamplePlIlluminance_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSamplePlIlluminance_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSamplePlIlluminance_SectionDetail 解説
 *
 * @subsection PageSamplePlIlluminance_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  PlIlluminance.cpp
 *  @includelineno PlIlluminance.cpp
 *
 * @subsection PageSamplePlIlluminance_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは 50 ミリ秒に一度、現在の照度を取得して表示します。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 以下のループを 10 秒程度繰り返します。
 *      - @ref nn::pl::GetCurrentIlluminance() を呼び出し、現在の照度を取得して表示します。
 *          - 関数から overflow フラグが返された場合は、その旨も表示します。
 *      - 50 ミリ秒間スリープします。
 *  - 照度の取得に失敗した場合は、その場でデモを終了します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  PlIlluminance_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nns/gfxLog.h>
#include <nns/nns_Log.h>

#include <nn/pl/pl_IlluminanceApi.h>

//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // ログ描画用スレッドのコアを指定します。
    nns::gfxLog::SetThreadCoreNumber(0);

    // 20 fps 程度の頻度で現在の照度値を表示します。
    for ( int i = 0; i < 200; i++ )
    {
        auto isAvailable = nn::pl::IsIlluminanceAvailable();
        if ( isAvailable )
        {
            bool isOverflown;
            auto illuminance = nn::pl::GetCurrentIlluminance(&isOverflown);
            NNS_LOG("\r[%d] Current illuminance is %4.2f lux. %s", i + 1, illuminance,
                isOverflown ? "[OVERFLOW ALERT]" : "                   ");
        }
        else
        {
            NNS_LOG("\rIlluminance is not available. ");
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50)); // 画面表示されるのを待ちます
    }

    // プログラムの終了
    NNS_LOG("\nEnd of Demo.\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
}
