﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

namespace {

void ProcessReceivedMessage(const nn::ovln::Message& message, nn::os::Tick tick)
{
    switch (message.tag)
    {
        // サンプル用
        case 100001:
        case 100002:
        case 100003:
        case 100004:
        case 100005:
        {
            // TODO: サンプルの簡略化のためここでは行っていないが、
            //       message.data が正しい NULL 終端文字列であることなどをチェックする必要がある
            auto s = reinterpret_cast<const char*>(&message.data);
            NN_LOG("received[%lldms]: tag = %u message: '%s'\n", tick.ToTimeSpan().GetMilliSeconds(), message.tag, s);
            return;
        }
        default:
        {
            NN_LOG("received[%lldms]: Unknown Message tag = %u\n", tick.ToTimeSpan().GetMilliSeconds(), message.tag);
            return;
        }
    }
}

}

void DisplaySubMain(nn::os::Event* pTerminateRequestEvent)
{
    // 受信ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    NN_UTIL_SCOPE_EXIT
    {
        nn::ovln::FinalizeReceicerLibraryForOverlay();
    };

    // Receiver の初期化
    nn::ovln::ReceiverForOverlayType receiver;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
    NN_UTIL_SCOPE_EXIT
    {
        nn::ovln::FinalizeReceiver(&receiver);
    };

    // メッセージ受信-処理ループ
#if 1
    // 終了する必要があるのであれば、Receiver と終了リクエストイベントを同時待ちする必要がある

    // MultiWait を準備
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::FinalizeMultiWait(&multiWait);
    };

    // 同時待ちホルダーを receiver を使用して初期化し MultiWait にリンク
    nn::os::MultiWaitHolderType receiverHolder;
    nn::ovln::InitializeMultiWaitHolder(&receiverHolder, &receiver);
    nn::os::LinkMultiWaitHolder(&multiWait, &receiverHolder);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlinkMultiWaitHolder(&receiverHolder);
        nn::os::FinalizeMultiWaitHolder(&receiverHolder);
    };

    // 同時待ちホルダーを終了イベントを使用して初期化し MultiWait にリンク
    nn::os::MultiWaitHolderType eventHolder;
    nn::os::InitializeMultiWaitHolder(&eventHolder, pTerminateRequestEvent->GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &eventHolder);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlinkMultiWaitHolder(&eventHolder);
        nn::os::FinalizeMultiWaitHolder(&eventHolder);
    };

    // ループ本体
    for (;;)
    {
        // Receiver と終了リクエストイベントを同時待ちする
        nn::os::WaitAny(&multiWait);
        // いずれかがシグナルされたらメッセージの受信を試行する
        // ※ 終了リクエストが来ていてもメッセージが残っていたら処理する
        nn::ovln::Message message;
        nn::os::Tick tick;
        if (!nn::ovln::TryReceive(&message, &tick, &receiver))
        {
            // 受信できなかった場合には終了通知を確認し、ループを抜ける
            if (pTerminateRequestEvent->TryWait())
            {
                break;
            }
        }
        // 受信に成功した場合にはメッセージを処理する
        ProcessReceivedMessage(message, tick);
    }
#else
    // 終了する必要がないのであれば、単に、メッセージを受け取って処理すれば十分
    NN_UNUSED(pTerminateRequestEvent);
    for (;;)
    {
        nn::ovln::Message message;
        nn::ovln::Receive(&message, &receiver);
        ProcessReceivedMessage(message);
    }
#endif
}
