﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsTickAndTimeSpan.cpp,PageSampleOsTickAndTimeSpan}
 *
 * @brief
 *  時間管理機能のサンプルプログラム
 */

/**
 * @page PageSampleOsTickAndTimeSpan TickAndTimeSpan
 * @tableofcontents
 *
 * @brief
 *  時間管理機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsTickAndTimeSpan_SectionBrief 概要
 *  ここでは、時間管理機能を使ったサンプルプログラムの説明をします。
 *
 *  時間管理機能の使い方については、時間管理機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsTickAndTimeSpan_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsTickAndTimeSpan Samples/Sources/Applications/OsTickAndTimeSpan @endlink 以下にあります。
 *
 * @section PageSampleOsTickAndTimeSpan_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsTickAndTimeSpan_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsTickAndTimeSpan_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsTickAndTimeSpan_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsTickAndTimeSpan_SectionDetail 解説
 *
 * @subsection PageSampleOsTickAndTimeSpan_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsTickAndTimeSpan.cpp
 *  @includelineno OsTickAndTimeSpan.cpp
 *
 * @subsection PageSampleOsTickAndTimeSpan_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() から SampleCodeFunction() を発行
 *  - SampleCodeFunction() ではまず、チックの周期を取得して表示
 *  - 次に、現在のチック値を取得
 *  - 現在のチック値を nn::TimeSpan 型（ナノ秒単位）に変換
 *  - チック値と nn::TimeSpan 値（ミリ秒に変換したもの）を表示
 *  - １秒間待機
 *  - 上記の処理を 10 回繰り返す
 *  - nnMain() に戻って終了
 *
 *  SampleCodeFunction() では、約１秒おきにチック値を取得し、
 *  その値を nn::TimeSpan 型に変換し、さらにミリ秒単位に換算して表示します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsTickAndTimeSpan_OutputExample.txt
 *
 *  ログ出力結果を見て分かる通り、１秒間の間にチックが進むカウント量は、
 *  GetSystemTickFrequency() で取得した値であり、このカウント量は、
 *  実行環境ごとに異なりますので注意して下さい。
 *
 *  一方、 nn::TimeSpan 型は 1 カウント当たり 1 ナノ秒相当となっています。
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

//
//  サンプルプログラムの本体
//
void SampleCodeFunction()
{
    // チックの周波数を取得して表示
    int64_t freq = nn::os::GetSystemTickFrequency();
    NN_LOG("Tick Frequency = %lld\n\n", freq);

    // 1 秒おきにチック値を取得して表示@n
    // 同時に、TimeSpan 型に変換してミリ秒に換算した値も合わせて表示
    NN_LOG("         Tick Count         TimeSpan\n");
    NN_LOG("------------------------------------------\n");
    for (int i=0; i<10; ++i)
    {
        nn::os::Tick    tick     = nn::os::GetSystemTick();
        nn::TimeSpan    timeSpan = nn::os::ConvertToTimeSpan( tick );
        NN_LOG("%d: %16lld %16lld(msec)\n", i, tick.GetInt64Value(), timeSpan.GetMilliSeconds());

        // 1 秒待機
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    }
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    // サンプルコード関数を発行
    SampleCodeFunction();
}

