﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsMemoryHeap.cpp,PageSampleOsMemoryHeap}
 *
 * @brief
 *  メモリヒープ機能のサンプルプログラム
 */

/**
 * @page PageSampleOsMemoryHeap MemoryHeap
 * @tableofcontents
 *
 * @brief
 *  メモリヒープ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsMemoryHeap_SectionBrief 概要
 *  ここでは、メモリヒープ機能を使ったサンプルプログラムの説明をします。
 *
 *  メモリヒープ機能の使い方については、メモリヒープ機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsMemoryHeap_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsMemoryHeap Samples/Sources/Applications/OsMemoryHeap @endlink 以下にあります。
 *
 * @section PageSampleOsMemoryHeap_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsMemoryHeap_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsMemoryHeap_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsMemoryHeap_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsMemoryHeap_SectionDetail 解説
 *
 * @subsection PageSampleOsMemoryHeap_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsMemoryHeap.cpp
 *  @includelineno OsMemoryHeap.cpp
 *
 * @subsection PageSampleOsMemoryHeap_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nninitStartup() にて nn::os::SetMemoryHeapSize() でメモリヒープを確保
 *  - nnMain() にて確保できたメモリヒープのサイズ情報をログ出力
 *  - MemorySample() にてサンプルのメイン処理を実行
 *    - nn::os::AllocateMemoryBlock() でメモリブロックを獲得
 *    - メモリブロックのアドレスとサイズをログに出力
 *    - nn::os::FreeMemoryBlock() でメモリブロックを返却
 *    - 獲得するメモリブロックのサイズを変更してループ
 *  - nnMain() に戻って終了
 *
 *  まず、nninitStartup() 関数にてメモリヒープのサイズを設定し、
 *  メモリヒープを利用できる環境を整えます。
 *  通常、この処理はアプリケーション起動時に一度だけ実行することが望ましいため、
 *  nninitStartup() 関数の中で行います。
 *
 *  MemorySample() では、メモリが獲得できなくなるまでサイズを大きくしながら
 *  nn::os::AllocateMemoryBlock() と nn::os::FreeMemoryBlock() を繰り返します。
 *
 *  なお、nn::os::AllocateMemoryBlock() では指定サイズを超えてメモリヒープを
 *  消費することはありません。そのため、フラグメントが起きていないならば、
 *  メモリヒープを 1byte も無駄にすることなく AllocateMemoryBlock() で
 *  使い切ることも出来ます。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  ただし、実行結果のメモリブロックアドレスについては実行環境によって異なります。
 *
 *  @verbinclude  OsMemoryHeap_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

const size_t HeapSize = 32u * 1024 * 1024;

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    // メモリヒープを確保
    auto result = nn::os::SetMemoryHeapSize( HeapSize );
    NN_ASSERT( result.IsSuccess(), "Failed in SetMemoryHeapSize()." );
}


//-----------------------------------------------------------------------------
//  サンプルプログラムの本体
//
void MemorySample()
{
    // サイズを大きくしながらメモリ獲得を繰り返す
    size_t size = nn::os::MemoryBlockUnitSize;
    while (size <= HeapSize)
    {
        uintptr_t  address;
        auto result = nn::os::AllocateMemoryBlock( &address, size );
        if (!result.IsSuccess())
        {
            // メモリ獲得に失敗したらループを抜ける
            break;
        }

        NN_LOG("Allocated memory block address : 0x%p\n", address);
        NN_LOG("Allocated memory block size    : 0x%zx\n", size);
        NN_LOG("\n");

        nn::os::FreeMemoryBlock( address, size );
        size = size * 2;
    }
}


//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    // メモリヒープ情報の出力
    NN_LOG("Size of memory heap : 0x%zx\n", HeapSize);
    NN_LOG("\n");

    // サンプルプログラム関数を発行
    MemorySample();

    NN_LOG("End\n");
}

