﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_InterruptEvent.h>

#include "OsInterruptEvent-hw.smma53.h"

namespace nns { namespace os { namespace interruptEvent {

//-----------------------------------------------------------------------------
//  100msec 毎にタイマー割込みを発生させる
//-----------------------------------------------------------------------------

uintptr_t   TimerRegisterVirtualAddress;

#define NNS_REG_TIMER1LOAD      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNS_REG_TIMER1VALUE     (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNS_REG_TIMER1CONTROL   (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x08))
#define NNS_REG_TIMER1INTCLR    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x0c))
#define NNS_REG_TIMER1RIS       (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x10))
#define NNS_REG_TIMER1MIS       (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#define NNS_REG_TIMER1BGLOAD    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x18))

//-----------------------------------------------------------------------------

void ClearTimerInterruptStatus() NN_NOEXCEPT
{
    NNS_REG_TIMER1INTCLR = 0x1;

    nn::dd::EnsureMemoryAccess();
}

//-----------------------------------------------------------------------------

void StartTimerInterrupt() NN_NOEXCEPT
{
    NN_LOG("Timer: padrs=0x%016llx\n", TimerRegisterPhysicalAddress);
    NN_LOG("Timer: size =0x%08zx\n",    TimerRegisterRegionSize);

    // タイマー I/O がマッピングされている仮想アドレスを取得
    uintptr_t vadrs = nn::dd::QueryIoMappingAddress(TimerRegisterPhysicalAddress, TimerRegisterRegionSize);
    if (!vadrs)
    {
        NN_ABORT("*** Timer Register is not mapped.");
    }
    TimerRegisterVirtualAddress = vadrs;

    NN_LOG("Timer: vadrs=0x%p\n",    TimerRegisterVirtualAddress);

    // タイマー動作を停止させた状態で、他の設定を行なう
    NNS_REG_TIMER1CONTROL   = (0u << 7) |   // 0: Disable timer module
                              (1u << 6) |   // 1: periodic mode
                              (0u << 5) |   // 0: Disable timer interrupt
                              (1u << 1);    // 1: 32bit counter

    // タイマー周期（100msec 周期に設定）
    const uint32_t baseCount = Sp804Timer1ClockFrequency / (10 /* Hz */);
    NNS_REG_TIMER1LOAD       = baseCount;
    NNS_REG_TIMER1BGLOAD     = baseCount;

    ClearTimerInterruptStatus();

    // 割込みを許可
    NNS_REG_TIMER1CONTROL |= 0x1 << 5;

    // タイマーの動作を開始
    NNS_REG_TIMER1CONTROL |= 0x1 << 7;
}

//-----------------------------------------------------------------------------

void StopTimerInterrupt() NN_NOEXCEPT
{
    // Timer1 の割込み禁止と動作の停止
    NNS_REG_TIMER1CONTROL &= ~((0x1 << 7) | (0x1 << 5));

    ClearTimerInterruptStatus();
}

//-----------------------------------------------------------------------------

}}} // namespace nns::os::interruptEvent

