﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_InterruptEvent.h>

#include "OsInterruptEvent-hw.imx6.h"

namespace nns { namespace os { namespace interruptEvent {

//-----------------------------------------------------------------------------
//  100msec 毎にタイマー割込みを発生させる
//-----------------------------------------------------------------------------

uintptr_t   TimerRegisterVirtualAddress;

#define NNS_REG_GPT_CR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNS_REG_GPT_PR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNS_REG_GPT_SR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x08))
#define NNS_REG_GPT_IR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x0c))
#define NNS_REG_GPT_OCR1    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x10))
#define NNS_REG_GPT_OCR2    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#define NNS_REG_GPT_OCR3    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x18))
#define NNS_REG_GPT_ICR1    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x1c))
#define NNS_REG_GPT_ICR2    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x20))
#define NNS_REG_GPT_CNT     (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x24))

//-----------------------------------------------------------------------------

void ClearTimerInterruptStatus() NN_NOEXCEPT
{
    NNS_REG_GPT_SR = 0x1;

    nn::dd::EnsureMemoryAccess();
}

//-----------------------------------------------------------------------------

void StartTimerInterrupt() NN_NOEXCEPT
{
    NN_LOG("Timer: padrs=0x%016llx\n", TimerRegisterPhysicalAddress);
    NN_LOG("Timer: size =0x%08zx\n",    TimerRegisterRegionSize);

    // タイマー I/O がマッピングされている仮想アドレスを取得
    uintptr_t vadrs = nn::dd::QueryIoMappingAddress(TimerRegisterPhysicalAddress, TimerRegisterRegionSize);
    if (!vadrs)
    {
        NN_ABORT("*** Timer Register is not mapped.");
    }
    TimerRegisterVirtualAddress = vadrs;

    NN_LOG("Timer: vadrs=0x%p\n",    TimerRegisterVirtualAddress);

    // GPT の無効化
    NNS_REG_GPT_CR = 0;

    // GPT の割込みを全て禁止
    NNS_REG_GPT_IR = 0x00;

    // GPT の SW Reset をアサート
    NNS_REG_GPT_CR |= 0x1 << 15;

    // GPT の SW Reset が自動的にデアサートするのを待つ
    while ( NNS_REG_GPT_CR & (0x1 << 15) )
    {
        ;
    }

    ClearTimerInterruptStatus();

    // GPT_SR のクリア
    NNS_REG_GPT_SR = 0x3f;

    // GPT のクロックソースの設定（Peripheral Clock）
    NNS_REG_GPT_CR |= 0x1 << 6;

    // ENMOD=1 によりカウンタをクリア
    NNS_REG_GPT_CR |= 0x1 << 1;

    // GPT のタイマー周期を設定
    NNS_REG_GPT_OCR1 = GptPeripheralClock / 10;

    // Enable GPT
    NNS_REG_GPT_CR |= 0x1 << 0;

    // GPT の割込み許可の設定（Output Compare1 のみ許可）
    NNS_REG_GPT_IR = 0x01;
}

//-----------------------------------------------------------------------------

void StopTimerInterrupt() NN_NOEXCEPT
{
    // GPT の無効化
    NNS_REG_GPT_CR = 0;

    // GPT の割込みを全て禁止
    NNS_REG_GPT_IR = 0x00;

    ClearTimerInterruptStatus();
}

//-----------------------------------------------------------------------------

}}} // namespace nns::os::interruptEvent

