﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsFiber.cpp,PageSampleOsFiber}
 *
 * @brief
 *  ファイバ機能のサンプルプログラム
 */

/**
 * @page PageSampleOsFiber Fiber
 * @tableofcontents
 *
 * @brief
 *  ファイバ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsFiber_SectionBrief 概要
 *  ここでは、ファイバ機能を使ったサンプルプログラムの説明をします。
 *
 *  ファイバ機能の使い方については、ファイバ機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsFiber_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsFiber Samples/Sources/Applications/OsFiber @endlink 以下にあります。
 *
 * @section PageSampleOsFiber_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsFiber_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsFiber_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsFiber_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsFiber_SectionDetail 解説
 *
 * @subsection PageSampleOsFiber_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsFiber.cpp
 *  @includelineno OsFiber.cpp
 *
 * @subsection PageSampleOsFiber_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのファイバを初期化
 *  - 2 つのファイバは FiberFunction1() と FiberFunction2()
 *  - nnMain()         から FiberFunction1() へ SwitchToFiber() で遷移
 *  - FiberFunction1() から FiberFunction2() へ SwitchToFiber() で遷移
 *  - FiberFunction2() から FiberFunction1() へ SwitchToFiber() で遷移
 *  - FiberFunction1() から FiberFunction2() へ SwitchToFiber() で遷移
 *  - FiberFunction2() から nnMain()         へ SwitchToFiber() で遷移
 *  - nnMain() にて 2 つのファイバを破棄
 *
 *  nnMain() や各ファイバ関数にて SwitchToFiber() を呼び出して他のファイバへ
 *  遷移しています。その前後では実行順序に関するログ出力と、カレントファイバが
 *  自分自身であることをチェックするようにしています。
 *
 *  本サンプルプログラムで示すように、ファイバ機能を使うと SwitchToFiber() で
 *  一時的に処理を中断して他のファイバへ実行を移し、また自分自身のファイバへ
 *  戻ってきた場合には、SwitchToFiber() を呼び出したコードの続きから実行を
 *  継続することができます。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsFiber_OutputExample.txt
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t FiberStackSize = 16 * 1024;

// スタックガードなしの場合は NN_OS_ALIGNAS_FIBER_STACK でアライメントし、
// サイズは nn::os::FiberStackAlignment でアライメントしておきます。
NN_OS_ALIGNAS_FIBER_STACK   uint8_t  g_FiberStack1[ FiberStackSize ];
static_assert( FiberStackSize % nn::os::FiberStackAlignment == 0, "" );

// スタックガードありの場合は NN_OS_ALIGNAS_GUARDED_STACK でアライメントし、
// サイズは nn::os::GuardedStackAlignment でアライメントしておきます。
NN_OS_ALIGNAS_GUARDED_STACK uint8_t  g_FiberStack2[ FiberStackSize ];
static_assert( FiberStackSize % nn::os::GuardedStackAlignment == 0, "" );

nn::os::FiberType  g_Fiber1;
nn::os::FiberType  g_Fiber2;

int g_Step;

}   // namespace

//-----------------------------------------------------------------------------
//  実行順序 と カレントファイバのチェック
//
void CheckCurrentFiber(int expectStep, nn::os::FiberType* myFiber)
{
    ++g_Step;
    NN_ASSERT( g_Step == expectStep );

    NN_LOG("Step%d: nn::os::GetCurrentFiber()= ", expectStep);
    if (myFiber == NULL)
    {
        NN_LOG("NULL\n");
    }
    else if (myFiber == &g_Fiber1)
    {
        NN_LOG("&g_Fiber1\n");
    }
    else if (myFiber == &g_Fiber2)
    {
        NN_LOG("&g_Fiber2\n");
    }
    else
    {
        NN_LOG("UNKNOWN FIBER\n");
    }

    auto* currentFiber = nn::os::GetCurrentFiber();
    NN_ASSERT( currentFiber == myFiber );
}


//-----------------------------------------------------------------------------
//  ファイバ 1
//
nn::os::FiberType* FiberFunction1(void *arg)
{
    auto argNum = reinterpret_cast<uintptr_t>(arg);
    NN_LOG("FiberFunction1: arg=%zu\n", argNum);

    // カレントファイバのチェック
    CheckCurrentFiber( 2, &g_Fiber1 );

    // SwtichToFiber() を使ってファイバ 2 に遷移する
    nn::os::SwitchToFiber( &g_Fiber2 );

    // カレントファイバのチェック
    CheckCurrentFiber( 4, &g_Fiber1 );

    // return でファイバ 2 の続きから再実行
    return &g_Fiber2;
}


//-----------------------------------------------------------------------------
//  ファイバ 2
//
nn::os::FiberType* FiberFunction2(void *arg)
{
    auto argNum = reinterpret_cast<uintptr_t>(arg);
    NN_LOG("FiberFunction2: arg=%zu\n", argNum);

    // カレントファイバのチェック
    CheckCurrentFiber( 3, &g_Fiber2 );

    // SwtichToFiber() を使ってファイバ 1 の続きから再実行
    nn::os::SwitchToFiber( &g_Fiber1 );

    // カレントファイバのチェック
    CheckCurrentFiber( 5, &g_Fiber2 );

    // return で元々のスレッドの続きから再実行
    return NULL;
}


//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    g_Step = 0;

    // ファイバ 1 を初期化（スタックガードなし）
    nn::os::InitializeFiber( &g_Fiber1, FiberFunction1, reinterpret_cast<void*>(1), g_FiberStack1, FiberStackSize, nn::os::FiberFlag_NoStackGuard );

    // ファイバ 2 を初期化（スタックガードあり）
    nn::os::InitializeFiber( &g_Fiber2, FiberFunction2, reinterpret_cast<void*>(2), g_FiberStack2, FiberStackSize, 0 );

    // カレントファイバのチェック
    CheckCurrentFiber( 1, NULL );

    // ファイバ 1 に遷移する
    nn::os::SwitchToFiber( &g_Fiber1 );

    // カレントファイバのチェック
    CheckCurrentFiber( 6, NULL );

    // ファイバを破棄する
    nn::os::FinalizeFiber( &g_Fiber1 );
    nn::os::FinalizeFiber( &g_Fiber2 );
}

