﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeIdlePolicy.cpp,PageSampleOeIdlePolicy}
 *
 * @brief
 *  無操作状態の制御のサンプルプログラム
 */

/**
 * @page PageSampleOeIdlePolicy 無操作状態の制御
 * @tableofcontents
 *
 * @brief
 *  無操作状態での画面焼け軽減等の機能の開始時間を制御する方法の解説です。
 *
 * @section PageSampleOeIdlePolicy_SectionBrief 概要
 *  無操作状態での画面焼け軽減等の機能の開始時間を制御する方法を解説します。
 *
 * @section PageSampleOeIdlePolicy_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeIdlePolicy
 *  Samples/Sources/Applications/OeIdlePolicy @endlink 以下にあります。
 *
 * @section PageSampleOeIdlePolicy_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeIdlePolicy_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleOeIdlePolicy_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *  サンプル内で呼び出している機能の効果を確認するには、あらかじめ本体設定で自動スリープを有効にしておく必要があります。
 *
 * @section PageSampleOeIdlePolicy_SectionDetail 解説
 *
 * @subsection PageSampleOeIdlePolicy_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeIdlePolicy.cpp
 *  @includelineno OeIdlePolicy.cpp
 *
 * @subsection PageSampleOeIdlePolicy_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは、以下の 3 つの無操作状態検出の開始時間の延長モードを切り替えます。
 *
 *  - nn::oe::UserInactivityDetectionTimeExtensionMode_Disabled
 *  - nn::oe::UserInactivityDetectionTimeExtensionMode_Enabled
 *  - nn::oe::UserInactivityDetectionTimeExtensionMode_EnabledUnsafe
 *
 *  モードは以下の操作のどれかによって順番に切り替えることができます。
 *
 *  - DebugPad の A ボタンを押す
 *  - GamePad の A ボタンを押す
 *  - タッチパネルにタッチする
 *
 *  画面上には、最後にモードを切り替えてから 1 秒ごとにログを出しています。
 *  これにより、画面焼け軽減や自動スリープの機能が発動するタイミングがわかります。
 *
 *  このデモは操作によってモードを切り替えているため、操作するたびに無操作時間のカウントはリセットされています。
 *  API の呼び出しそのものによって無操作時間のカウントがリセットされているわけではないことにご注意ください。
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nns/gfxLog.h>
#include <nns/nns_Log.h>
#include <nns/hid.h>

#include <nn/oe/oe_IdlePolicyApi.h>

//-----------------------------------------------------------------------------

namespace {
    nn::os::TimerEvent g_OneSecondTimer(nn::os::EventClearMode_AutoClear);
    int g_ElapsedSeconds = 0;
}

void UpdateExtensionMode(nn::oe::UserInactivityDetectionTimeExtensionMode mode) NN_NOEXCEPT
{
    switch ( mode )
    {
        case nn::oe::UserInactivityDetectionTimeExtensionMode_Disabled:
        {
            nn::oe::SetUserInactivityDetectionTimeExtended(false);
            NNS_LOG("Mode: Disabled.\n");
            break;
        }
        case nn::oe::UserInactivityDetectionTimeExtensionMode_Enabled:
        {
            nn::oe::SetUserInactivityDetectionTimeExtended(true);
            NNS_LOG("Mode: Enabled.\n");
            break;
        }
        case nn::oe::UserInactivityDetectionTimeExtensionMode_EnabledUnsafe:
        {
            nn::oe::SetUserInactivityDetectionTimeExtendedUnsafe(true);
            NNS_LOG("Mode: EnabledUnsafe.\n");
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    g_ElapsedSeconds = 0;
    g_OneSecondTimer.StartPeriodic(nn::TimeSpan::FromSeconds(1), nn::TimeSpan::FromSeconds(1));
}

nn::oe::UserInactivityDetectionTimeExtensionMode NextExtensionMode(nn::oe::UserInactivityDetectionTimeExtensionMode mode) NN_NOEXCEPT
{
    switch ( mode )
    {
        case nn::oe::UserInactivityDetectionTimeExtensionMode_Disabled:
        {
            return nn::oe::UserInactivityDetectionTimeExtensionMode_Enabled;
        }
        case nn::oe::UserInactivityDetectionTimeExtensionMode_Enabled:
        {
            return nn::oe::UserInactivityDetectionTimeExtensionMode_EnabledUnsafe;
        }
        case nn::oe::UserInactivityDetectionTimeExtensionMode_EnabledUnsafe:
        {
            return nn::oe::UserInactivityDetectionTimeExtensionMode_Disabled;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void RunTimeCountingThread() NN_NOEXCEPT
{
    static nn::os::ThreadType s_Thread;
    static NN_ALIGNAS(4096) char s_ThreadStack[4 * nn::os::MemoryPageSize];
    nn::os::CreateThread(
        &s_Thread,
        [](void* arg) {
            NN_UNUSED(arg);
            while ( NN_STATIC_CONDITION(true) )
            {
                g_OneSecondTimer.Wait();
                ++g_ElapsedSeconds;
                NNS_LOG("%d sec\n", g_ElapsedSeconds);
            }
        },
        nullptr,
        s_ThreadStack,
        sizeof(s_ThreadStack),
        nn::os::DefaultThreadPriority
    );
    nn::os::SetThreadNamePointer(&s_Thread, "1sTimerThread");
    nn::os::StartThread(&s_Thread);
}


//
//  メイン関数
//
extern "C" void nnMain()
{
    // ログ描画用スレッドのコアを指定します。
    nns::gfxLog::SetThreadCoreNumber(0);

    NNS_LOG("Press A key on Debug Pad or Game Pad, or touch the screen to change extension mode.\n");

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);

    RunTimeCountingThread();

    nn::oe::UserInactivityDetectionTimeExtensionMode currentMode = nn::oe::UserInactivityDetectionTimeExtensionMode_Disabled;
    UpdateExtensionMode(currentMode);

    auto& gamePad = *reinterpret_cast<nns::hid::GamePad*>(controllerManager.GetController(nns::hid::ControllerId_GamePad, 0));
    auto& debugPad = *reinterpret_cast<nns::hid::DebugPad*>(controllerManager.GetController(nns::hid::ControllerId_DebugPad, 0));
    auto& touchScreen = *reinterpret_cast<nns::hid::TouchScreen*>(controllerManager.GetController(nns::hid::ControllerId_TouchScreen, 0));

    bool wasTouched = false;
    while ( NN_STATIC_CONDITION(true) )
    {
        controllerManager.Update();

        bool changeMode = false;
        bool isTouched = touchScreen.IsConnected() && touchScreen.GetTouchStates().size() > 0;
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(nns::hid::Button::A::Mask)) ||
             (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(nns::hid::Button::A::Mask)) ||
             (isTouched && !wasTouched) )
        {
            changeMode = true;
        }
        wasTouched = isTouched;

        if ( changeMode )
        {
            currentMode = NextExtensionMode(currentMode);
            UpdateExtensionMode(currentMode);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // プログラムの終了
    NNS_LOG("\nEnd of Demo.\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
}
