<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Import network setting

    .DESCRIPTION
        Import network setting
#>

function Resolve-PathSafe
{
    param
    (
        [string] $Path
    )

    $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($Path)
}

$SettingsManagerExe = (Get-ChildItem -path ..\..\..\..\Tools\CommandLineTools\Settings\ -r -include SettingsManager.exe | Sort-Object LastWriteTime)
$SettingsManagerExe = If(Test-Path $SettingsManagerExe) { Resolve-Path $SettingsManagerExe } Else { $null }

$ControlTarget = "..\..\..\..\Tools\CommandLineTools\ControlTarget.exe"
$ControlTarget = If(Test-Path $ControlTarget) { Resolve-Path $ControlTarget } Else { $null }

$TempExportFile = Resolve-PathSafe tmp.txt

$SettingsList = Get-ChildItem -r -path .\ -include networkSettings*.txt

###############################

If($null -eq $SettingsManagerExe)
{
    Write-Host "[ERROR] SettingsManager.exe not found." -ForegroundColor Red
    return
}

$Options = @()
$Options += New-Object System.Management.Automation.Host.ChoiceDescription "&NX", "NX"
$Options += New-Object System.Management.Automation.Host.ChoiceDescription "&Win", "Windows"

$Target = $host.ui.PromptForChoice("<Target>", "Select target", [System.Management.Automation.Host.ChoiceDescription[]]($Options), 0)

Switch($Options[$Target].label)
{
    "&NX"  { $RemoteMode = $True }
    "&Win" { $RemoteMode = $False }
}

$Options = @()
$Initials = @()
Foreach($i in $SettingsList)
{
    $Index = 0
    $OptionName = $i.Name.Replace("networkSettings_For","").Replace(".txt","")
    For($j = 0; $j -lt $OptionName.length; $j++)
    {
        If(-Not($Initials -contains $OptionName[$j]))
        {
            $Initials += $OptionName[$j]
            $Index = $j
            break
        }
    }
    $OptionName = $OptionName.Substring(0,$Index) + "&" + $OptionName.Substring($Index,$OptionName.length-$Index)
    $Options += new-object System.Management.Automation.Host.ChoiceDescription $OptionName, $i
}

$Setting = $host.ui.PromptForChoice("<Setting>", "Select setting", [System.Management.Automation.Host.ChoiceDescription[]]($Options), 0)

$PInfo = New-Object System.DIagnostics.ProcessStartInfo
$PInfo.RedirectStandardError = $true
$PInfo.RedirectStandardOutput = $true
$PInfo.UseShellExecute = $false

$P = New-Object System.Diagnostics.Process

# improt
Write-Host "Importing " $SettingsList[$Setting].Name
If($RemoteMode)
{
    $PInfo.FileName = $SettingsManagerExe
    $PInfo.Arguments = "Import --remote " + $SettingsList[$Setting]
}
Else
{
    $PInfo.FileName = $SettingsManagerExe
    $PInfo.Arguments = "Import " + $SettingsList[$Setting]
}
$P.StartInfo = $PInfo
$P.Start() | Out-Null
$P.WaitForExit()

$Stdout = $P.StandardOutput.ReadToEnd()
$Stderr = $P.StandardError.ReadToEnd()

If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
{
    Write-Host $Stdout $Stderr -ForegroundColor Red
    return
}
Else
{
    Write-Host "[SUCCESS] Import network setting" -ForegroundColor Green
}

# reset
If($RemoteMode)
{
    Write-Host "Resetting"
    $PInfo.FileName = $ControlTarget
    $PInfo.Arguments = "reset"
    $P.StartInfo = $PInfo
    $P.Start() | Out-Null
    $P.WaitForExit()

    $Stdout = $P.StandardOutput.ReadToEnd()
    $Stderr = $P.StandardError.ReadToEnd()

    If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
    {
        Write-Host $Stdout $Stderr -ForegroundColor Red
        return
    }
    Else
    {
        Write-Host "[SUCCESS] Reset target" -ForegroundColor Green
    }
}

Start-Sleep -s 10

# export
Write-Host "Exporting to " $TempExportFile
If($RemoteMode)
{
    $PInfo.FileName = $SettingsManagerExe
    $PInfo.Arguments = "Export --remote " +  $TempExportFile
}
Else
{
    $PInfo.FileName = $SettingsManagerExe
    $PInfo.Arguments = "Export " +  $TempExportFile
}

$P.StartInfo = $PInfo
$P.Start() | Out-Null
$P.WaitForExit()

$Stdout = $P.StandardOutput.ReadToEnd()
$Stderr = $P.StandardError.ReadToEnd()

If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
{
    Write-Host $Stdout $Stderr -ForegroundColor Red
    return
}
Else
{
    Write-Host "[SUCCESS] Export network setting" -ForegroundColor Green
}
