﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include "HidController.h"
#include "TagUtility.h"
#include "TagAccessor.h"
#include "UpdateState.h"

//------------------------------------------------------------------------------
//  NFP 処理フロー
//------------------------------------------------------------------------------
namespace nns { namespace nfp { namespace {

    //--------------------------------------------------------------------------
    //  定数、型、変数定義
    //--------------------------------------------------------------------------

    // アプリケーション専用領域に書き込まれるカウンタの初期値です。
    const uint32_t InitialCounter = 0;

    // 状態処理関数の型です。
    typedef void (*UpdateStateFunc)( ApplicationData& data );

    // タグ発見、タグ喪失時の状態遷移情報です。
    struct StateChain
    {
        State                               prev;
        State                               next;
        UpdateStateFunc                     writer;
        nn::Result                          result;
    };

    // アプリケーション専用領域に書き込むデータ型です。
    struct ApplicationAreaData
    {
        uint32_t                            counter;
    };

    // amiibo 設定用の作業データ型です。
    struct AmiiboSettingsData
    {
        bool                                isRegistered;
        nn::nfp::AmiiboSettingsStartParam   startParam;
        nn::nfp::DeviceHandle               deviceHandle;
    };

    // NFP 処理用の作業データ型です。
    struct NfpWorkData
    {
        StateChain                          chain;
        nn::os::EventType                   processFinishEvent;

        bool                                isListDevicesCanceled;
        nn::nfp::DeviceHandle               deviceHandles[ DeviceCountMax ];
        nn::hid::NpadIdType                 deviceNpadIds[ DeviceCountMax ];
        int                                 deviceCount;
        int                                 deviceIndex;

        nn::os::SystemEvent*                activateEventArray;
        nn::os::SystemEvent*                deactivateEventArray;
        nn::os::SystemEvent*                availabilityChangeEvent;

        nn::nfp::TagInfo                    tagInfo;
        nn::nfp::ModelInfo                  modelInfo;
        nn::nfp::CommonInfo                 commonInfo;
        nn::nfp::RegisterInfo               registerInfo;
        ApplicationAreaData                 applicationAreaData;
        size_t                              applicationAreaSize;
        AmiiboSettingsData                  amiiboSettingsData;
        nn::nfp::DeviceState                deviceState;
    };

    // NFP 処理用の作業変数です。
    NfpWorkData         g_Work;
    NN_ALIGNAS(16) char g_ActivateEventBuffer[ sizeof(nn::os::SystemEvent) * DeviceCountMax ];
    NN_ALIGNAS(16) char g_DeactivateEventBuffer[ sizeof(nn::os::SystemEvent) * DeviceCountMax ];

    //--------------------------------------------------------------------------
    //  通知イベント、デバイスハンドルの生成・破棄
    //--------------------------------------------------------------------------

    // 通知イベント、デバイスハンドルを生成します。
    void CreateSystemEvent() NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceCount > 0 );
        NN_ASSERT( g_Work.activateEventArray == nullptr );
        NN_ASSERT( g_Work.deactivateEventArray == nullptr );
        g_Work.activateEventArray = new(g_ActivateEventBuffer) nn::os::SystemEvent[ g_Work.deviceCount ];
        g_Work.deactivateEventArray = new(g_DeactivateEventBuffer) nn::os::SystemEvent[ g_Work.deviceCount ];
        std::memset(g_Work.deviceNpadIds, 0, sizeof(g_Work.deviceNpadIds));
    }

    // 通知イベント、デバイスハンドルを破棄します。
    void DestroySystemEvents() NN_NOEXCEPT
    {
        if( g_Work.activateEventArray != nullptr )
        {
            for( int i = 0; i < g_Work.deviceCount; i++ )
            {
                g_Work.activateEventArray[i].~SystemEvent();
            }
            g_Work.activateEventArray = nullptr;
        }
        if( g_Work.deactivateEventArray != nullptr )
        {
            for( int i = 0; i < g_Work.deviceCount; i++ )
            {
                g_Work.deactivateEventArray[i].~SystemEvent();
            }
            g_Work.deactivateEventArray = nullptr;
        }
        g_Work.deviceCount = 0;
        g_Work.deviceIndex = 0;
        std::memset(g_Work.deviceHandles, 0, sizeof(g_Work.deviceHandles));
        std::memset(g_Work.deviceNpadIds, 0, sizeof(g_Work.deviceNpadIds));
    }

    // 選択中のデバイスハンドルを取得します。
    nn::nfp::DeviceHandle* GetSelectedDeviceHandle() NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceIndex >= 0 && g_Work.deviceIndex < g_Work.deviceCount );
        return &g_Work.deviceHandles[g_Work.deviceIndex];
    }

    // 選択中のタグ発見通知イベントを取得します。
    nn::os::SystemEvent& GetSelectedActivateEvent() NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceIndex >= 0 && g_Work.deviceIndex < g_Work.deviceCount );
        return g_Work.activateEventArray[g_Work.deviceIndex];
    }

    // 選択中のタグ喪失通知イベントを取得します。
    nn::os::SystemEvent& GetSelectedDeactivateEvent() NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceIndex >= 0 && g_Work.deviceIndex < g_Work.deviceCount );
        return g_Work.deactivateEventArray[g_Work.deviceIndex];
    }

    //--------------------------------------------------------------------------
    //  タグ検知開始処理
    //--------------------------------------------------------------------------

    // タグの検出を開始します。
    void DoStartDetection(ApplicationData& data, State next, UpdateStateFunc writer) NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceCount > 0 );
        NN_ASSERT( (next == State_TagRead) || (next == State_TagWrite) );

        g_Work.chain.prev = data.state;
        g_Work.chain.next = next;
        g_Work.chain.writer = writer;
        g_Work.chain.result = nn::ResultSuccess();

        // 前回のタグ発見でイベントがシグナルされたままの状態になっている可能性があるため、
        // タグの発見を開始する前にイベントのシグナルをクリアしておくことを推奨します。
        GetSelectedActivateEvent().Clear();
        GetSelectedDeactivateEvent().Clear();

        // NFC デバイスの利用可/不可イベントをこのアプリではタグ検索中にしか必要としていませんので、このタイミングでクリアしておきます。
        g_Work.availabilityChangeEvent->Clear();

        TagAccessor& accessor = TagAccessor::GetInstance();
        accessor.StartDetection(GetSelectedDeviceHandle());
        accessor.Run(&g_Work.processFinishEvent);
        data.state = State_StartDetection;
    }

    // タグの検出を開始します。タグの読み込みのみを行う場合に呼び出します。
    void DoStartDetectionForTagRead(ApplicationData& data) NN_NOEXCEPT
    {
        DoStartDetection(data, State_TagRead, nullptr);
    }

    // タグの検出を開始します。タグの書き込みも行う場合に呼び出します。
    void DoStartDetectionForTagWrite(ApplicationData& data, UpdateStateFunc writer) NN_NOEXCEPT
    {
        NN_ASSERT( writer != nullptr );

        DoStartDetection(data, State_TagWrite, writer);
    }

    // タグ発見時に処理されます。
    void OnTagActivated(ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.state == State_TagSearch );

        // タグを発見したら必要な情報を取得します。
        TagAccessor& accessor = TagAccessor::GetInstance();
        switch( g_Work.chain.next )
        {
        case State_TagRead:
            {
                accessor.GetTagInfo(&g_Work.tagInfo, GetSelectedDeviceHandle());
                accessor.Mount(GetSelectedDeviceHandle());
                accessor.GetModelInfo(&g_Work.modelInfo, GetSelectedDeviceHandle());
                accessor.GetCommonInfo(&g_Work.commonInfo, GetSelectedDeviceHandle());
                accessor.GetRegisterInfo(&g_Work.registerInfo, GetSelectedDeviceHandle());
                accessor.OpenApplicationArea(GetSelectedDeviceHandle(), AccessId);
                accessor.GetApplicationArea(
                        &g_Work.applicationAreaData,
                        &g_Work.applicationAreaSize,
                        GetSelectedDeviceHandle(),
                        sizeof(g_Work.applicationAreaData)
                    );
                accessor.Unmount(GetSelectedDeviceHandle());
                accessor.Run(&g_Work.processFinishEvent);
                data.state = State_TagRead;
            }
            break;
        case State_TagWrite:
            {
                accessor.GetTagInfo(&g_Work.tagInfo, GetSelectedDeviceHandle());
                accessor.Run(&g_Work.processFinishEvent);
                data.state = State_TagCheck;
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // 書き込み対象のタグ発見時に処理されます。
    void OnTagChecked(ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.state == State_TagCheck );
        NN_ASSERT( g_Work.chain.next == State_TagWrite );
        NN_ASSERT( g_Work.chain.writer != nullptr );

        // 書き込み処理を行って次遷移先を設定します。
        (*g_Work.chain.writer)(data);
        data.state = State_TagWrite;
    }

    //--------------------------------------------------------------------------
    //  タグ検知停止処理
    //--------------------------------------------------------------------------

    // タグの検出を停止します。
    void DoStopDetection(ApplicationData& data, State next, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceCount > 0 );
        NN_ASSERT( next != State_None );

        g_Work.chain.prev = data.state;
        g_Work.chain.next = next;
        g_Work.chain.result = result;

        TagAccessor& accessor = TagAccessor::GetInstance();
        accessor.StopDetection(GetSelectedDeviceHandle());
        accessor.Run(&g_Work.processFinishEvent);
        data.state = State_StopDetection;
    }

    // タグの検出を停止します。停止後に次の状態に遷移する場合に呼び出します。
    void DoStopDetectionForNext(ApplicationData& data, State next) NN_NOEXCEPT
    {
        DoStopDetection(data, next, nn::ResultSuccess());
    }

    #if 0 // 未使用
    // タグの検出を停止します。停止後にエラー表示を行う場合に呼び出します。
    void DoStopDetectionForError(ApplicationData& data, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( result.IsFailure() );
        DoStopDetection(data, State_Error, result);
    }
    #endif

    // タグ検知停止時に処理されます。
    void OnTagStopped(ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.state == State_StopDetection );
        NN_ASSERT( g_Work.chain.prev != State_None );
        NN_ASSERT( g_Work.chain.next != State_None );

        // 次遷移先と実行結果を設定します。
        data.state = g_Work.chain.next;
        data.lastResult = g_Work.chain.result;
    }

    //--------------------------------------------------------------------------
    //  タグ喪失待機処理
    //--------------------------------------------------------------------------

    // タグが喪失するまで待機します。
    void WaitTagRelease(ApplicationData& data, State next, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( g_Work.deviceCount > 0 );
        NN_ASSERT( next != State_None );

        g_Work.chain.prev = data.state;
        g_Work.chain.next = next;
        g_Work.chain.result = result;

        data.state = State_TagRelease;
    }

    // タグが喪失するまで待機します。喪失後に次の状態に遷移する場合に呼び出します。
    void WaitTagReleaseForNext(ApplicationData& data, State next) NN_NOEXCEPT
    {
        WaitTagRelease(data, next, nn::ResultSuccess());
    }

    // タグが喪失するまで待機します。喪失後にエラー表示を行う場合に呼び出します。
    void WaitTagReleaseForError(ApplicationData& data, nn::Result result) NN_NOEXCEPT
    {
        NN_ASSERT( result.IsFailure() );
        WaitTagRelease(data, State_Error, result);
    }

    // タグ喪失時に処理されます。
    void OnTagDeactivated(ApplicationData& data) NN_NOEXCEPT
    {
        NN_ASSERT( data.state == State_TagRelease );
        NN_ASSERT( g_Work.chain.prev != State_None );
        NN_ASSERT( g_Work.chain.next != State_None );

        TagAccessor& accessor = TagAccessor::GetInstance();
        accessor.StopDetection(GetSelectedDeviceHandle());
        accessor.Run(&g_Work.processFinishEvent);
        data.state = State_StopDetection;
    }

    //--------------------------------------------------------------------------
    //  状態処理
    //--------------------------------------------------------------------------

    // NFP ライブラリを初期化していない状態です。
    void UpdateStateNone(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンで NFP ライブラリの初期化処理を開始します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            nn::os::InitializeEvent(&g_Work.processFinishEvent, false, nn::os::EventClearMode_AutoClear);
            TagAccessor& accessor = TagAccessor::CreateInstance();
            accessor.Initialize();
            g_Work.availabilityChangeEvent = new nn::os::SystemEvent();
            accessor.AttachAvailabilityChangeEvent(g_Work.availabilityChangeEvent->GetBase());
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_Initializing;
        }

        // Y ボタンで終了します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask) )
        {
            data.state = State_Exit;
        }
    }

    // NFP ライブラリを初期中です。
    void UpdateStateInitializing(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 初期化処理が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                data.state = State_Initialized;
            }
            else
            {
                // 初期化処理に失敗することはありません。
                data.state = State_None;
            }
        }
    }

    // NFP ライブラリが利用可能な状態です。
    void UpdateStateInitialized(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // デバイス情報の初期化
        data.deviceCount = 0;
        data.deviceIndex = 0;

        // B ボタンで NFP ライブラリを終了します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.Finalize();
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_Finalize;
        }

        // NFC デバイスの取得を開始します。
        else
        {
            g_Work.isListDevicesCanceled = false;

            NN_ASSERT( g_Work.deviceCount == 0 );
            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.ListDevices(g_Work.deviceHandles, &g_Work.deviceCount, DeviceCountMax);
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_DeviceListing;
        }
    }

    // NFC デバイスリストを取得中です。
    void UpdateStateDeviceListing(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // NFC デバイスの取得が完了したら状態遷移します。
        // NFC デバイスが見つからなかったら再取得を行います。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                if( g_Work.isListDevicesCanceled != false )
                {
                    accessor.Finalize();
                    accessor.Run(&g_Work.processFinishEvent);
                    data.state = State_Finalize;
                    DestroySystemEvents();
                }
                else if( g_Work.deviceCount > 0 )
                {
                    CreateSystemEvent();
                    for( int i = 0; i < g_Work.deviceCount; i++ )
                    {
                        accessor.AttachEvents(
                                &g_Work.deviceNpadIds[i],
                                g_Work.activateEventArray[i].GetBase(),
                                g_Work.deactivateEventArray[i].GetBase(),
                                &g_Work.deviceHandles[i]
                            );
                    }
                    accessor.Run(&g_Work.processFinishEvent);
                    data.state = State_DeviceInitialize;
                }
                else
                {
                    data.state = State_Initialized;
                }
            }
            else
            {
                data.state = State_Error;
                data.lastResult = result;
                DestroySystemEvents();
            }
        }

        // B ボタンで NFP ライブラリを終了します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            g_Work.isListDevicesCanceled = true;
        }
    }

    // NFC デバイスを初期化中です。
    void UpdateStateDeviceInitialize(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // NFC デバイスの初期化が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                NN_ASSERT( sizeof(data.npadIds) >= sizeof(g_Work.deviceNpadIds) );
                std::memcpy(data.npadIds, g_Work.deviceNpadIds, sizeof(g_Work.deviceNpadIds));
                data.deviceCount = g_Work.deviceCount;
                data.state = State_DeviceSelect;
            }
            else
            {
                data.state = State_Error;
                data.lastResult = result;
                DestroySystemEvents();
            }
        }
    }

    // NFC デバイスを選択中です。
    void UpdateStateDeviceSelect(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでタグの検知を開始します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            g_Work.deviceIndex = data.deviceIndex;
            DoStartDetectionForTagRead(data);
        }

        // B ボタンで NFP ライブラリを終了します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.Finalize();
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_Finalize;
        }

        // 上下ボタンでタグを検知する NFC デバイスを選択します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask) )
        {
            data.deviceIndex = data.deviceIndex > 0 ?
                (data.deviceIndex - 1) : (data.deviceCount - 1);
        }
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask) )
        {
            data.deviceIndex = data.deviceIndex < (data.deviceCount - 1) ?
                (data.deviceIndex + 1) : 0;
        }
    }

    // タグの検知を開始しています。
    void UpdateStateStartDetection(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグの検知が開始したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                data.state = State_TagSearch;
            }
            else
            {
                data.state = State_Error;
                data.lastResult = result;
            }
        }
    }

    // タグの検知を停止しています。
    void UpdateStateStopDetection(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグの検知が停止したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            OnTagStopped(data);
        }
    }

    // タグの喪失待機中です。
    void UpdateStateTagRelease(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグが喪失したら状態遷移します。
        if( GetSelectedDeactivateEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(100)) )
        {
            OnTagDeactivated(data);
        }
    }

    // タグを検索中です。
    void UpdateStateTagSearch(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグを発見したら情報を取得して状態遷移します。
        if( GetSelectedActivateEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(100)) )
        {
            OnTagActivated(data);
        }

        // B ボタンでタグの検知を停止します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            DoStopDetectionForNext(data, State_DeviceSelect);
        }

        // タグを検索中、あるいはタグ検出後にデバイスの状態が変わってしまった可能性がある場合
        else if(g_Work.availabilityChangeEvent->TryWait())
        {
            // デバイスの状態を確認します。
            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.GetDeviceState(&g_Work.deviceState, GetSelectedDeviceHandle());
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_DeviceCheck;
        }
    }

    // NFC デバイスの状態を確認中です。
    void UpdateStateDeviceCheck(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            if(g_Work.deviceState != nn::nfp::DeviceState_Search
               && g_Work.deviceState != nn::nfp::DeviceState_Active)
            {
                data.state = State_Error;
                if(g_Work.deviceState == nn::nfp::DeviceState_Deactive)
                {
                    data.lastResult = nn::nfp::ResultNeedRestart();
                }
                else
                {
                    data.lastResult = nn::nfp::ResultNfcDeviceNotFound();
                }
            }
            else
            {
                // タグを検索中に戻ります。
                data.state = State_TagSearch;
            }
        }
    }

    // タグを読み込み中です。
    void UpdateStateTagRead(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 読み込みが完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result  result = accessor.GetLastResult();
            bool isModelUpdated = false;

            // 読み込み結果をコピーして保存します。
            bool existResult;
            nn::Result commandResult;
            existResult = accessor.GetResult(&commandResult, Command_GetTagInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.tagInfo = g_Work.tagInfo;
            }
            else
            {
                std::memset(&data.tagInfo, 0, sizeof(data.tagInfo));
            }
            existResult = accessor.GetResult(&commandResult, Command_GetModelInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.modelInfo = g_Work.modelInfo;
                isModelUpdated = true;
            }
            else
            {
                std::memset(&data.modelInfo, 0, sizeof(data.modelInfo));
            }
            existResult = accessor.GetResult(&commandResult, Command_GetCommonInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.commonInfo = g_Work.commonInfo;
            }
            else
            {
                std::memset(&data.commonInfo, 0, sizeof(data.commonInfo));
            }
            existResult = accessor.GetResult(&commandResult, Command_GetRegisterInfo);
            if( existResult && commandResult.IsSuccess() )
            {
                data.registerInfo = g_Work.registerInfo;
            }
            else
            {
                std::memset(&data.registerInfo, 0, sizeof(data.registerInfo));
            }
            existResult = accessor.GetResult(&commandResult, Command_GetApplicationArea);
            if( existResult && commandResult.IsSuccess() )
            {
                data.counter = g_Work.applicationAreaData.counter;
                data.counter = ConvertFromTagByteOrder(data.counter);
            }
            else
            {
                data.counter = 0;
            }

            // 読み込み処理の結果に応じて状態遷移します。
            if( isModelUpdated && !IsKnownAmiibo(data.modelInfo.characterId) )
            {
                // Character ID からアプリが利用可能な amiibo ではないと判断されました。
                WaitTagReleaseForNext(data, State_AmiiboNotSupported);
            }
            else if( result.IsSuccess() )
            {
                // 利用可能な amiibo の情報を取得しました。
                WaitTagReleaseForNext(data, State_TagEnabled);
            }
            else if( result <= nn::nfp::ResultNeedCreate() )
            {
                // タグにアプリケーション領域が存在しません。作成する必要があります。
                // ただし、作成する前にアプリが対応する amiibo であることを確認します。
                WaitTagReleaseForNext(data, State_NeedCreate);
            }
            else if( result <= nn::nfp::ResultNeedRestore() )
            {
                // タグ上のデータが壊れていますがバックアップデータが存在するため復旧可能です。
                WaitTagReleaseForNext(data, State_NeedRestore);
            }
            else if( result <= nn::nfp::ResultNeedFormat() )
            {
                // タグ上のデータが壊れていますがバックアップデータが存在せず初期化が必要です。
                WaitTagReleaseForNext(data, State_NeedFormat);
            }
            else if( result <= nn::nfp::ResultNeedRegister() )
            {
                // タグに初期登録情報が登録されていません。登録が必要です。
                WaitTagReleaseForNext(data, State_NeedRegister);
            }
            else if( result <= nn::nfp::ResultAccessIdMisMatch() )
            {
                // 既にタグ上に他のアプリケーションが作成したデータが存在します。
                // 利用するためにはアプリケーション領域を削除しなければなりません。
                WaitTagReleaseForNext(data, State_NeedDelete);
            }
            else
            {
                WaitTagReleaseForError(data, result);
            }
        }
    }

    // タグを確認中です。
    void UpdateStateTagCheck(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // タグ情報を取得したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                if( IsEqualId(g_Work.tagInfo.tagId, data.tagInfo.tagId) )
                {
                    OnTagChecked(data);
                }
                else
                {
                    WaitTagReleaseForError(data, nn::nfp::ResultNeedRestart());
                }
            }
            else
            {
                WaitTagReleaseForError(data, result);
            }
        }
    }

    // タグを書き込み中です。
    void UpdateStateTagWrite(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 書き込みが完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                WaitTagReleaseForNext(data, State_DeviceSelect);
            }
            else
            {
                WaitTagReleaseForError(data, result);
            }
        }
    }

    // 利用可能なタグの情報を取得できた状態です。
    void UpdateStateTagEnabled(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでタグのカウンタをインクリメントします。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](ApplicationData& data)
            {
                g_Work.applicationAreaData.counter = ConvertToTagByteOrder(data.counter + 1);

                TagAccessor& accessor = TagAccessor::GetInstance();
                accessor.Mount(GetSelectedDeviceHandle());
                accessor.OpenApplicationArea(GetSelectedDeviceHandle(), AccessId);
                accessor.SetApplicationArea(
                        GetSelectedDeviceHandle(),
                        &g_Work.applicationAreaData,
                        sizeof(g_Work.applicationAreaData)
                    );
                accessor.Flush(GetSelectedDeviceHandle());
                accessor.Unmount(GetSelectedDeviceHandle());
                accessor.Run(&g_Work.processFinishEvent);
            };
            DoStartDetectionForTagWrite(data, writer);
        }

        // B ボタンで NFC デバイス選択に戻ります。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }
    }

    // アプリケーション領域を作成する必要があります。
    void UpdateStateNeedCreate(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでアプリケーション領域を作成します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](ApplicationData& data)
            {
                NN_UNUSED(data);
                g_Work.applicationAreaData.counter = ConvertToTagByteOrder(InitialCounter);

                TagAccessor& accessor = TagAccessor::GetInstance();
                accessor.Mount(GetSelectedDeviceHandle());
                accessor.CreateApplicationArea(
                        GetSelectedDeviceHandle(),
                        AccessId,
                        &g_Work.applicationAreaData,
                        sizeof(g_Work.applicationAreaData)
                    );
                accessor.Unmount(GetSelectedDeviceHandle());
                accessor.Run(&g_Work.processFinishEvent);
            };
            DoStartDetectionForTagWrite(data, writer);
        }

        // B ボタンで NFC デバイス選択に戻ります。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }
    }

    // タグの復旧が必要です。
    void UpdateStateNeedRestore(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンでバックアップデータからのタグの復旧を開始します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            auto writer = [](ApplicationData& data)
            {
                NN_UNUSED(data);
                TagAccessor& accessor = TagAccessor::GetInstance();
                accessor.Restore(GetSelectedDeviceHandle());
                accessor.Run(&g_Work.processFinishEvent);
            };
            DoStartDetectionForTagWrite(data, writer);
        }

        // B ボタンで NFC デバイス選択に戻ります。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンで amiibo 設定経由でタグを復旧します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.amiiboSettingsData.startParam, 0, sizeof(g_Work.amiiboSettingsData.startParam));
            g_Work.amiiboSettingsData.startParam.deviceHandle = *GetSelectedDeviceHandle();

            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.StartRestorer(
                    &g_Work.amiiboSettingsData.deviceHandle,
                    &g_Work.amiiboSettingsData.startParam,
                    &g_Work.tagInfo
                );
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // タグの初期化が必要です。
    void UpdateStateNeedFormat(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // この状態に陥るとアプリケーションからタグを復旧することはできません。
        // 以前 amiibo を使用したことのある本体で復旧できる可能性があることと、
        // 復旧せずに使用する場合は本体設定の amiibo設定で初期化できることを通知してください。

        // B ボタンで NFC デバイス選択に戻ります。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }
    }

    // 初期登録情報を登録する必要があります。
    void UpdateStateNeedRegister(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // B ボタンで NFC デバイス選択に戻ります。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンで初期登録情報を登録するために amiibo 設定を実行します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.amiiboSettingsData.startParam, 0, sizeof(g_Work.amiiboSettingsData.startParam));
            g_Work.amiiboSettingsData.startParam.deviceHandle = *GetSelectedDeviceHandle();

            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.StartNicknameAndOwnerSettings(
                    &g_Work.amiiboSettingsData.deviceHandle,
                    &g_Work.amiiboSettingsData.isRegistered,
                    &g_Work.registerInfo,
                    &g_Work.amiiboSettingsData.startParam,
                    &g_Work.tagInfo
                );
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // アプリケーション領域を削除する必要があります。
    void UpdateStateNeedDelete(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // B ボタンで NFC デバイス選択に戻ります。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }

        #if defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
        // X ボタンでアプリケーション領域を削除するために amiibo 設定を実行します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask) )
        {
            std::memset(&g_Work.amiiboSettingsData.startParam, 0, sizeof(g_Work.amiiboSettingsData.startParam));
            g_Work.amiiboSettingsData.startParam.deviceHandle = *GetSelectedDeviceHandle();

            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.StartGameDataEraser(
                    &g_Work.amiiboSettingsData.deviceHandle,
                    &g_Work.amiiboSettingsData.startParam,
                    &g_Work.tagInfo
                );
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_AmiiboSettings;
        }
        #endif // defined( NFPDEMO_ENABLE_AMIIBO_SETTING )
    }

    // アプリケーションが対応していない amiibo です。
    void UpdateStateAmiiboNotSupported(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // サンプルプログラムは amiibo データリストに登録されている一部のキャラクターにしか対応しておらず、
        // 非対応のキャラクターを書き込んだ amiibo に対してはアプリケーション専用領域の読み書きができません。
        // 新しい amiibo に対応する場合は、TagUtility.cpp の CharacterIdList と SampleCharacterName に定義を追加してください。

        // B ボタンで NFC デバイス選択に戻ります。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            data.state = State_DeviceSelect;
        }
    }

    // amiibo 設定の起動中です。
    void UpdateStateAmiiboSettings(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // amiibo 設定が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            nn::Result result = accessor.GetLastResult();
            if( result.IsSuccess() )
            {
                DoStartDetectionForTagRead(data);
            }
            else
            {
                data.state = State_Error;
                data.lastResult = result;
            }
        }
    }

    // NFP ライブラリを終了中です。
    void UpdateStateFinalize(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 終了処理が完了したら状態遷移します。
        if( nn::os::TryWaitEvent(&g_Work.processFinishEvent) )
        {
            TagAccessor::DestroyInstance();
            nn::os::FinalizeEvent(&g_Work.processFinishEvent);
            data.state = State_None;
            DestroySystemEvents();
            delete g_Work.availabilityChangeEvent;
        }
    }

    // エラー状態です。
    void UpdateStateError(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        // A ボタンで NFC デバイス取得、または、NFC デバイス選択からやり直します。
        if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask) )
        {
            if( g_Work.deviceCount > 0 )
            {
                // NFC デバイス喪失なので、NFC デバイス取得からやり直します。
                if( data.lastResult <= nn::nfp::ResultNfcDeviceNotFound() )
                {
                    data.state = State_Initialized;
                    DestroySystemEvents();
                }

                // NFC デバイス取得済なので、NFC デバイス選択に戻ります。
                else
                {
                    data.state = State_DeviceSelect;
                }
            }

            // NFC デバイス未取得なので、NFC デバイス取得からやり直します。
            else
            {
                data.state = State_Initialized;
            }
            data.lastResult = nn::ResultSuccess();
        }

        // B ボタンで NFP ライブラリを終了します。
        else if( HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask) )
        {
            TagAccessor& accessor = TagAccessor::GetInstance();
            accessor.Finalize();
            accessor.Run(&g_Work.processFinishEvent);
            data.state = State_Finalize;
        }
    }

    // 終了状態です。
    void UpdateStateExit(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);
    }

    //--------------------------------------------------------------------------
    //  状態処理関数の取得
    //--------------------------------------------------------------------------
    struct UpdateStateEntry
    {
        State           state;
        UpdateStateFunc func;
    };

    const UpdateStateEntry UpdateStateTable[] = {
        {   State_None,                 UpdateStateNone             },
        {   State_Initializing,         UpdateStateInitializing     },
        {   State_Initialized,          UpdateStateInitialized      },
        {   State_DeviceListing,        UpdateStateDeviceListing    },
        {   State_DeviceInitialize,     UpdateStateDeviceInitialize },
        {   State_DeviceSelect,         UpdateStateDeviceSelect     },
        {   State_StartDetection,       UpdateStateStartDetection   },
        {   State_StopDetection,        UpdateStateStopDetection    },
        {   State_TagRelease,           UpdateStateTagRelease       },
        {   State_TagSearch,            UpdateStateTagSearch        },
        {   State_DeviceCheck,          UpdateStateDeviceCheck      },
        {   State_TagRead,              UpdateStateTagRead          },
        {   State_TagCheck,             UpdateStateTagCheck         },
        {   State_TagWrite,             UpdateStateTagWrite         },
        {   State_TagEnabled,           UpdateStateTagEnabled       },
        {   State_NeedCreate,           UpdateStateNeedCreate       },
        {   State_NeedRestore,          UpdateStateNeedRestore      },
        {   State_NeedFormat,           UpdateStateNeedFormat       },
        {   State_NeedRegister,         UpdateStateNeedRegister     },
        {   State_NeedDelete,           UpdateStateNeedDelete       },
        {   State_AmiiboNotSupported,   UpdateStateAmiiboNotSupported },
        {   State_AmiiboSettings,       UpdateStateAmiiboSettings   },
        {   State_Finalize,             UpdateStateFinalize         },
        {   State_Error,                UpdateStateError            },
        {   State_Exit,                 UpdateStateExit             },
    };
    const int UpdateStateCountMax = sizeof(UpdateStateTable) / sizeof(UpdateStateTable[0]);
    NN_STATIC_ASSERT( UpdateStateCountMax == static_cast<int>(State_CountMax) );

    // 現在の状態を処理する関数を取得します。
    UpdateStateFunc GetUpdateStateFunc(State state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < UpdateStateCountMax );
        NN_ASSERT( UpdateStateTable[index].state == state );
        return UpdateStateTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    void UpdateState(
            ApplicationData& data
        ) NN_NOEXCEPT
    {
        data.frame += 1;
        State lastState = data.state;

        (*GetUpdateStateFunc(data.state))(data);
        if( lastState != data.state )
        {
            data.transitFrame = data.frame;
        }
    }

    void Cancel(ApplicationData& data) NN_NOEXCEPT
    {
        if( TagAccessor::HasInstance() )
        {
            // 現在実行中の処理だけを待って、以降の登録されている処理をキャンセルします。
            TagAccessor::GetInstance().Cancel();

            // ライブラリ未初期化、デバイス未取得、タグアクセス前に応じた状態に移行させます。
            if( nn::nfp::GetState() == nn::nfp::State_None )
            {
                data.state = State_None;
                DestroySystemEvents();
            }
            else if( g_Work.deviceCount == 0 )
            {
                data.state = State_Initialized;
            }
            else
            {
                data.state = State_DeviceSelect;
            }
        }
    }

    void Finalize(ApplicationData& data) NN_NOEXCEPT
    {
        if( TagAccessor::HasInstance() )
        {
            // 現在実行中の処理だけを待って、以降の登録されている処理をキャンセルします。
            TagAccessor::GetInstance().Cancel();

            // ライブラリ未初期化なら、応じた状態に移行させます。
            if( nn::nfp::GetState() == nn::nfp::State_None )
            {
                data.state = State_None;
                DestroySystemEvents();
            }

            // ライブラリが初期化済みなら終了させます。
            else
            {
                TagAccessor::GetInstance().Finalize();
                TagAccessor::GetInstance().Run(&g_Work.processFinishEvent);
                data.state = State_Finalize;

                // 終了処理が完了するまで待ちます。
                while( data.state != State_None )
                {
                    UpdateStateFinalize(data);
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                }
            }
        }
    }

}} // end of namespace nns::nfp
