﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{NfcPassThroughSimple.cpp,PageSampleNfcPassThroughSimple}

    @brief
    NFC ライブラリによる NFC タグへのパススルーアクセス機能のサンプルプログラム
 */

/**
    @page PageSampleNfcPassThroughSimple NFC ライブラリによる NFC タグへのパススルーアクセス機能のサンプル
    @tableofcontents

    @brief
    NFC ライブラリによる NFC タグへのパススルーアクセス機能のサンプルプログラムの解説です。

    @section PageSampleNfcPassThroughSimple_SectionBrief 概要
    NFC ライブラリによる NFC タグへのパススルーアクセス機能は、NFC（Near Field Communication）を介して、
    NFC タグへコマンド送信を行う機能です。
    NFC ライブラリによる NFC タグへのパススルーアクセス機能を用いることで、アプリケーションと NFC タグを連携させることが可能です。

    このサンプルプログラムは、以下の NFC タグ操作に対応しています。
    @n
    @li 読み書き

    サンプルプログラムでは NTAG21x アドレス 4 の先頭 1 バイトにカウンタを書き込みます。
    書き込みを行うたびに、カウンタを 1 つずつインクリメントしていきます。

    @section PageSampleNfcPassThroughSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/NfcPassThroughSimple
    Samples/Sources/Applications/NfcPassThroughSimple @endlink 以下にあります。

    @section PageSampleNfcPassThroughSimple_SectionNecessaryEnvironment 必要な環境
    NFC タグを用意してください。

    @section PageSampleNfcPassThroughSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、NFC ライブラリの状態が表示されます。
    ゲームパッドの操作により、NFC ライブラリの状態を変化させることができます。

    @section PageSampleNfcPassThroughSimple_SectionPrecaution 注意事項
    サンプルでは NTAG21x タグを想定しています。
    想定している NFC タグと異なる場合は、プログラム内で指定しているコマンドを適切に書き変えてください。

    @section PageSampleNfcPassThroughSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleNfcPassThroughSimple_SectionDetail 解説
    NfcPassThroughSimple では NFC タグの状態に関わらずカウンタのインクリメント処理を実行します。

 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include "Graphics.h"
#include "HidController.h"

#include "Config.h"
#include "UpdateState.h"
#include "WriteState.h"

namespace {

    // Graphics.cpp で malloc しているヒープの内訳
    //  VisiblePoolMemory       16 MB
    //  InvisiblePoolMemory     20 MB
    //  for CommandBuffer       32 MB
    //  nv::InitializeGraphics   8 MB
    //  for DebugFont            2 MB
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    // アプリケーションで扱うデータと状態です。
    nns::nfc::pt::ApplicationData g_Data;

} // namespace

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    InitializeGraphicSystem();
    InitializeHidController();

    while( NN_STATIC_CONDITION(true) )
    {
        UpdateHidController();

        nns::nfc::pt::UpdateState(g_Data);

        BeginText();
        nns::nfc::pt::WriteState(g_Data);
        EndText();

        if( g_Data.state == nns::nfc::pt::State_Exit )
        {
            break;
        }
    }

    FinalizeHidController();
    FinalizeGraphicSystem();
}
