﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Version.h>
#include <cstdlib>
#include "../Common/DebugFont.h"


namespace
{
struct SaveDataStructure
{
    int bootCount;
};
int IncrementBootCount()
{
    const char* MountName = "save";
    const char* SaveFilePath = "save:/bootCount.txt";

    auto result = nn::fs::MountSaveDataForDebug(MountName);
    NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::MountSaveData");

    const size_t FileSize = sizeof(SaveDataStructure);

    // create file if not exist
    {
        nn::fs::DirectoryEntryType type;
        result = nn::fs::GetEntryType(&type, SaveFilePath);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            result = nn::fs::CreateFile(SaveFilePath, FileSize);
            NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::CreateFile(%s)", SaveFilePath);

            nn::fs::FileHandle handle;
            result = nn::fs::OpenFile(&handle, SaveFilePath, nn::fs::OpenMode_Write);
            NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::OpenFile(%s)", SaveFilePath);

            SaveDataStructure data = { 0 };
            result = nn::fs::WriteFile(handle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::WriteFile(%s)", SaveFilePath);

            nn::fs::CloseFile(handle);
        }
    }

    // read file and increment
    int returnValue = -1;
    {
        SaveDataStructure data;

        nn::fs::FileHandle handle;
        result = nn::fs::OpenFile(&handle, SaveFilePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
        NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::OpenFile(%s) for read and increment", SaveFilePath);

        size_t readSize;
        result = nn::fs::ReadFile(&readSize, handle, 0, &data, sizeof(data));
        NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::ReadFile(%s)", SaveFilePath);
        NN_ASSERT_EQUAL(readSize, sizeof(data));

        returnValue = data.bootCount;

        data.bootCount++;
        result = nn::fs::WriteFile(handle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to WriteFile(%s) for increment data", SaveFilePath);

        nn::fs::CloseFile(handle);

        result = nn::fs::Commit(MountName);
        NN_ABORT_UNLESS(result.IsSuccess(), "[ERROR] failed to nn::fs::Commit");
    }

    nn::fs::Unmount(MountName);
    return returnValue;
}

const char* GetTargetName()
{
    static const char* DefaultName = "no name";
    const char* targetName = DefaultName;

#if (NN_NX_ADDON_VERSION_MAJOR == 0) && (NN_NX_ADDON_VERSION_MINOR <= 7)
    auto argc = nn::GetArgc();
    auto argv = nn::GetArgv();
#else
    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
#endif

    for(int i = 1; i < argc; i++)
    {
        if(strncmp(argv[i], "--target", strlen("--target")) == 0)
        {
            targetName = argv[++i];
            break;
        }
    }
    return targetName;
}

int GetLineCountPerFrame()
{
    int lineCountPerFrame = 0;

#if (NN_NX_ADDON_VERSION_MAJOR == 0) && (NN_NX_ADDON_VERSION_MINOR <= 7)
    auto argc = nn::GetArgc();
    auto argv = nn::GetArgv();
#else
    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
#endif

    for(int i = 1; i < argc; i++)
    {
        if(strncmp(argv[i], "--linesperframe", strlen("--linesperframe")) == 0)
        {
            lineCountPerFrame = atoi(argv[++i]);
            break;
        }
    }
    return lineCountPerFrame == 0 ? 1 : lineCountPerFrame;
}

int GetDurationInFrames()
{
    int durationInFrames = 0;

#if (NN_NX_ADDON_VERSION_MAJOR == 0) && (NN_NX_ADDON_VERSION_MINOR <= 7)
    auto argc = nn::GetArgc();
    auto argv = nn::GetArgv();
#else
    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
#endif

    for(int i = 1; i < argc; i++)
    {
        if(strncmp(argv[i], "--frames", strlen("--frames")) == 0)
        {
            durationInFrames = std::atoi(argv[++i]);
            break;
        }
    }
    return durationInFrames;
}

bool GetForceCrash()
{
    bool forceCrash = false;

#if (NN_NX_ADDON_VERSION_MAJOR == 0) && (NN_NX_ADDON_VERSION_MINOR <= 7)
    auto argc = nn::GetArgc();
    auto argv = nn::GetArgv();
#else
    auto argc = nn::os::GetHostArgc();
    auto argv = nn::os::GetHostArgv();
#endif

    for(int i = 1; i < argc; i++)
    {
        if(strncmp(argv[i], "--forcecrash", strlen("--forcecrash")) == 0)
        {
            forceCrash = true;
            break;
        }
    }
    return forceCrash;
}

void Print(const char* targetName, int bootCount, int count)
{
    const nn::util::Color4u8Type Color = { { 255, 255, 255, 255 } };

    nn::gfx::util::DebugFontTextWriter& writer = nns::MultipleDevkits::DebugFont::GetDebugFontWriter();
    const float scale = 4.0f;
    writer.SetScale(scale, scale);
    writer.SetTextColor(Color);
    writer.SetCursor(10, 10);

    writer.SetFixedWidthEnabled(true);

#if (NN_NX_ADDON_VERSION_MAJOR == 0) && (NN_NX_ADDON_VERSION_MINOR <= 7)
    writer.SetFixedWidth(10); // SIGLO-28257
#else
    writer.SetFixedWidth(10 * scale);
#endif
    writer.Print("[TargetName] %s\n", targetName);
    writer.Print("[BootCount]  %-4d\n", bootCount);
    writer.Print("[FrameCount] %-4d (60fps)\n", count);
}

void LogForDuration(int bootCount, int lineCountPerFrame, int durationInFrames)
{
    auto targetName = GetTargetName();
    auto count = 0;
    bool logForever = durationInFrames == 0;

    while (logForever || count < durationInFrames)
    {
        for (auto i = 0; i < lineCountPerFrame; i++)
        {
            NN_LOG("[%-20s:%4d:%6d:%3d] hello, world\n", targetName, bootCount, count, i);
        }
        Print(targetName, bootCount, count);
        nns::MultipleDevkits::DebugFont::Update();
        count++;
    }
}

void ExitByAbort(const char* targetName, int bootCount)
{
    NN_ABORT("[%-20s:%4d] !!!ABORT!!!", targetName, bootCount);
}

} // anonymous namespace





extern "C" void nnMain()
{
    const char* targetName = GetTargetName();
    int lineCountPerFrame = GetLineCountPerFrame();
    int durationInFrames = GetDurationInFrames();
    bool forceCrash = GetForceCrash();

    auto bootCount = IncrementBootCount();
    NN_LOG("\n-------------------- %4d --------------------\n", bootCount);

    if(forceCrash)
    {
        ExitByAbort(targetName, bootCount);
    }
    else
    {
        nns::MultipleDevkits::DebugFont::Initialize();
        LogForDuration(bootCount, lineCountPerFrame, durationInFrames);

        NN_LOG("[%-20s:%4d] Execution finished. Application entering idle state.\n", targetName, bootCount);
        while(1)
        {
            // Do not allow application to terminate.
        }

        // We will never hit this code, but just for symmetry's sake...
        nns::MultipleDevkits::DebugFont::Finalize();
    }
}
