﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief MoviePlayer API implementation.
 */

/**
 * @brief
 * MoviePlayer class for playback of media file.
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MoviePlayer is provide an interface for clients to paly media files. The MoviePlayer supports
 * audio, video and AV playback. Trick modes like pause, resume and seek are also supported. MoviePlayer uses multiple
 * Nintendo SDK components to achieve media playback. Audio and video samples are extracted using movie::Extractor
 * APIs. Decoding is done using movie::Decoders. Audio is played back using nn::AudioOut APIs. Video playback is done
 * using NVN APIs.
 *
 * Setup:
 *     An application creates an instance of @class MoviePlayer. movie::DecoderMode and movie::DecoderOutputFormat are
 * passed to the player using constructor parameters. When @class MoviePlayer is initialized, a thread is created to perform
 * asynchronous task. MoviePlayer events and movie::DecoderEvents are initialized. Client can also register MoviePlayer
 * events which will be signaled when signaling criteria is met. StateChangedEvent and ErrorEvent are needed. MoviePlayer
 * will signal the event when player transitions states or in the event of an error. Typically application will start
 * playback and waits for State_PlaybackCompleted state changed event. Next application needs to set a media path for
 * playback using SetDataSource() API. When Prepare() API called Player will create necessary components for media playback.
 * Player creates movie::Extractors for extraction of audio video samples, movie::decoders for decoding of audio video frames.
 * Input handlers are created to read data from extractor. Output handlers are created to get decoded audio and video samples.
 * Decoder event handler is created to receive decoders events. Finally audio and video renderer are created.
 *
 * Main Processing:
 *     To start playback of the media file, application needs to call Start() API. All player components transitions into
 * running state. Decoders starts sending input buffer events, which will be received by decoder event handler. Input buffers
 * are filled by input handlers by reading data from extractors. Filled input buffers are sent to the decoder. Decoders will
 * send output buffer available events. The output video and audio frames are acquired by output handlers and are sent for
 * rendering. When all frames are rendered, player will transition into State_PlaybackCompleted state. An event is signaled to
 * the application. Application can call Stop() API to finish playback. There is an option to replay by calling Start() again.
 *
 * Teardown:
 *    When @class MoviePlayer instance is deleted, all player resources are released.
 *
 */
#include "MoviePlayer.h"

static bool IsPreferredLanguageTrack(const char* lang);

MoviePlayer::MoviePlayer()
    : m_LastSeekPositionUs(0),
      m_State(State_NotInitialized),
      m_ErrorEvent(nullptr),
      m_StateChangedEvent(nullptr),
      m_NativeWindow(nullptr),
      m_Loop(false),
      m_Extractor(nullptr),
      m_AudioDecoder(nullptr),
      m_AudioDecoderType(movie::DecoderType_Unknown),
      m_VideoDecoder(nullptr),
      m_VideoDecoderType(movie::DecoderType_Unknown),
      m_MovieVideoInputHandler(nullptr),
      m_MovieVideoOutputHandler(nullptr),
      m_MovieAudioInputHandler(nullptr),
      m_MovieAudioOutputHandler(nullptr),
      m_NumberOfTracks(-1),
      m_AudioSampleRate(-1),
      m_NumberOfAudioChannels(-1),
      m_AudioTrackNumber(-1),
      m_VideoTrackNumber(-1),
      m_Width(-1),
      m_Height(-1),
      m_MovieDecoderEventHandler(nullptr),
      m_MediaClock(nullptr),
      m_CoreMask(0),
      m_ThreadStackSize(1024 * 128),
      m_ThreadStack(nullptr),
      m_LastError(movie::Status_Success),
      m_AudioPlaybackComplete(true),
      m_VideoPlaybackComplete(true),
      m_EventsInitialized(false),
      m_ThreadCreated(false),
      m_ThreadStarted(false),
      m_MediaDurationUs(0),
      m_AudioDecoderMode(movie::DecoderMode_Cpu),
      m_VideoDecoderMode(movie::DecoderMode_Cpu),
      m_AudioDecoderOutputFormat(movie::DecoderOutputFormat_AudioPcm16),
      m_VideoDecoderOutputFormat(movie::DecoderOutputFormat_VideoColorNv12)
{
}

MoviePlayer::~MoviePlayer()
{
    State currentState = State_NotInitialized;
    GetState(&currentState);
    if( currentState != State_NotInitialized )
    {
        if( m_ThreadCreated == true )
        {
            if( m_ThreadStarted == true )
            {
                nn::os::SignalEvent(&m_ThreadExitEvent);
                nn::os::WaitThread(&m_ThreadType);
            }
            nn::os::DestroyThread(&m_ThreadType);
        }
        if( m_EventsInitialized == true )
        {
            nn::os::UnlinkMultiWaitHolder(&m_PrepareEventHolder);
            nn::os::UnlinkMultiWaitHolder(&m_SeekEventHolder);
            nn::os::UnlinkMultiWaitHolder(&m_AudioRenderCompleteEventHolder);
            nn::os::UnlinkMultiWaitHolder(&m_VideoRenderCompleteEventHolder);
            nn::os::UnlinkMultiWaitHolder(&m_ThreadExitEventHolder);

            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PrepareEventHolder));
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_SeekEventHolder));
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_AudioRenderCompleteEventHolder));
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_VideoRenderCompleteEventHolder));
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_ThreadExitEventHolder));

            nn::os::FinalizeMultiWaitHolder(&m_PrepareEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_SeekEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_AudioRenderCompleteEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_VideoRenderCompleteEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_ThreadExitEventHolder);

            // Finalize audio decoder events
            nn::os::FinalizeEvent(&m_AudioErrorEvent);
            nn::os::FinalizeEvent(&m_AudioFormatChangedEvent);
            nn::os::FinalizeEvent(&m_AudioInputEvent);
            nn::os::FinalizeEvent(&m_AudioOutputEvent);

            // Finalize video decoder events
            nn::os::FinalizeEvent(&m_VideoErrorEvent);
            nn::os::FinalizeEvent(&m_VideoFormatChangedEvent);
            nn::os::FinalizeEvent(&m_VideoInputEvent);
            nn::os::FinalizeEvent(&m_VideoOutputEvent);

            nn::os::FinalizeEvent(&m_AudioRenderCompleteEvent);
            nn::os::FinalizeEvent(&m_VideoRenderCompleteEvent);
            nn::os::FinalizeEvent(&m_ThreadExitEvent);

            nn::os::FinalizeMultiWait(&m_PlayerMultiWait);
        }
        if( m_ThreadStack != nullptr )
        {
            free(m_ThreadStack);
            m_ThreadStack = nullptr;
        }
        if( m_MovieDecoderEventHandler != nullptr )
        {
            delete m_MovieDecoderEventHandler;
            m_MovieDecoderEventHandler = nullptr;
        }
        if( m_MovieVideoOutputHandler != nullptr )
        {
            delete m_MovieVideoOutputHandler;
            m_MovieVideoOutputHandler = nullptr;
        }
        if( m_MovieAudioOutputHandler != nullptr )
        {
            delete m_MovieAudioOutputHandler;
            m_MovieAudioOutputHandler = nullptr;
        }
        DeletePlayerOwnedObjects();
    }
}

movie::Status MoviePlayer::Initialize()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Initialized) )
    {
        return movie::Status_Success;
    }
    if( m_EventsInitialized == false )
    {
        // Initialize multiWait for receivng decoder events.
        nn::os::InitializeMultiWait(&m_PlayerMultiWait);

        // Initialize Events to register with audio decoder
        nn::os::InitializeEvent(&m_AudioErrorEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_AudioFormatChangedEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_AudioInputEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_AudioOutputEvent, false, nn::os::EventClearMode_ManualClear);

        nn::os::InitializeEvent(&m_PrepareEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_SeekEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_AudioRenderCompleteEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_VideoRenderCompleteEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_ThreadExitEvent, false, nn::os::EventClearMode_ManualClear);

        RegisterPlayerEvent(&m_PrepareEvent, PlayerEventType_Prepare);
        RegisterPlayerEvent(&m_SeekEvent, PlayerEventType_Seek);
        RegisterPlayerEvent(&m_AudioRenderCompleteEvent, PlayerEventType_AudioRenderComplete);
        RegisterPlayerEvent(&m_VideoRenderCompleteEvent, PlayerEventType_VideoRenderComplete);
        RegisterPlayerEvent(&m_ThreadExitEvent, PlayerEventType_ThreadExit);

        // Fill movie::DecoderEvents structure with audio events (for audio decoder)
        m_AudioDecoderEvents.errorEvent = &m_AudioErrorEvent;
        m_AudioDecoderEvents.formatChangedEvent = &m_AudioFormatChangedEvent;
        m_AudioDecoderEvents.inputBufferAvailableEvent = &m_AudioInputEvent;
        m_AudioDecoderEvents.outputBufferAvailableEvent = &m_AudioOutputEvent;

        // Initialize Events to register with video decoder
        nn::os::InitializeEvent(&m_VideoErrorEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_VideoFormatChangedEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_VideoInputEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_VideoOutputEvent, false, nn::os::EventClearMode_ManualClear);

        // Fill movie::DecoderEvents structure with video events (for video decoder)
        m_VideoDecoderEvents.errorEvent = &m_VideoErrorEvent;
        m_VideoDecoderEvents.formatChangedEvent = &m_VideoFormatChangedEvent;
        m_VideoDecoderEvents.inputBufferAvailableEvent = &m_VideoInputEvent;
        m_VideoDecoderEvents.outputBufferAvailableEvent = &m_VideoOutputEvent;

        m_EventsInitialized = true;
    }
    if( m_ThreadCreated == false )
    {
        // Create movie Player thread.
        m_ThreadStackSize = 1024 * 128;
        m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
        if( m_ThreadStack == NULL )
        {
            movieStatus = movie::Status_OutOfMemory;
        }
        else
        {
            nn::Result nnResult = nn::os::CreateThread(&m_ThreadType,
                &MoviePlayerThreadFunction,
                ( void* )this,
                m_ThreadStack,
                m_ThreadStackSize,
                nn::os::DefaultThreadPriority);
            if( nnResult.IsSuccess() )
            {
                m_ThreadCreated = true;
                nn::os::SetThreadName(&m_ThreadType, "MoviePlayerThread");
                nn::os::StartThread(&m_ThreadType);
                m_ThreadStarted = true;
            }
            else
            {
                movieStatus = movie::Status_FailedToCreateThread;
            }
        }
    }
    movieStatus = ValidateStateTransition(State::State_Initialized);
    if( movieStatus == movie::Status_Success )
    {
        SetState(State::State_Initialized);
    }
    return movieStatus;
}

movie::Status MoviePlayer::SetViNativeWindowHandle(nn::vi::NativeWindowHandle nativeWindowHandle)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( nativeWindowHandle != NULL )
    {
        m_NativeWindow = nativeWindowHandle;
        movieStatus = movie::Status_Success;
    }
    else
    {
        movieStatus = movie::Status_ErrorBadValue;
    }
    return movieStatus;
}

movie::Status MoviePlayer::RegisterEvent(EventType eventType, nn::os::EventType *event)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    switch( eventType )
    {
        case EventType_ErrorEvent:
            m_ErrorEvent = event;
            break;

        case EventType_StateChangedEvent:
            m_StateChangedEvent = event;
            break;

        default:
            break;
    }
    movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MoviePlayer::SetDataSource(const char* mediaPath)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( mediaPath == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    movieStatus = CreateExtractorForPlayer(mediaPath);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    movieStatus = m_Extractor->SetDataSource(mediaPath);
    if( movieStatus == movie::Status_Success )
    {
        movieStatus = UpdateTrackPropertiesForPlayer();
    }
    return movieStatus;
}

movie::Status MoviePlayer::SetDataSource(movie::ClientStreamReader &streamReader, const char* mediaExtension)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movieStatus = CreateExtractorForPlayer(mediaExtension);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    movieStatus = m_Extractor->SetDataSource(&streamReader, mediaExtension);
    if( movieStatus == movie::Status_Success )
    {
        movieStatus = UpdateTrackPropertiesForPlayer();
    }
    return movieStatus;
}

movie::Status MoviePlayer::SetDataSource(const void* buffer, int length, const char* mediaExtension)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movieStatus = CreateExtractorForPlayer(mediaExtension);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    movieStatus = m_Extractor->SetDataSource(buffer, length);
    if( movieStatus == movie::Status_Success )
    {
        movieStatus = UpdateTrackPropertiesForPlayer();
    }
    return movieStatus;
}

movie::Status MoviePlayer::Prepare(bool syncMode)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Preparing) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Preparing);
    if( movieStatus == movie::Status_Success )
    {
        SetState(State::State_Preparing);
    }
    else
    {
        return movieStatus;
    }
    if( syncMode == true )
    {
        OnPrepare();
        movieStatus = GetLastError();
    }
    else
    {
        nn::os::SignalEvent(&m_PrepareEvent);
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status MoviePlayer::Start()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Started) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Started);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    State currentState;
    movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }

    if( currentState == State::State_PlaybackCompleted )
    {
        HandlePlaybackCompleteForPlayer();
    }
    else
    {
        if( m_AudioDecoder != nullptr )
        {
            if( m_MovieAudioOutputHandler != nullptr )
            {
                m_MovieAudioOutputHandler->Initialize(m_MediaClock, &m_AudioRenderCompleteEvent);
            }
            m_AudioDecoder->Start();
        }
        if( m_VideoDecoder != nullptr )
        {
            if( m_MovieVideoOutputHandler != nullptr )
            {
                m_MovieVideoOutputHandler->Initialize(m_MediaClock, &m_VideoRenderCompleteEvent);
            }
            m_VideoDecoder->Start();
        }
        if( m_MovieDecoderEventHandler != nullptr )
        {
            m_MovieDecoderEventHandler->WaitForFormatChange();
        }
        if( m_AudioDecoder != nullptr )
        {
            movie::MediaData audioFormat;
            movieStatus = m_AudioDecoder->GetOutputFormat(&audioFormat);
            if( movieStatus == movie::Status_Success )
            {
                int32_t numChannels = 0;
                int32_t sampleRate = 0;
                audioFormat.FindInt32("channel-count", &numChannels);
                audioFormat.FindInt32("sample-rate", &sampleRate);
                if( numChannels > 0 )
                {
                    m_NumberOfAudioChannels = numChannels;
                }
                if( m_AudioSampleRate > 0 )
                {
                    m_AudioSampleRate = sampleRate;
                }
            }
            audioFormat.Clear();
            bool variableSizeAudioFrames = false;
            if( m_AudioDecoderType == movie::DecoderType_AudioVorbis )
            {
                variableSizeAudioFrames = true;
            }
            if( m_MovieAudioOutputHandler != nullptr )
            {
                m_MovieAudioOutputHandler->Open(m_AudioDecoder, m_AudioSampleRate, m_NumberOfAudioChannels, variableSizeAudioFrames);
            }
        }
        if( m_VideoDecoder != nullptr )
        {
            movie::MediaData videoFormat;
            movieStatus = m_VideoDecoder->GetOutputFormat(&videoFormat);
            if( movieStatus == movie::Status_Success )
            {
                int32_t width = 0;
                int32_t height = 0;
                videoFormat.FindInt32("width", &width);
                videoFormat.FindInt32("height", &height);
                if( width > 0 )
                {
                    m_Width = width;
                }
                if( height > 0 )
                {
                    m_Height = height;
                }
            }
            videoFormat.Clear();
            if( m_MovieVideoOutputHandler != nullptr )
            {
                m_MovieVideoOutputHandler->Open(m_VideoDecoder, m_Width, m_Height);
            }
        }
        // Wait untill audio and video renderer has data to render or maximum of 500ms (50 x 10) to start playback
        int waitTimeLoopCount = 0;
        int waitTimeLoopCountMax = 50;
        // Vorbis decoder needs more time to set up
        if( m_AudioDecoderType == movie::DecoderType_AudioVorbis )
        {
            waitTimeLoopCountMax = 125;
        }
        while( waitTimeLoopCount < waitTimeLoopCountMax )
        {
            bool canAudioStartRendering = true;
            bool canVideoStartRendering = true;
            waitTimeLoopCount ++;
            if( m_MovieVideoOutputHandler != nullptr )
            {
                canVideoStartRendering = m_MovieVideoOutputHandler->CanStartRendering();
            }
            if( m_MovieAudioOutputHandler != nullptr )
            {
                canAudioStartRendering = m_MovieAudioOutputHandler->CanStartRendering();
            }
            if( ( canAudioStartRendering == true ) && ( canVideoStartRendering == true ) )
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        if( m_MovieAudioOutputHandler != nullptr )
        {
            m_MovieAudioOutputHandler->Start();
        }
        if( m_MovieVideoOutputHandler != nullptr )
        {
            m_MovieVideoOutputHandler->Start(m_Width, m_Height);
        }
    }
    SetState(State::State_Started);
    movieStatus = movie::Status_Success;
    return movieStatus;
}//NOLINT(impl/function_size)

movie::Status MoviePlayer::Stop()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Stopped) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Stopped);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    if( m_MovieDecoderEventHandler != nullptr )
    {
        delete m_MovieDecoderEventHandler;
        m_MovieDecoderEventHandler = nullptr;
    }
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Pause();
        m_MovieVideoOutputHandler->Stop();
        m_MovieVideoOutputHandler->Close();
        m_MovieVideoOutputHandler->Finalize();
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Pause();
        m_MovieAudioOutputHandler->Stop();
        m_MovieAudioOutputHandler->Close();
        m_MovieAudioOutputHandler->Finalize();
    }
    if( m_VideoDecoder != nullptr )
    {
        m_VideoDecoder->Stop();
    }
    if( m_AudioDecoder != nullptr )
    {
        m_AudioDecoder->Stop();
    }
    DeletePlayerOwnedObjects();
    m_AudioTrackNumber = -1;
    m_VideoTrackNumber = -1;
    m_VideoDecoderType = movie::DecoderType_Unknown;
    m_AudioDecoderType = movie::DecoderType_Unknown;
    SetState(State::State_Stopped);
    movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MoviePlayer::Pause()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Paused) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Paused);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Pause();
    }
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Pause();
    }

    SetState(State::State_Paused);
    return movieStatus;
}

movie::Status MoviePlayer::Resume()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Started) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Started);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    m_MovieDecoderEventHandler->VideoInputBufferAvailableEvent();
    m_MovieDecoderEventHandler->VideoOutputBufferAvailableEvent();
    m_MovieDecoderEventHandler->AudioInputBufferAvailableEvent();
    m_MovieDecoderEventHandler->AudioOutputBufferAvailableEvent();

    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Resume();
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Resume();
    }

    SetState(State::State_Started);
    return movieStatus;
}

movie::Status MoviePlayer::GetPlaybackPosition(int64_t *playbackPositionUs) const
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( playbackPositionUs == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    *playbackPositionUs = 0;
    if( m_MovieAudioOutputHandler != nullptr )
    {
        movieStatus = m_MovieAudioOutputHandler->GetPlaybackPosition(playbackPositionUs);
    }
    else if( m_MovieVideoOutputHandler != nullptr )
    {
        movieStatus = m_MovieVideoOutputHandler->GetPlaybackPosition(playbackPositionUs);
    }
    return movieStatus;
}

movie::Status MoviePlayer::GetPlaybackDuration(int64_t *durationUs) const
{
    if( durationUs == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    else
    {
        *durationUs = m_MediaDurationUs;
        return movie::Status_Success;
    }
}

movie::Status MoviePlayer::SelectAudioTrack(int *trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movie::MediaData trackFormat;
    int trackId = -1;
    if( trackNumber == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    if( m_Extractor == nullptr )
    {
        return movie::Status_NullExtractor;
    }

    for( int i = 0; i < m_NumberOfTracks; i++ )
    {
        m_Extractor->GetTrackConfiguration(i, &trackFormat);
        const char *mime;
        if( trackFormat.FindString("mime", &mime) )
        {
            if( !strncasecmp("audio/", mime, 6) )
            {
                int32_t numChannels = 0;
                int32_t sampleRate = 0;
                trackId = i;
                trackFormat.FindInt32("channel-count", &numChannels);
                trackFormat.FindInt32("sample-rate", &sampleRate);
                if( sampleRate > 48000 )
                {
                    if( !strncasecmp("audio/mp4a-latm", mime, 15) )
                    {
                        trackId = -1;
                        movieStatus = movie::Status_UnsupportedSampleRate;
                        NN_LOG("\n MoviePlayer:: Unsupported sampleRate \n");
                    }
                }
                if( sampleRate < 16000)
                {
                    *trackNumber = -1;
                    movieStatus = movie::Status_UnsupportedMediaType;
                    NN_LOG("\n MoviePlayerSimple sample does not supports sample rate lower than 16kHz for audio playback\n");
                }
                if( numChannels > 6 )
                {
                    trackId = -1;
                    movieStatus = movie::Status_UnsupportedMediaType;
                    NN_LOG("\n MoviePlayer:: Unsupported number of channels \n");
                }


                const char *lang;
                if( trackFormat.FindString("media-language", &lang) )
                {
                    if(true == IsPreferredLanguageTrack(lang))
                    {
                        NN_LOG("\n MoviePlayer:: find language: track:%d, language:%s\n", trackId, lang);
                        break;
                    }
                }
                else
                {
                    NN_LOG("\n MoviePlayer:: cannot get language meta data from the extractor\n");
                }

            }
        }
    }
    if(trackId >= 0)
    {
        movieStatus = m_Extractor->SelectTrack(trackId);
        *trackNumber = trackId;
        m_AudioTrackNumber = trackId;
    }

    trackFormat.Clear();
    return movieStatus;
}

movie::Status MoviePlayer::SelectVideoTrack(int *trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movie::MediaData trackFormat;

    if( trackNumber == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    if( m_Extractor == nullptr )
    {
        return movie::Status_NullExtractor;
    }

    for( int i = 0; i < m_NumberOfTracks; i++ )
    {
        m_Extractor->GetTrackConfiguration(i, &trackFormat);
        const char *mime;
        if( trackFormat.FindString("mime", &mime) )
        {
            if( !strncasecmp("video/", mime, 6) )
            {
                movieStatus = m_Extractor->SelectTrack(i);
                *trackNumber = i;
                m_VideoTrackNumber = i;
                trackFormat.FindInt32("width", &m_Width);
                trackFormat.FindInt32("height", &m_Height);
                break;
            }
        }
    }
    trackFormat.Clear();
    return movieStatus;
}

movie::Status MoviePlayer::GetTrackCount(int *trackCount) const
{
    movie::Status movieStatus = movie::Status_UnknownError;
    int numberOfTracks = 0;

    if( trackCount == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }

    if( m_Extractor == nullptr )
    {
        return movie::Status_NullExtractor;
    }

    movieStatus = m_Extractor->GetTrackCount(&numberOfTracks);
    if( movieStatus == movie::Status_Success )
    {
        if( numberOfTracks <= 0 )
        {
            movieStatus = movie::Status_TrackNotFound;
        }
    }
    *trackCount = numberOfTracks;
    return movieStatus;
}

movie::Status MoviePlayer::SelectTrack(int trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movieStatus = ValidateTrackNumber(trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        if( m_Extractor != NULL )
        {
            movieStatus = m_Extractor->SelectTrack(trackNumber);
        }
        else
        {
            movieStatus = movie::Status_NullExtractor;
        }
    }
    return movieStatus;
}

movie::Status MoviePlayer::UnSelectTrack(int trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    movieStatus = ValidateTrackNumber(trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        if( m_Extractor != NULL )
        {
            movieStatus = m_Extractor->UnselectTrack(trackNumber);
        }
        else
        {
            movieStatus = movie::Status_NullExtractor;
        }
    }
    return movieStatus;
}

movie::Status MoviePlayer::GetTrackType(int trackNumber, TrackType *trackType) const
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( trackType == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    *trackType = TrackType::TrackType_Unknown;
    int trackCount = 0;
    movieStatus = GetTrackCount(&trackCount);
    if( movieStatus == movie::Status_Success )
    {
        if( ( trackNumber < trackCount ) && ( trackNumber >= 0 ) )
        {
            movie::MediaData trackFormat;
            if( m_Extractor != nullptr )
            {
                m_Extractor->GetTrackConfiguration(trackNumber, &trackFormat);
                const char *mime;
                if( trackFormat.FindString("mime", &mime) )
                {
                    if( !strncasecmp("audio/", mime, 6) )
                    {
                        *trackType = TrackType::TrackType_Audio;
                    }
                    else if( !strncasecmp("video/", mime, 6) )
                    {
                        *trackType = TrackType::TrackType_Video;
                    }
                }
            }
            else
            {
                movieStatus = movie::Status_NullExtractor;
            }
            trackFormat.Clear();
        }
        else
        {
            movieStatus = movie::Status_TrackNotFound;
        }
    }
    return movieStatus;
}

movie::Status MoviePlayer::Reset()
{
    movie::Status movieStatus = movie::Status_UnknownError;

    return movieStatus;
}

movie::Status MoviePlayer::SetLooping(bool loop)
{
    movie::Status status = movie::Status_Success;
    Mutex::ScopedLock lock(m_ApiLock);
    m_Loop = loop;
    return status;
}

movie::Status MoviePlayer::GetLooping(bool *loop) const
{
    movie::Status status = movie::Status_Success;
    if( loop == nullptr )
    {
        status = movie::Status_ErrorBadValue;
    }
    else
    {
        *loop = m_Loop;
    }
    return status;
}

movie::Status MoviePlayer::GetState(State *state)
{
    movie::Status status = movie::Status_Success;
    Mutex::ScopedLock lock(m_StateLock);
    if( state == nullptr )
    {
        status = movie::Status_ErrorBadValue;
    }
    else
    {
        *state = m_State;
    }
    return status;
}

movie::Status MoviePlayer::GetTrackInfo(int trackNumber, TrackInfo* trackInfo)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( trackInfo == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    movieStatus = ValidateTrackNumber(trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        movie::MediaData trackFormat;
        if( m_Extractor != NULL )
        {
            movieStatus = m_Extractor->GetTrackConfiguration(trackNumber, &trackFormat);
            if( movieStatus == movie::Status_Success )
            {
                const char *mime;
                if( trackFormat.FindString("mime", &mime) )
                {
                    int32_t avgBitRate = 0;
                    trackFormat.FindInt32("bit-rate", &avgBitRate);
                    if( !strncasecmp("video/", mime, 6) )
                    {
                        int32_t width = 0;
                        int32_t height = 0;
                        int32_t videoFrameRate = 0;
                        trackInfo->trackInfoVideo.decoderType = movie::DecoderType_Unknown;
                        trackInfo->trackInfoVideo.mime = std::string(mime);
                        trackFormat.FindInt32("frame-rate", &videoFrameRate);
                        trackFormat.FindInt32("width", &width);
                        trackFormat.FindInt32("height", &height);
                        trackInfo->trackInfoVideo.width = width;
                        trackInfo->trackInfoVideo.height = height;
                        trackInfo->trackInfoVideo.frameRate = videoFrameRate;
                        trackInfo->trackInfoVideo.bitRate = avgBitRate;
                        if( std::strstr(mime, "avc") )
                        {
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoAvc;
                        }
                        else if( std::strstr(mime, "x-vnd.on2.vp8") )
                        {
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoVp8;
                        }
                        else if( std::strstr(mime, "x-vnd.on2.vp9") )
                        {
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoVp9;
                        }
                    }
                    else if( !strncasecmp("audio/", mime, 6) )
                    {
                        trackInfo->trackInfoAudio.decoderType = movie::DecoderType_Unknown;
                        trackInfo->trackInfoAudio.mime = std::string(mime);
                        int32_t numChannels = 0;
                        int32_t sampleRate = 0;
                        trackFormat.FindInt32("channel-count", &numChannels);
                        trackFormat.FindInt32("sample-rate", &sampleRate);
                        trackInfo->trackInfoAudio.bitRate = avgBitRate;
                        trackInfo->trackInfoAudio.sampleRate = sampleRate;
                        trackInfo->trackInfoAudio.channels = numChannels;
                        if( std::strstr(mime, "mp4a") )
                        {
                            trackInfo->trackInfoAudio.decoderType = movie::DecoderType_AudioAac;
                        }
                        else if( std::strstr(mime, "vorbis") )
                        {
                            trackInfo->trackInfoAudio.decoderType = movie::DecoderType_AudioVorbis;
                        }
                    }
                }
            }
        }
        else
        {
            movieStatus = movie::Status_NullExtractor;
        }
    }
    return movieStatus;
}

movie::Status MoviePlayer::Seek(int64_t seekTimeUs, bool syncMode )
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);
    if( true == IsInRequestedState(State::State_Seeking) )
    {
        return movie::Status_Success;
    }
    movieStatus = ValidateStateTransition(State::State_Seeking);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    bool flush = false;
    State state;
    GetState(&state);
    if( state > MoviePlayer::State::State_Prepared )
    {
        flush = true;
    }
    SetState(State::State_Seeking);
    if( flush == true )
    {
        if( m_MovieDecoderEventHandler != nullptr )
        {
            if( m_AudioDecoder != nullptr )
            {
                m_MovieDecoderEventHandler->SetAudioDecoderFlushState(true);
            }
            if( m_VideoDecoder != nullptr )
            {
                m_MovieDecoderEventHandler->SetVideoDecoderFlushState(true);
            }
        }
        FlushDecoders();
        FlushInputAndOutputHandlers();
    }
    movieStatus = m_Extractor->SeekTo(seekTimeUs);
    SetState(State::State_SeekCompleted);
    if( flush == true )
    {
        FlushInputAndOutputHandlers();
        if( m_MovieDecoderEventHandler != nullptr )
        {
            if( m_AudioDecoder != nullptr )
            {
                m_MovieDecoderEventHandler->SetAudioDecoderFlushState(false);
            }
            if( m_VideoDecoder != nullptr )
            {
                m_MovieDecoderEventHandler->SetVideoDecoderFlushState(false);
            }
        }
        if( m_VideoDecoder != nullptr )
        {
            m_VideoDecoder->Start();
        }
        if( m_AudioDecoder != nullptr )
        {
            m_AudioDecoder->Start();
        }
    }
    return movieStatus;
}

movie::Status MoviePlayer::GetLastSeekPosition(int64_t *lastSeekTimeUs) const
{
    movie::Status status = movie::Status_Success;
    if( lastSeekTimeUs == nullptr )
    {
        status = movie::Status_ErrorBadValue;
    }
    else
    {
        *lastSeekTimeUs = m_LastSeekPositionUs;
    }
    return status;
}

movie::Status MoviePlayer::GetLastError()
{
    return m_LastError;
}

const char* MoviePlayer::StateToString(State state)
{
    switch( state )
    {
    case State::State_NotInitialized:
        return "State_NotInitialized";
    case State::State_Initialized:
        return "State_Initialized";
    case State::State_Preparing:
        return "State_Preparing";
    case State::State_Prepared:
        return "State_Prepared";
    case State::State_Started:
        return "State_Started";
    case State::State_Stopped:
        return "State_Stopped";
    case State::State_Paused:
        return "State_Paused";
    case State::State_Seeking:
        return "State_Seeking";
    case State::State_SeekCompleted:
        return "State_SeekCompleted";
    case State::State_PlaybackCompleted:
        return "State_PlaybackCompleted";
    case State::State_Error:
        return "State_Error";
    default:
        return "Unknown_State";
    }
}

void MoviePlayer::UpdateLastErrorAndSignalClient(movie::Status lastError)
{
    SetLastError(lastError);
    if( m_ErrorEvent != nullptr )
    {
        nn::os::SignalEvent(m_ErrorEvent);
    }
}

void MoviePlayer::OnPrepare()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_ApiLock);

    if( ( m_VideoTrackNumber < 0 ) && ( m_AudioTrackNumber < 0 ) )
    {
        UpdateLastErrorAndSignalClient(movie::Status_UnsupportedMediaType);
        return;
    }
    if( m_VideoTrackNumber >= 0 )
    {
        if( m_VideoDecoderType == movie::DecoderType_Unknown )
        {
            UnSelectTrack(m_VideoTrackNumber);
            NN_LOG("\n Unsupported VideoTrack ! \n");
        }
        else
        {
            if( m_VideoDecoderMode == movie::DecoderMode_NativeTexture )
            {
                m_VideoDecoderOutputFormat = movie::DecoderOutputFormat_VideoColorAbgr;
            }
            m_VideoDecoder = new movie::Decoder(m_VideoDecoderType, m_VideoDecoderOutputFormat, m_VideoDecoderMode);
            if( m_VideoDecoder == NULL )
            {
                UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
                return;
            }
            movieStatus = m_VideoDecoder->RegisterClientEvents(&m_VideoDecoderEvents);
            if( movieStatus == movie::Status_Success )
            {
                if( m_Extractor != nullptr )
                {
                    movie::MediaData videoTrackFormat;
                    m_Extractor->GetTrackConfiguration(m_VideoTrackNumber, &videoTrackFormat);
                    movieStatus = m_VideoDecoder->Configure(&videoTrackFormat);
                    videoTrackFormat.Clear();
                    m_MovieVideoInputHandler = new MovieVideoInputHandler(m_Extractor, m_VideoTrackNumber, m_VideoDecoder);
                    if( m_MovieVideoOutputHandler == nullptr )
                    {
                        m_MovieVideoOutputHandler = new MovieVideoOutputHandler(m_NativeWindow, m_VideoDecoderMode, m_VideoDecoderOutputFormat);
                    }
                    if( ( m_MovieVideoInputHandler == nullptr ) || ( m_MovieVideoOutputHandler == nullptr ) )
                    {
                        UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
                        return;
                    }
                    m_MovieVideoInputHandler->Initialize(m_CoreMask);
                    m_VideoPlaybackComplete = false;
                }
                else
                {
                    UpdateLastErrorAndSignalClient(movieStatus);
                    return;
                }
            }
            else
            {
                UpdateLastErrorAndSignalClient(movieStatus);
                return;
            }
        }
    }
    if( m_AudioTrackNumber >= 0 )
    {
        if( m_AudioDecoderType == movie::DecoderType_Unknown )
        {
            UnSelectTrack(m_AudioTrackNumber);
            NN_LOG("\n Unsupported AudioTrack ! \n");
        }
        else
        {
            m_AudioDecoder = new movie::Decoder(m_AudioDecoderType, m_AudioDecoderOutputFormat, m_AudioDecoderMode);
            if( m_AudioDecoder == NULL )
            {
                UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
                return;
            }
            movieStatus = m_AudioDecoder->RegisterClientEvents(&m_AudioDecoderEvents);
            if( movieStatus == movie::Status_Success )
            {
                movie::MediaData audioTrackFormat;
                m_Extractor->GetTrackConfiguration(m_AudioTrackNumber, &audioTrackFormat);
                movieStatus = m_AudioDecoder->Configure(&audioTrackFormat);
                audioTrackFormat.Clear();
                m_MovieAudioInputHandler = new MovieAudioInputHandler(m_Extractor, m_AudioTrackNumber, m_AudioDecoder);
                if( m_MovieAudioOutputHandler == nullptr )
                {
                    m_MovieAudioOutputHandler = new MovieAudioOutputHandler();
                }
                if( ( m_MovieAudioInputHandler == nullptr ) || ( m_MovieAudioOutputHandler == nullptr ) )
                {
                    UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
                    return;
                }
                m_MovieAudioInputHandler->Initialize(m_CoreMask);
                m_AudioPlaybackComplete = false;
            }
            else
            {
                UpdateLastErrorAndSignalClient(movieStatus);
                return;
            }
        }
    }
    m_MovieDecoderEventHandler = new MovieDecoderEventHandler(m_AudioDecoder,
        m_VideoDecoder,
        m_MovieVideoInputHandler,
        m_MovieVideoOutputHandler,
        m_MovieAudioInputHandler,
        m_MovieAudioOutputHandler);
    if( m_MovieDecoderEventHandler == nullptr )
    {
        UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
        return;
    }
    m_MovieDecoderEventHandler->Initialize(m_CoreMask);
    m_MovieDecoderEventHandler->RegisterEvent(&m_AudioInputEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_AudioInputBuffer);
    m_MovieDecoderEventHandler->RegisterEvent(&m_AudioOutputEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_AudioOutputBuffer);
    m_MovieDecoderEventHandler->RegisterEvent(&m_AudioFormatChangedEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_AudioFormatChanged);
    m_MovieDecoderEventHandler->RegisterEvent(&m_AudioErrorEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_AudioError);
    m_MovieDecoderEventHandler->RegisterEvent(&m_VideoInputEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_VideoInputBuffer);
    m_MovieDecoderEventHandler->RegisterEvent(&m_VideoOutputEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_VideoOutputBuffer);
    m_MovieDecoderEventHandler->RegisterEvent(&m_VideoFormatChangedEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_VideoFormatChanged);
    m_MovieDecoderEventHandler->RegisterEvent(&m_VideoErrorEvent, MovieDecoderEventHandler::DecoderEventType::DecoderEventType_VideoError);

    m_MediaClock = new MediaClock();
    if( m_MediaClock == nullptr )
    {
        UpdateLastErrorAndSignalClient(movie::Status_OutOfMemory);
        return;
    }
    movieStatus = ValidateStateTransition(State::State_Prepared);
    if( movieStatus == movie::Status_Success )
    {
        SetState(State::State_Prepared);
    }
    else
    {
        UpdateLastErrorAndSignalClient(movieStatus);
    }
}//NOLINT(impl/function_size)

movie::Status MoviePlayer::GetContainerType(const char* mediaPath, movie::ContainerType *containertype)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( ( mediaPath == nullptr ) || ( containertype == nullptr) )
    {
        movieStatus = movie::Status_ErrorBadValue;
    }
    else
    {
        movieStatus = movie::Status_Success;
        movie::ContainerType container = movie::ContainerType_Unknown;
        std::string path(mediaPath);
        std::size_t dotLocation = path.rfind(".");
        std::string mediaExtension = path.substr(dotLocation, path.size() - dotLocation);

        for( int i = 0; mediaExtension[ i ]; i++ )
        {
            mediaExtension[ i ] = tolower(mediaExtension[ i ]);
        }
        if( mediaExtension.compare(std::string(".mp4")) == 0 )
        {
            container = movie::ContainerType_Mpeg4;
        }
        else if( mediaExtension.compare(std::string(".m4a")) == 0 )
        {
            container = movie::ContainerType_Mpeg4;
        }
        else if( mediaExtension.compare(std::string(".mkv")) == 0 )
        {
            container = movie::ContainerType_Matroska;
        }
        else if( mediaExtension.compare(std::string(".webm")) == 0 )
        {
            container = movie::ContainerType_WebM;
        }
        else
        {
            container = movie::ContainerType_Unknown;
        }
        *containertype = container;
    }
    return movieStatus;
}

movie::Status MoviePlayer::ValidateStateTransition(State newState)
{
    if( newState == State_NotInitialized )
    {
        return movie::Status_Success;
    }
    movie::Status movieStatus = movie::Status_InvalidStateTransition;
    State currentState = State_NotInitialized;
    movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    movieStatus = movie::Status_InvalidStateTransition;
    switch( currentState )
    {
        case State_NotInitialized:
        {
            if( newState == State_Initialized )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Initialized:
        {
            if( newState == State_Preparing )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Preparing:
        {
            if( ( newState == State_Prepared ) || ( newState == State_Initialized ) )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Prepared:
        {
            if( ( newState == State_Started ) || ( newState == State_Seeking ) )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Started:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Paused ) )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Stopped:
        {
            if( newState == State_Preparing )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Paused:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Started ) ||
                ( newState == State_Seeking ) || ( newState == State_Paused ) )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_Seeking:
        {
            if( newState == State_SeekCompleted )
            {
                movieStatus = movie::Status_Success;
            }

            break;
        }
        case State_SeekCompleted:
        {
            if( ( newState == State_Started ) || ( newState == State_Paused ) ||
                ( newState == State_Seeking ) || ( newState == State_Prepared ) ||
                ( newState == State_Stopped ) || ( newState == State_PlaybackCompleted ) )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        case State_PlaybackCompleted:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Seeking ) ||
                ( newState == State_Started ) )
            {
                movieStatus = movie::Status_Success;
            }

            break;
        }
        case State_Error:
        {
            if( newState == State_NotInitialized )
            {
                movieStatus = movie::Status_Success;
            }
            break;
        }
        default:
            break;
    }
    return movieStatus;
}//NOLINT(impl/function_size)

movie::Status MoviePlayer::SetState(State state)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    Mutex::ScopedLock lock(m_StateLock);
    if( ( state >= State::State_NotInitialized ) && ( state <= State::State_Error ) )
    {
        m_State = state;
        if( m_StateChangedEvent != nullptr )
        {
            nn::os::SignalEvent(m_StateChangedEvent);
        }
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status MoviePlayer::RegisterPlayerEvent(nn::os::EventType *event, PlayerEventType eventType)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::MultiWaitHolderType *eventHolder = CreateEventHolder(event, eventType);
    nn::os::LinkMultiWaitHolder(&m_PlayerMultiWait, eventHolder);
    return movieStatus;
}

nn::os::MultiWaitHolderType* MoviePlayer::CreateEventHolder(nn::os::EventType *event, PlayerEventType eventType)
{
    switch( eventType )
    {
        case PlayerEventType_Prepare:
        {
            EventUserData *userData = new EventUserData(event, PlayerEventType_Prepare);
            nn::os::InitializeMultiWaitHolder(&m_PrepareEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_PrepareEventHolder, ( uintptr_t ) userData);
            return &m_PrepareEventHolder;
        }
        case PlayerEventType_Seek:
        {
            EventUserData *userData = new EventUserData(event, PlayerEventType_Seek);
            nn::os::InitializeMultiWaitHolder(&m_SeekEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_SeekEventHolder, ( uintptr_t ) userData);
            return &m_SeekEventHolder;
        }
        case PlayerEventType_AudioRenderComplete:
        {
            EventUserData *userData = new EventUserData(event, PlayerEventType_AudioRenderComplete);
            nn::os::InitializeMultiWaitHolder(&m_AudioRenderCompleteEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_AudioRenderCompleteEventHolder, ( uintptr_t ) userData);
            return &m_AudioRenderCompleteEventHolder;
        }
        case PlayerEventType_VideoRenderComplete:
        {
            EventUserData *userData = new EventUserData(event, PlayerEventType_VideoRenderComplete);
            nn::os::InitializeMultiWaitHolder(&m_VideoRenderCompleteEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_VideoRenderCompleteEventHolder, ( uintptr_t ) userData);
            return &m_VideoRenderCompleteEventHolder;
        }
        case PlayerEventType_ThreadExit:
        {
            EventUserData *userData = new EventUserData(event, PlayerEventType_ThreadExit);
            nn::os::InitializeMultiWaitHolder(&m_ThreadExitEventHolder, event);
            nn::os::SetMultiWaitHolderUserData(&m_ThreadExitEventHolder, ( uintptr_t ) userData);
            return &m_ThreadExitEventHolder;
        }
        default:
            break;
    }
    return nullptr;
}

void MoviePlayer::DeleteUserData(EventUserData* userData)
{
    if( userData != nullptr )
    {
        delete userData;
    }
}

void MoviePlayerThreadFunction(void *arg)
{
    MoviePlayer *moviePlayer = ( MoviePlayer* ) arg;
    bool exitThread = false;

    // Wait for events from movie::decoder and process them.
    for( ;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&moviePlayer->m_PlayerMultiWait);
        uintptr_t userData = GetMultiWaitHolderUserData(holder);
        MoviePlayer::EventUserData *eventUserData = ( MoviePlayer::EventUserData* ) userData;
        MoviePlayer::PlayerEventType eventType = eventUserData->m_EventType;
        nn::os::EventType *event = eventUserData->m_Event;
        nn::os::ClearEvent(event);
        switch( eventType )
        {
            case MoviePlayer::PlayerEventType_Prepare:
                moviePlayer->OnPrepare();
            break;
            case MoviePlayer::PlayerEventType_Seek:
            break;
            case MoviePlayer::PlayerEventType_AudioRenderComplete:
                moviePlayer->SetAudioPlaybackComplete();
                if( moviePlayer->IsMediaPlaybackComplete() == true )
                {
                    moviePlayer->SetState(MoviePlayer::State_PlaybackCompleted);
                }
                break;
            case MoviePlayer::PlayerEventType_VideoRenderComplete:
                moviePlayer->SetVideoPlaybackComplete();
                if( moviePlayer->IsMediaPlaybackComplete() == true )
                {
                    moviePlayer->SetState(MoviePlayer::State_PlaybackCompleted);
                }
                break;
            case MoviePlayer::PlayerEventType_ThreadExit:
                exitThread = true;
                break;
            default:
                break;
        }
        if( exitThread == true )
        {
            break;
        }
    }
}

void MoviePlayer::SetAudioPlaybackComplete()
{
    m_AudioPlaybackComplete = true;;
}

void MoviePlayer::SetVideoPlaybackComplete()
{
    m_VideoPlaybackComplete = true;
}

bool MoviePlayer::IsMediaPlaybackComplete()
{
    if( ( m_AudioPlaybackComplete == true ) && ( m_VideoPlaybackComplete == true ) )
    {
        return true;
    }
    else
    {
        return false;
    }
}

movie::Status MoviePlayer::ValidateTrackNumber(int trackNumber)
{
    movie::Status movieStatus = movie::Status_TrackNotFound;
    int trackCount = 0;
    movieStatus = GetTrackCount(&trackCount);
    if( movieStatus == movie::Status_Success )
    {
        if( ( trackNumber >= trackCount ) || ( trackNumber < 0 ) )
        {
            movieStatus = movie::Status_TrackNotFound;
        }
    }
    return movieStatus;
}

void MoviePlayer::SetLastError(movie::Status movieStatus)
{
    m_LastError = movieStatus;
}

movie::Status MoviePlayer::CreateExtractorForPlayer(const char* mediaPath)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    movie::ContainerType containertype = movie::ContainerType_Unknown;
    movieStatus = GetContainerType(mediaPath, &containertype);
    if( movieStatus != movie::Status_Success )
    {
        return movieStatus;
    }
    if( containertype == movie::ContainerType_Unknown )
    {
        return movie::Status_UnsupportedMediaType;
    }
    m_Extractor = new movie::Extractor(containertype);
    if( m_Extractor == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    return movieStatus;
}

movie::Status MoviePlayer::UpdateTrackPropertiesForPlayer()
{
    movie::Status movieStatus = movie::Status_TrackNotFound;
    if( m_Extractor == nullptr )
    {
        return movieStatus;
    }
    movieStatus = m_Extractor->GetTrackCount(&m_NumberOfTracks);
    if( movieStatus == movie::Status_Success )
    {
        if( m_NumberOfTracks <= 0 )
        {
            movieStatus = movie::Status_TrackNotFound;
        }
    }
    movie::MediaData trackFormat;
    for( int i = 0; i < m_NumberOfTracks; i++ )
    {
        m_Extractor->GetTrackConfiguration(i, &trackFormat);
        trackFormat.FindInt64("durationUs", &m_MediaDurationUs);
        const char *mime;
        if( trackFormat.FindString("mime", &mime) )
        {
            if( !strncasecmp("video/", mime, 6) )
            {
                if( !strncasecmp("video/avc", mime, 9) )
                {
                    m_VideoDecoderType = movie::DecoderType_VideoAvc;
                }
                else if( !strncasecmp("video/x-vnd.on2.vp8", mime, 19) )
                {
                    m_VideoDecoderType = movie::DecoderType_VideoVp8;
                }
                else if( !strncasecmp("video/x-vnd.on2.vp9", mime, 19) )
                {
                    m_VideoDecoderType = movie::DecoderType_VideoVp9;
                }
            }
            else if( !strncasecmp("audio/", mime, 6) )
            {
                if( !strncasecmp("audio/mp4a-latm", mime, 15) )
                {
                    m_AudioDecoderType = movie::DecoderType_AudioAac;
                }
                else if( !strncasecmp("audio/vorbis", mime, 12) )
                {
                    m_AudioDecoderType = movie::DecoderType_AudioVorbis;
                }
                trackFormat.FindInt32("sample-rate", &m_AudioSampleRate);
                trackFormat.FindInt32("channel-count", &m_NumberOfAudioChannels);
                if( m_AudioSampleRate > 48000 )
                {
                    if( !strncasecmp("audio/mp4a-latm", mime, 15) )
                    {
                        m_AudioDecoderType = movie::DecoderType_Unknown;
                        m_AudioTrackNumber = -1;
                    }
                }
                if(( m_NumberOfAudioChannels > 6 ) || ( m_AudioSampleRate < 16000 ))
                {
                    m_AudioDecoderType = movie::DecoderType_Unknown;
                    m_AudioTrackNumber = -1;
                }

            }
        }
    }
    trackFormat.Clear();
    if( ( m_VideoDecoderType == movie::DecoderType_Unknown ) && ( m_AudioDecoderType == movie::DecoderType_Unknown) )
    {
        movieStatus = movie::Status_UnsupportedMediaType;
    }
    return movieStatus;
}

movie::Status MoviePlayer::HandlePlaybackCompleteForPlayer()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Stop();
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Stop();
    }
    if( m_VideoDecoder != nullptr )
    {
        m_VideoDecoder->Flush();
    }
    if( m_AudioDecoder != nullptr )
    {
        m_AudioDecoder->Flush();
    }
    if( m_Extractor != nullptr )
    {
        m_Extractor->SeekTo(0);
    }
    m_MediaClock->ClearAnchorTime();
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Start(m_Width, m_Height);
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Start();
    }
    if( m_VideoDecoder != nullptr )
    {
        m_VideoDecoder->Start();
    }
    if( m_AudioDecoder != nullptr )
    {
        m_AudioDecoder->Start();
    }
    return movieStatus;
}

bool MoviePlayer::IsInRequestedState(State state)
{
    State currentState;
    movie::Status movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
    {
        return false;
    }
    if( currentState == state )
    {
        return true;
    }
    else
    {
        return false;
    }
}

void MoviePlayer::DeletePlayerOwnedObjects()
{
    if( m_MovieAudioInputHandler != nullptr )
    {
        delete m_MovieAudioInputHandler;
        m_MovieAudioInputHandler = nullptr;
    }
    if( m_MovieVideoInputHandler != nullptr )
    {
        delete m_MovieVideoInputHandler;
        m_MovieVideoInputHandler = nullptr;
    }
    if( m_Extractor != nullptr )
    {
        delete m_Extractor;
        m_Extractor = nullptr;
    }
    if( m_VideoDecoder != nullptr )
    {
        delete m_VideoDecoder;
        m_VideoDecoder = nullptr;
    }
    if( m_AudioDecoder != nullptr )
    {
        delete m_AudioDecoder;
        m_AudioDecoder = nullptr;
    }
    if( m_MediaClock != nullptr )
    {
        delete m_MediaClock;
        m_MediaClock = nullptr;
    }
}

void MoviePlayer::FlushInputAndOutputHandlers()
{
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Flush();
    }
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Flush();
    }
    if( m_MovieAudioInputHandler != nullptr )
    {
        m_MovieAudioInputHandler->SignalFlush();
    }
    if( m_MovieVideoInputHandler != nullptr )
    {
        m_MovieVideoInputHandler->SignalFlush();
    }
}

void MoviePlayer::FlushDecoders()
{
    if( m_VideoDecoder != nullptr )
    {
        m_VideoDecoder->Flush();
    }
    if( m_AudioDecoder != nullptr )
    {
        m_AudioDecoder->Flush();
    }
}

movie::Status MoviePlayer::SetVideoDecoderMode(movie::DecoderMode videoDecoderMode)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( m_VideoDecoderMode == videoDecoderMode )
    {
        return movie::Status_Success;
    }
    else
    {
        State currentState;
        movieStatus = GetState(&currentState);
        if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
        if( currentState < State::State_Preparing )
        {
            m_VideoDecoderMode = videoDecoderMode;
        }
        else
        {
            movieStatus = movie::Status_OperationDenied;
        }
    }
    return movieStatus;
}

static bool IsPreferredLanguageTrack(const char* lang)
{
    nn::settings::LanguageCode userLanguage;
    userLanguage = nn::oe::GetDesiredLanguage();

    if(userLanguage == nn::settings::Language_Japanese)
    {
       if( !strncasecmp("jpn", lang, 3) )
           return true;
    }
    else if (( userLanguage == nn::settings::Language_AmericanEnglish )
                || ( userLanguage == nn::settings::Language_BritishEnglish ))
    {
        if( !strncasecmp("eng", lang, 3) )
            return true;
    }
    else if (( userLanguage == nn::settings::Language_French )
                || ( userLanguage == nn::settings::Language_CanadianFrench ))
    {
        if( (!strncasecmp("fra", lang, 3)) || (!strncasecmp("fre", lang, 3)) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_German)
    {
        if( (!strncasecmp("ger", lang, 3)) || (!strncasecmp("deu", lang, 3)) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_Italian)
    {
        if( !strncasecmp("ita", lang, 3) )
            return true;
    }
    else if (( userLanguage == nn::settings::Language_Spanish )
                || ( userLanguage == nn::settings::Language_LatinAmericanSpanish ))
    {
        if( !strncasecmp("spa", lang, 3) )
            return true;
    }
    else if (( userLanguage == nn::settings::Language_Chinese )
                || ( userLanguage == nn::settings::Language_Taiwanese )
                || ( userLanguage == nn::settings::Language_SimplifiedChinese )
                || ( userLanguage == nn::settings::Language_TraditionalChinese ) )
    {
        if( (!strncasecmp("chi", lang, 3)) || (!strncasecmp("zho", lang, 3)) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_Korean)
    {
        if( !strncasecmp("kor", lang, 3) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_Dutch)
    {
        if( (!strncasecmp("nld", lang, 3)) || (!strncasecmp("dut", lang, 3)) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_Portuguese)
    {
        if( !strncasecmp("por", lang, 3) )
            return true;
    }
    else if(userLanguage == nn::settings::Language_Russian)
    {
        if( !strncasecmp("rus", lang, 3) )
            return true;
    }
    return false;
}
