﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nns/mm.h>
#include "MovieVideoInputHandler.h"
#include "MovieAudioInputHandler.h"

/**
 * @brief
 * MovieDecoderEventHandler class for handling audio and video decoder events.
 *
 */
void MovieDecoderEventHandlerThreadFunction(void *arg);
class MovieDecoderEventHandler
{
public:
   /**
    * @brief
    * Enumeration of decoder event types.
    */
    enum DecoderEventType
    {
        DecoderEventType_Unknown            = 0,        //!< Unknown event.
        DecoderEventType_VideoInputBuffer   = 1,        //!< Video decoder input buffer available event.
        DecoderEventType_VideoOutputBuffer  = 2,        //!< Video decoder output buffer available event.
        DecoderEventType_VideoFormatChanged = 3,        //!< Video decoder format changed event.
        DecoderEventType_VideoError         = 4,        //!< Video decoder error event.
        DecoderEventType_AudioInputBuffer   = 5,        //!< Audio decoder input buffer available event.
        DecoderEventType_AudioOutputBuffer  = 6,        //!< Audio decoder output buffer available event.
        DecoderEventType_AudioFormatChanged = 7,        //!< Audio decoder format changed event.
        DecoderEventType_AudioError         = 8,        //!< Video decoder error event.
        DecoderEventType_ThreadExit         = 9,        //!< MovieDecoderEventHandlerthread exit event.
    };

   /**
    * @brief
    * MovieDecoderEventHandler constructor.
    *
    * @param[in] audioDecoder                   Movie audio decoder
    * @param[in] videoDecoder                   Movie video decoder
    * @param[in] videoInputHandler              Video input handler instance
    * @param[in] videoOutputHandler             Video output handler instance
    * @param[in] audioInputHandler              Audio input handler instance
    * @param[in] audioOutputHandler             Audio output handler instance
    *
    * @details
    * @class MovieDecoderEventHandler instance will be created. Audio/video decoder and corresponding input/output
    * handlers need to be passed through constructional parameters.
    *
    */
    MovieDecoderEventHandler(movie::Decoder* audioDecoder,
        movie::Decoder* videoDecoder,
        MovieVideoInputHandler *videoInputHandler,
        MovieVideoOutputHandler *videoOutputHandler,
        MovieAudioInputHandler *audioInputHandler,
        MovieAudioOutputHandler *audioOutputHandler);

   /**
    * @brief
    * Initialize @class MovieDecoderEventHandler instance.
    *
    * @param[in]  coreMask                      Usable CPU cores in core mask.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_OutOfMemory
    * @retval ::Status_FailedToCreateThread
    *
    * @details
    * Usable cores in core mask. If Nth bit is set to 1, the thread can run on CPU core number N. The core mask need
    * to be set before creation of thread. @class MovieDecoderEventHandler thread is created. The thread will wait
    * on nn::os::MultiWaitType. Any client events need to be linked to this multi wait.
    */
    movie::Status Initialize(uint64_t coreMask);

   /**
    * @brief
    * RegisterEvent. API to register events with @class MovieDecoderEventHandler multi wait.
    *
    * @param[in]  event                         nn::os::EventType to register.
    * @param[in]  eventType                     Type of event, should be one of enum from DecoderEventType.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register all events from decoders. Movie decoders uses events extensively.
    * @class MovieDecoderEventHandler can receive all events from decoder and do appropriate actions
    * based on type of event. Any event from DecoderEventType can be registered.
    */
    movie::Status RegisterEvent(nn::os::EventType *event, DecoderEventType eventType);

   /**
    * @brief
    * CreateEventHolder. API to create an event holder for a given an event.
    *
    * @param[in]  event                         nn::os::EventType to register.
    * @param[in]  eventType                     Type of event, should be one of enum from DecoderEventType.
    *
    * @return nn::os::MultiWaitHolderType
    *
    * @details
    * This API can be used to create an event holder to link an event to a multi wait.
    *
    */
    nn::os::MultiWaitHolderType* CreateEventHolder(nn::os::EventType *event, DecoderEventType eventType);

   /**
    * @brief
    * VideoInputBufferAvailableEvent. API to handle video decoder input buffer events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receives DecoderEventType_VideoInputBuffer event from the decoder, this API
    * is called to handle the event. The handler will acquire all available input buffer indices and add them into @class
    * MovieVideoInputHandler input list.
    *
    */
    movie::Status VideoInputBufferAvailableEvent();

   /**
    * @brief
    * VideoOutputBufferAvailableEvent. API to handle video decoder output buffer events.
    *
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoOutputBuffer event from the decoder, this API
    * is called to handle the event. The handler will acquire all available output buffer indices and add them into @class
    * MovieVideoOutputHandler list.
    *
    */
    movie::Status VideoOutputBufferAvailableEvent();

   /**
    * @brief
    * VideoFormatChangedEvent. API to handle video decoder format changed events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_VideoFormatChanged event from the decoder, this API
    * is called to handle the event. The handler will call CheckAndSignalFormatChangeWaiter() API. If there are any
    * waiters they will be signaled.
    *
    */
    movie::Status VideoFormatChangedEvent();

   /**
    * @brief
    * VideoErrorEvent. API to handle error events from video decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receives DecoderEventType_VideoError event from the decoder, VideoErrorEvent() API
    * is called to handle the event.
    *
    */
    movie::Status VideoErrorEvent();

   /**
    * @brief
    * AudioInputBufferAvailableEvent. API to handle audio decoder input buffer events.
    *
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receives DecoderEventType_AudioInputBuffer event from the decoder, this API
    * is called to handle the event. The handler will acquire all available input buffer indices and add them into
    * @class MovieAudioInputHandler list.
    *
    */
    movie::Status AudioInputBufferAvailableEvent();

   /**
    * @brief
    * AudioOutputBufferAvailableEvent. API to handle audio decoder output buffer events.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * When @class MovieDecoderEventHandler receives DecoderEventType_AudioOutputBuffer event from the decoder, this API
    * is called to handle the event. The handler will acquire all available output buffer indices and add them into
    * @class MovieAudioOutputHandler list.
    *
    */
    movie::Status AudioOutputBufferAvailableEvent();

   /**
    * @brief
    * AudioFormatChangedEvent. API to handle audio decoder format changed events.
    *
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receives DecoderEventType_AudioFormatChanged event from the decoder, this API
    * is called to handle the event. The handler will call CheckAndSignalFormatChangeWaiter() API. If there are any
    * waiters they will be signaled.
    *
    */
    movie::Status AudioFormatChangedEvent();

   /**
    * @brief
    * WaitForFormatChange. API to wait for format changed events from audio and video decoder
    *
    * @return none
    *
    * @details
    * This API can be used to wait for format changed events from audio and video decoders. This API will return when
    * format changed events received from both audio and video decoders or 5 seconds wait time is expired.
    *
    */
    void WaitForFormatChange();

   /**
    * @brief
    * CheckAndSignalFormatChangeWaiter. API to check for format changed event and signal any waiter.
    *
    * @return none
    *
    * @details
    * This API can be used to check format changed event and signal if any waiters available.
    *
    */
    void CheckAndSignalFormatChangeWaiter();

   /**
    * @brief
    * AudioErrorEvent. API to handle error events from audio decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When @class MovieDecoderEventHandler receive DecoderEventType_AudioError event, AudioErrorEvent() API is called.
    *
    */
    movie::Status AudioErrorEvent();

   /**
    * @brief
    * MovieDecoderEventHandler destructor.
    *
    * @return None
    *
    * @details
    * All the resources allocated by @class MovieDecoderEventHandler are released including the thread.
    *
    */
    ~MovieDecoderEventHandler();

   /**
    * @brief
    * SetAudioDecoderFlushState, API to set audio decoder flush state.
    *
    * @return None
    *
    * @details
    * This API can be used set audio decoder flush state.
    *
    */
    void SetAudioDecoderFlushState(bool isflushing) { m_AudioDecoderInFlush = isflushing; }

   /**
    * @brief
    * SetVideoDecoderFlushState, API to set video decoder flush state.
    *
    * @return None
    *
    * @details
    * This API can be used set video decoder flush state.
    *
    */
    void SetVideoDecoderFlushState(bool isflushing) { m_VideoDecoderInFlush = isflushing; }

   /**
    * @brief
    * GetAudioDecoderFlushState, API to get audio decoder flush state.
    *
    * @return None
    *
    * @details
    * This API can be used get audio decoder flush state.
    *
    */
    bool GetAudioDecoderFlushState() { return m_AudioDecoderInFlush.load(); }

   /**
    * @brief
    * GetVideoDecoderFlushState, API to get video decoder flush state.
    *
    * @return None
    *
    * @details
    * This API can be used get video decoder flush state.
    *
    */
    bool GetVideoDecoderFlushState() { return m_VideoDecoderInFlush.load(); }

    nn::os::MultiWaitType m_DecoderMultiWait;

    struct EventUserData
    {
        EventUserData(nn::os::EventType *event, DecoderEventType eventType)
            : m_Event(event),
              m_EventType(eventType)
        { }
        nn::os::EventType *m_Event;
        DecoderEventType m_EventType;
    };

private:

    void DeleteUserData(EventUserData* userData);

    movie::Decoder* m_VideoDecoder;
    movie::Decoder* m_AudioDecoder;

    MovieVideoInputHandler *m_MovieVideoInputHandler;
    MovieVideoOutputHandler *m_MovieVideoOutputHandler;
    MovieAudioInputHandler *m_MovieAudioInputHandler;
    MovieAudioOutputHandler *m_MovieAudioOutputHandler;

    size_t m_ThreadStackSize;

    void* m_ThreadStack;
    nn::os::ThreadType m_ThreadType;
    nn::os::EventType m_ThreadExitEvent;
    nn::os::MultiWaitHolderType m_ThreadExitEventHolder;

    nn::os::MultiWaitHolderType m_AudioInputEventHolder;
    nn::os::MultiWaitHolderType m_AudioOutputEventHolder;
    nn::os::MultiWaitHolderType m_AudioFormatChangedEventHolder;
    nn::os::MultiWaitHolderType m_AudioErrorEventHolder;

    nn::os::MultiWaitHolderType m_VideoInputEventHolder;
    nn::os::MultiWaitHolderType m_VideoOutputEventHolder;
    nn::os::MultiWaitHolderType m_VideoFormatChangedEventHolder;
    nn::os::MultiWaitHolderType m_VideoErrorEventHolder;

    nn::os::EventType m_FormatChangedPlayerEvent;
    bool m_AudioFormatChangeEventReceived;
    bool m_VideoFormatChangeEventReceived;
    std::atomic<bool> m_AudioDecoderInFlush;
    std::atomic<bool> m_VideoDecoderInFlush;
};
