﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * MovieAudioInputHandler class for handling audio decoder input events and feed input data to the decoder
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MovieAudioInputHandler is to handle audio decoder input events. @class
 * MovieDecoderEventHandler registers events with movie decoder. When any of input related events are signaled from the
 * decoder, @class MovieDecoderEventHandler sends message to @class MovieAudioInputHandler message queue. When the
 * @class MovieAudioInputHandler receives input buffer available message, it will check whether the extractor has data
 * for audio track. If there is data for audio track, it will get an input buffer from decoder, gets input data from
 * extractor. The filled input buffer is sent for decoding. It also handles flush message by clearing all input buffers
 * from input list.
 *
 * Setup:
 *     @class MoviePlayer class will create an instance of @class MovieAudioInputHandler. An extractor and an audio
 * decoder is passed in the constructor. During initialization a message queue and a thread and is created for handling
 * messages. The thread will wait for messages in this message queue. Other resources like input buffer list are
 * initialized.
 *
 * Main Processing:
 *     When @class MovieDecoderEventHandler receives input buffer available event from audio decoder, it sends a
 * message to @class MovieAudioInputHandler message queue. When the message is serviced, extractor is checked for
 * audio data availability. If audio data is available, audio decoder input buffer is requested from decoder. This
 * will be filled with compressed audio data. Filled input buffer is sent to decoder for decoding. This process
 * continues as long as input data is available from extractor. When end of stream is received from extractor, an
 * input buffer with EOS buffer flag is sent to decoder.
 *
 * Teardown:
 *    When @class MovieDecoderEventHandler receives EOS buffer from audio decoder media playback will be stopped. At
 * the end player will delete @class MovieAudioInputHandler instance. This will result in sending thread exit message
 * to MovieAudioInputHandler thread. The thread will exit and resources are freed.
 *
 */
#include "MovieAudioInputHandler.h"

MovieAudioInputHandler::MovieAudioInputHandler(movie::Extractor* extractor,
    int32_t audioTrackIndex, movie::Decoder* audioDecoder )
    : m_Extractor(extractor),
      m_AudioDecoder(audioDecoder),
      m_AudioTrackIndex(audioTrackIndex),
      m_ThreadStackSize(1024 * 128),
      m_ThreadStack(nullptr),
      m_MessageQueueBufferSize(256),
      m_MessageQueueBuffer(nullptr),
      m_FlushInput(false)
{
}

MovieAudioInputHandler::~MovieAudioInputHandler()
{
    nn::os::SendMessageQueue(&m_MessageQueue, MovieAudioInputHandler::MoviePlayerMessage_AudioInputThreadExit);
    nn::os::WaitThread(&m_ThreadType);
    nn::os::DestroyThread(&m_ThreadType);

    // Finalize MessageQueue used by movie decoder player
    nn::os::FinalizeMessageQueue(&m_MessageQueue);
    if( m_MessageQueueBuffer != NULL )
    {
        delete[] m_MessageQueueBuffer;
    }

    // Delete thread stack memory.
    if( m_ThreadStack != NULL )
    {
        free(m_ThreadStack);
    }
}
movie::Status MovieAudioInputHandler::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();

    m_AudioBufferIndexList.reserve(64);
    // Set only valid core mask
    if( coreMask )
    {
        movie::SetCoreMask(coreMask);
    }
    m_MessageQueueBuffer = new uintptr_t[ m_MessageQueueBufferSize ];
    if( m_MessageQueueBuffer == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    // Initialize MovieAudioInputHandler MessageQueue.
    nn::os::InitializeMessageQueue(&m_MessageQueue, m_MessageQueueBuffer, m_MessageQueueBufferSize);
    nn::os::InitializeMutex(&m_AudioListMutex, false, 0);

    // Create Audio InputReader thread.
    m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_ThreadStack == NULL )
    {
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        nnResult = nn::os::CreateThread(&m_ThreadType,
            &MovieAudioInputHandlerThreadFunction,
            ( void* )this,
            m_ThreadStack,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName(&m_ThreadType, "MovieAudioInputReadThread");
            nn::os::StartThread(&m_ThreadType);
        }
        else
        {
            movieStatus = movie::Status_FailedToCreateThread;
        }
    }
    return movieStatus;
}

movie::Status MovieAudioInputHandler::AddBufferToAudioIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_AudioListMutex );
    m_AudioBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex( &m_AudioListMutex );
    return movieStatus;
}

movie::Status MovieAudioInputHandler::RemoveBufferFromAudioIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_AudioListMutex );
    int32_t index = -1;
    if( m_AudioBufferIndexList.size() > 0 )
    {
        index = m_AudioBufferIndexList.front();
        m_AudioBufferIndexList.erase(m_AudioBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex( &m_AudioListMutex );
    return movieStatus;
}

movie::Status MovieAudioInputHandler::GetAudioIndexListSize(int32_t* indexListSize)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_AudioListMutex);
    *indexListSize = m_AudioBufferIndexList.size();
    nn::os::UnlockMutex(&m_AudioListMutex);
    return movieStatus;
}

movie::Status MovieAudioInputHandler::AudioInputDataAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    SignalAudioInputBufferAvailable();
    return movieStatus;
}

movie::Status MovieAudioInputHandler::ReadInputDataFromExtractorSendTodecoder(int32_t index)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    movieStatus = m_AudioDecoder->GetInputBuffer(index, &buffer);
    if( movieStatus == movie::Status_Success )
    {
        buffer.SetRange(0, 0);
        movieStatus = m_Extractor->Read(&buffer);
        if( ( movieStatus != movie::Status_Success ) ||
            ( buffer.Size() <= 0 ) ||
            ( movieStatus == movie::Status_EndOfStream ) )
        {
            flags = movie::BufferFlags_EndOfStream;
        }
        m_Extractor->GetSampleTime(&presentationTimeUs);
        m_AudioDecoder->SendInputBufferForDecode(index, buffer.Offset(), buffer.Size(), presentationTimeUs, flags);
        m_Extractor->Advance();
    }
    return movieStatus;
}

movie::Status MovieAudioInputHandler::CheckForInputBuffersAndReadAudioInputData()
{
    movie::Status movieStatus = movie::Status_NotFound;
    size_t currentTrackIndex = -1;

    if( m_Extractor != NULL )
    {
        currentTrackIndex = -1;
        movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
        if( movieStatus == movie::Status_EndOfStream )
        {
            currentTrackIndex = m_AudioTrackIndex;
        }
        else if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
    }

    movieStatus = movie::Status_NotFound;
    if( currentTrackIndex == m_AudioTrackIndex )
    {
        int index = -1;
        RemoveBufferFromAudioIndexList(&index);
        if( index != -1 )
        {
            movieStatus = ReadInputDataFromExtractorSendTodecoder(index);
        }
    }
    return movieStatus;
}

movie::Status MovieAudioInputHandler::SignalAudioInputBufferAvailable()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieAudioInputHandler::MoviePlayerMessage_AudioInputAvailable);
    return movieStatus;
}

void MovieAudioInputHandler::SignalFlush()
{
    m_FlushInput = true;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieAudioInputHandler::MoviePlayerMessage_AudioInputFlush);
}

void MovieAudioInputHandler::Flush()
{
    nn::os::LockMutex(&m_AudioListMutex);
    m_AudioBufferIndexList.clear();
    nn::os::UnlockMutex(&m_AudioListMutex);
    m_FlushInput = false;
}

void MovieAudioInputHandlerThreadFunction(void *arg)
{
    MovieAudioInputHandler *movieAudioInputHandler = ( MovieAudioInputHandler* ) arg;
    if( movieAudioInputHandler != nullptr )
    {
        for( ;;)
        {
            uintptr_t data = 0;
            nn::os::ReceiveMessageQueue(&data, &movieAudioInputHandler->m_MessageQueue);
            if( data == MovieAudioInputHandler::MoviePlayerMessage_AudioInputAvailable )
            {
                movie::Status movieStatus = movieAudioInputHandler->CheckForInputBuffersAndReadAudioInputData();
                if( movieStatus != movie::Status_Success )
                {
                    if( movieAudioInputHandler->m_FlushInput == false )
                    {
                        int64_t sleepNs = 10000; // Retry after 10000Ns. This sleep can be tweaked based on application need.
                                                 // Short sleep is needed when heavy(system calls at nanosecond intervals)
                                                 // system calls are done in another thread.
                        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(sleepNs));
                        movieAudioInputHandler->SignalAudioInputBufferAvailable();
                    }
                }
            }
            else if( data == MovieAudioInputHandler::MoviePlayerMessage_AudioInputThreadExit )
            {
                break;
            }
            else if( data == MovieAudioInputHandler::MoviePlayerMessage_AudioInputFlush )
            {
                movieAudioInputHandler->Flush();
            }
            else
            {
                movieAudioInputHandler->CheckForInputBuffersAndReadAudioInputData();
            }
        }
    }
}
