﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs.h>
#include "MovieDecoderPlayer.h"
#include <nns/mm/mm_MoviePlayerUtils.h>
#include <nns/mm/mm_HidHandler.h>

class MovieDecoderMultiInstance
{
public:
    struct DecoderState;

    NN_IMPLICIT MovieDecoderMultiInstance(int numberOfVideosToDecode);

    /**
    * @brief
    * Enumeration of MultiInstance sample event types.
    */
    enum MultiInstanceEventType
    {
        MultiInstanceEventType_Unknown      = 0,
        MultiInstanceEventType_StateChanged = 1,
        MultiInstanceEventType_Error        = 2,
        MultiInstanceEventType_ButtonPressA = 3,
        MultiInstanceEventType_ButtonPressB = 4,
        MultiInstanceEventType_ButtonPressX = 5,
    };

   /**
    * @brief Initialize @class MovieDecoderMultiInstance.
    *
    * @return none
    *
    * @details
    * An application can use this API to initialize an instance of @class MovieDecoderMultiInstance.
    * Thread multi wait is initialized and player events are linked to this multi wait.
    *
    */
    void Initialize();

   /**
    * @brief
    * Finalize @class MovieDecoderMultiInstance..
    *
    * @return none
    *
    * @details
    * Player events are unlinked from multi wait and multi wait is finalized.
    *
    */
    void Finalize();

   /**
    * @brief
    * CreateHidHandler.
    *
    * @return true                              HidHandler creation successful.
    * @return false                             HidHandler creation failed.
    *
    * @details
    * Create an instance of @class HidHandler for processing controller input. Registers events for controller
    * key press. Initializes @class HidHandler instance.
    *
    */
    bool CreateHidHandler();

   /**
    * @brief
    * DestroyHidHandler.
    *
    * @return none
    *
    * @details
    * If @class HidHandler instance exists, destroy the same. Finalize @class HidHandler instance.
    *
    */
    void DestroyHidHandler();

   /**
    * @brief
    * CreateMovieDecoderPlayer
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * Create an instance of @class MovieDecoderPlayer and initialize it. Events for player state change and error
    * reporting are registered with the player.
    *
    */
    movie::Status CreateMovieDecoderPlayer();

   /**
    * @brief
    * DestroyMovieDecoderPlayer
    *
    * @return none
    *
    * @details
    * If exists, destroy @class MovieDecoderPlayer instance.
    *
    */
    void DestroyMovieDecoderPlayer();

   /**
    * @brief
    * AddMovie
    *
    * @param[in] path                           Movie path.
    * @param[in] videoDecoderMode               Video decoder mode.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * Adds movie path and video coder mode to the list of movies to be played.
    *
    */
    movie::Status AddMovie(std::string path, movie::DecoderMode videoDecoderMode);

   /**
    * @brief
    * PlayMovies
    *
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    * Check for existence of given movie files. Set movie files to @class MovieDecoderPlayer. Select only
    * video tracks from the movie file, audio is not supported. Prepare and start playback of movie files.
    *
    */
    movie::Status PlayMovies();

   /**
    * @brief
    * HandleEvents.
    *
    * @return true                              Processing of events successful.
    * @return false                             Processing of events failed.
    *
    * @details
    * The sample main thread will wait for player and controller events. When sample receives playback completed state
    * change event, player will be stopped. Based on HID key press event, playback will be paused or resumed.
    *
    */
    bool HandleEvents();

   /**
    * @brief
    * RegisterEvent. API to register @class MultiInstanceEventType events.
    *
    * @param[in] event                          Event to register movie::Extractor.
    * @param[in] decoder                        Event type.
    * @param[in] decoderEvents                  MultiWaitHolderType for the event.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to register @class MultiInstanceEventType events. This API will link the event with sample
    * multi wait.
    *
    */
    void RegisterEvent(nn::os::EventType *event, MultiInstanceEventType eventType, nn::os::MultiWaitHolderType *holder);

   /**
    * @brief
    * MountFileSystem.
    *
    * @return true                              Mount file system successful.
    * @return false                             Mount file system failed.
    *
    * @details
    * Parses the media file path to determine the file system type. Either host or SD card files system is mounted.
    *
    */
    bool MountFileSystem(char* mediaFile);

   /**
    * @brief
    * UnMountFileSystem.
    *
    * @return none
    *
    * @details
    * Unmounts a file system, if exists.
    *
    */
    void UnMountFileSystem();

   /**
    * @brief
    * EnableLooping.
    *
    * @return none
    *
    * @details
    * Enables looping mode of player. The movie files will play continuously until stopped by user.
    *
    */
    void EnableLooping();

    ~MovieDecoderMultiInstance() {};

public:
    MovieDecoderPlayer *m_MovieDecoderPlayer;
    HidHandler         *m_HidHandler;
    nn::os::MultiWaitType       m_PlayerDecoderMultiWait;
    bool                        m_EventsInitialized;
    bool                        m_SdcardMounted;
    bool                        m_HostFsMounted;
    MovieDecoderPlayer::State   m_State;
    nn::vi::NativeWindowHandle m_NativeWindow;

    nn::os::EventType          m_PlayerStateChangedEvent;
    nn::os::EventType          m_PlayerErrorEvent;

    struct EventUserData
    {
        EventUserData(nn::os::EventType *event, MultiInstanceEventType eventType)
            : m_Event(event),
            m_EventType(eventType)
        { }
        nn::os::EventType *m_Event;
        MultiInstanceEventType m_EventType;
    };
    void DeleteEventUserData(EventUserData* userData);
    bool DoesMediaFileExists(char* mediaFileName);

    nn::os::EventType          m_HidEventButtonPressA;
    nn::os::EventType          m_HidEventButtonPressB;
    nn::os::EventType          m_HidEventButtonPressX;
    nn::os::EventType          m_HidEventButtonPressZr;
    nn::os::EventType          m_HidEventButtonPressL;
    nn::os::EventType          m_HidEventButtonPressR;

    nn::os::MultiWaitHolderType m_PlayerStateChangedEventHolder;
    nn::os::MultiWaitHolderType m_PlayerErrorEventHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressAHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressBHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressXHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressZrHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressLHolder;
    nn::os::MultiWaitHolderType m_HidEventButtonPressRHolder;
    int m_NumberOfVideosToDecode;
    struct MovieInfo
    {
        MovieInfo(std::string path, movie::DecoderMode mode) :
            moviePath(path),
            decoderMode(mode)
        { }
        std::string moviePath;
        movie::Extractor* extractor;
        movie::DecoderMode decoderMode;
    };
    std::vector<MovieInfo> m_MovieInfoList;
};

