﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nfc.h>
#include "Graphics.h"
#include "TagUtility.h"
#include "WriteState.h"

namespace nns { namespace nfc { namespace mifare { namespace {

    char g_Buffer[64];
    const int LabeledValueOffset = 20;

    inline int ToElapsedFrame(const ApplicationData& data) NN_NOEXCEPT
    {
        return data.frame - data.transitFrame;
    }

    inline void WriteLabeledValue(
            int x, int y,
            const char* label,
            const char* value
        ) NN_NOEXCEPT
    {
        WriteText(x, y, label);
        WriteText(x + LabeledValueOffset, y, value);
    }

    void WriteTagInfo(
            int& x, int& y,
            const nn::nfc::TagInfo& tagInfo
        ) NN_NOEXCEPT
    {
        const nn::Bit8* uid = reinterpret_cast<const nn::Bit8*>(tagInfo.tagId.uid);
        char* buffer = g_Buffer;
        std::sprintf(buffer, ":");
        buffer += 1;
        for(int i = 0; i < tagInfo.tagId.length && i < sizeof(tagInfo.tagId.uid); i++)
        {
            std::sprintf(buffer, " %02X", uid[i]);
            buffer += 3;
        }
        WriteLabeledValue(x, y++, "UID", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.protocol);
        WriteLabeledValue(x, y++, "PROTOCOL", g_Buffer);

        std::sprintf(g_Buffer, ": %d", tagInfo.type);
        WriteLabeledValue(x, y++, "TYPE", g_Buffer);
    }

    void WriteStateNone(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        NN_UNUSED(data);

        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: initialize nfc library");
        WriteText(x, y++, "Y: exit");
    }

    void WriteStateInitializing(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Initializing", ToElapsedFrame(data));
    }

    void WriteStateInitialized(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceListing(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceInitialize(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Connecting", ToElapsedFrame(data));
    }

    void WriteStateDeviceSelect(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: start detection");
        WriteText(x, y++, "B: finalize nfc library");

        ++y;
        for( int i = 0; i < data.deviceCount; i++)
        {
            std::sprintf(g_Buffer, "    NpadId 0x%02X", data.npadIds[i]);
            const bool selected = i == data.deviceIndex;
            WriteText(x, y++, g_Buffer, selected ? Color::Orange : Color::White);
        }
    }

    void WriteStateStartDetection(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Starting", ToElapsedFrame(data));
    }

    void WriteStateStopDetection(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Suspending", ToElapsedFrame(data));
    }

    void WriteStateTagRelease(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Finished", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Please remove the tag");
    }

    void WriteStateTagSearch(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Searching", ToElapsedFrame(data));

        ++y;
        std::sprintf(g_Buffer, "Please touch the tag to NpadId 0x%02X", data.npadIds[data.deviceIndex]);
        WriteText(x, y++, g_Buffer);
        WriteText(x, y++, "B: cancel");
    }

    void WriteStateDeviceCheck(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Checking", ToElapsedFrame(data));
    }

    void WriteStateTagRead(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Reading", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagCheck(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Checking", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagWrite(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Writing", ToElapsedFrame(data));

        ++y;
        WriteText(x, y++, "Do not remove the tag during access");
    }

    void WriteStateTagEnabled(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteText(x, y++, "A: update counter");
        WriteText(x, y++, "B: cancel");

        ++y;
        WriteTagInfo(x, y, data.tagInfo);

        ++y;
        std::sprintf(g_Buffer, ": %u", data.counter1);
        WriteLabeledValue(x, y++, "COUNTER1", g_Buffer);

        std::sprintf(g_Buffer, ": %u", data.counter6);
        WriteLabeledValue(x, y++, "COUNTER6", g_Buffer);
    }

    void WriteStateFinalize(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Finalizing", ToElapsedFrame(data));
    }

    void WriteStateError(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteErrorText(x, y, "ERROR: ");
        x += 7;

        nn::Result result = data.lastResult;
        if( result <= nn::nfc::ResultNfcDeviceNotFound() )
        {
            WriteErrorText(x, y++, "nfc device not found");
        }
        else if( result <= nn::nfc::ResultNfcDisabled() )
        {
            WriteErrorText(x, y++, "nfc disabled");
        }
        else if( result <= nn::nfc::ResultAccessError() )
        {
            WriteErrorText(x, y++, "please check a key and retry");
        }
        else if( result <= nn::nfc::ResultNeedRestart() )
        {
            WriteErrorText(x, y++, "keep applying the same tag");
        }
        else if( result <= nn::nfc::ResultNotSupported())
        {
            WriteErrorText(x, y++, "this tag is not supported");
        }
        else if( result <= nn::nfc::ResultMaxNfcDeviceActivated() )
        {
            WriteErrorText(x, y++, "nfc device already activated");
        }
        else if( result <= nn::nfc::ResultConflictFunction() )
        {
            WriteErrorText(x, y++, "nfc device is conflicted");
        }
        else
        {
            WriteErrorText(x, y++, "unexpected");
        }

        x  = 1;
        y += 1;
        WriteText(x, y++, "A: restart");
        WriteText(x, y++, "B: finalize nfc library");
    }

    void WriteStateExit(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        int x = 1;
        int y = 1;
        WriteLoadingText(x, y++, "Exit", ToElapsedFrame(data));
    }

    typedef void (*WriteStateFunc)( const ApplicationData& data );
    struct WriteStateEntry
    {
        State           state;
        WriteStateFunc  func;
    };

    const WriteStateEntry WriteStateTable[] = {
        {   State_None,                 WriteStateNone              },
        {   State_Initializing,         WriteStateInitializing      },
        {   State_Initialized,          WriteStateInitialized       },
        {   State_DeviceListing,        WriteStateDeviceListing     },
        {   State_DeviceInitialize,     WriteStateDeviceInitialize  },
        {   State_DeviceSelect,         WriteStateDeviceSelect      },
        {   State_StartDetection,       WriteStateStartDetection    },
        {   State_StopDetection,        WriteStateStopDetection     },
        {   State_TagRelease,           WriteStateTagRelease        },
        {   State_TagSearch,            WriteStateTagSearch         },
        {   State_DeviceCheck,          WriteStateDeviceCheck       },
        {   State_TagRead,              WriteStateTagRead           },
        {   State_TagCheck,             WriteStateTagCheck          },
        {   State_TagWrite,             WriteStateTagWrite          },
        {   State_TagEnabled,           WriteStateTagEnabled        },
        {   State_Finalize,             WriteStateFinalize          },
        {   State_Error,                WriteStateError             },
        {   State_Exit,                 WriteStateExit              },
    };
    const int WriteStateCountMax = sizeof(WriteStateTable) / sizeof(WriteStateTable[0]);
    NN_STATIC_ASSERT( WriteStateCountMax == static_cast<int>(State_CountMax) );

    WriteStateFunc GetWriteStateFunc(State state)
    {
        const int index = static_cast<int>(state);
        NN_ASSERT( index >= 0 && index < WriteStateCountMax );
        NN_ASSERT( WriteStateTable[index].state == state );
        return WriteStateTable[index].func;
    }

}}}} // end of namespace nns::nfc::mifare::unnamed

namespace nns { namespace nfc { namespace mifare {

    void WriteState(
            const ApplicationData& data
        ) NN_NOEXCEPT
    {
        // 状態に応じて画面に描画する内容を変更します。
        (*GetWriteStateFunc(data.state))(data);
    }

}}} // end of namespace nns::nfc::mifare
