﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "HeapTracker.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

//hashing algorithm borrowed from http://www.cse.yorku.ca/~oz/hash.html
static unsigned long djb2_hash(unsigned char *str, size_t length)
{
    unsigned long hash = 5381;

    while (length--)
        hash = ((hash << 5) + hash) + *str++; /* hash * 33 + *str */

    return hash;
}

static size_t Align(size_t size, size_t align)
{
    return (size + align - 1) / align * align;
}

HeapTracker::HeapTracker() : m_mutex(true), m_creationTime(nn::os::GetSystemTick()), m_total(0), m_maxSize(0), m_id(0), m_capacity(1024), m_allocations(0), m_memoryBlockSize(0), m_infos(nullptr)
{
    m_memoryBlockSize = Align(m_capacity * sizeof(AllocationInfo), nn::os::MemoryBlockUnitSize);
    uintptr_t address;
    nn::os::AllocateMemoryBlock(&address, m_memoryBlockSize);
    m_infos = new (reinterpret_cast<void*>(address)) AllocationInfo[m_capacity];
}

void HeapTracker::Track(void *memory, size_t size)
{
    // m_mutex.Lock();
    if(m_allocations > m_capacity * 3 / 4)
    {
        size_t newCapacity = m_capacity * 2;
        size_t newMemoryBlockSize = Align(newCapacity * sizeof(AllocationInfo), nn::os::MemoryBlockUnitSize);
        uintptr_t address;
        nn::os::AllocateMemoryBlock(&address, newMemoryBlockSize);
        AllocationInfo *newInfos = new (reinterpret_cast<void*>(address)) AllocationInfo[newCapacity];
        for(size_t i = 0; i < m_capacity; ++i)
        {
            if(m_infos[i].m_inUse)
            {
                size_t index = djb2_hash(reinterpret_cast<unsigned char*>(&m_infos[i].m_address), sizeof(void*)) % newCapacity;
                while(newInfos[index].m_inUse)
                    index = (index + 1) % newCapacity;
                newInfos[index] = m_infos[i];
            }
        }
        nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(m_infos), m_memoryBlockSize);
        m_infos = newInfos;
        m_memoryBlockSize = newMemoryBlockSize;
        m_capacity = newCapacity;
    }

    size_t index = djb2_hash(reinterpret_cast<unsigned char*>(&memory), sizeof(void*)) % m_capacity;
    while(m_infos[index].m_inUse)
        index = (index + 1) % m_capacity;
    m_infos[index] = AllocationInfo(size, m_id++, memory, true);
    m_total += size;
    m_maxSize = std::max(m_maxSize, m_total);
    ++m_allocations;
    // m_mutex.Unlock();
}

void HeapTracker::Untrack(void *memory)
{
    if(!memory)
        return;
    // m_mutex.Lock();
    size_t index = djb2_hash(reinterpret_cast<unsigned char*>(&memory), sizeof(void*)) % m_capacity;
    while(m_infos[index].m_address != memory)
        index = (index + 1) % m_capacity;
    m_infos[index].m_inUse = false;
    m_total -= m_infos[index].m_size;
    --m_allocations;
    // m_mutex.Unlock();
}

void HeapTracker::OutputUsage()
{

    // for(int i = 0; i < m_maxAllocations; ++i)
    // {
        // if(m_infoUsed[i])
        // {
            // AllocationInfo *info = m_infos + i;
            // nn::TimeSpan t = (info->m_time - m_creationTime).ToTimeSpan();
            // NN_LOG("memory: 0x%p, size: %lu, id: %lu, time: %ld.%ld\n", info->m_address, info->m_size, info->m_id, t.GetSeconds(), t.GetMilliSeconds() % 1000);
        // }

    // }
    NN_LOG("current size: %lu max size: %lu\n", m_total, m_maxSize);
}


HeapTracker::AllocationInfo::AllocationInfo(size_t size, size_t id, void *address, bool inUse) : m_size(size), m_id(id), m_address(address), m_time(nn::os::GetSystemTick()) , m_inUse(inUse)
{
}
