﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <sys/types.h>

#include <nn/fs.h>
#include <nn/nn_Assert.h>

#include <movie/Extractor.h>
#include <movie/PlayerExtractor.h>
#include <movie/DataSource.h>
#include <movie/MSESource.h>
#include <nn/os/os_Mutex.h>
#include <vector>


namespace movie {    namespace sample {

/****
* Implements a MSE DataSource using NX SDK
* You must initialize fs library (nn::fs::Initialize) before using this class
*/

enum class MseDataType
{
    INITIALIZATION_SEGMENT = 0,
    DATA_SEGMENT,
    END_OF_FILE_SEGMENT
};

enum class MseSegmentStatus{
    MSE_INITIALIZATION_SEGMENT = 0,
    MSE_INITIALIZATION_SEGMENT_INCOMPLETE,
    MEDIA_SEGMENT,
    MEDIA_SEGMENT_INCOMPLETE_HEADER,
    MEDIA_SEGMENT_INCOMPLETE,
    INVALID_SEGMENT
};

struct MseIdrEntry {
    int sample_index_;
    int64_t presentation_time_;
    friend bool operator<(int v, MseIdrEntry const& s) noexcept { // used by std::upper_bound comparison
        return v < s.presentation_time_;
    }
};

struct MseFrameData
{
    MseFrameData(size_t buffSize, size_t trackIndex) noexcept:
    mData{ buffSize },
    mStatus{ movie::Status::Status_Success },
    mPresentationTime{ -1 },
    mIsIFrame{false},
    mTrackIndex{trackIndex}
    {};

    bool is_idr() const noexcept {
        return mData.GetInt32Data() & movie::BufferFlags::BufferFlags_SyncFrame;
    }

    movie::Buffer mData;
    movie::Status mStatus;
    int64_t mPresentationTime;
    bool mIsIFrame;
    size_t mTrackIndex;
};

struct MseTrackBuffer {
    std::vector<MseFrameData> frameBuffers;
    std::vector<MseIdrEntry> mIdrIndices;
    size_t mCurrentSample;
    bool isAudio;
    int64_t totalDuration;
    int64_t totalSize;
    int64_t trackIndex;
};

/*
DataSegment has a similiar structure to a Skip List Node except it doesn't have
a key-value pair.
*/
struct DataSegment
{
    int64_t mOffset;
    int64_t mLength;
    uint8_t* mData;
    bool mPending;
    int64_t mTimeOffsetUs;
    int64_t mCachedDurationUs;
    int64_t mStartTimeUs;
    int32_t mSegmentIndex;
    int64_t mDataBegin;
    int64_t mDataEnd;

    // If DATA_SEGMENT
    //next = next data segment
    //nextInitialization = NULL
    // Else if INITIALIZATION_SEGMENT
    //next = next data segment
    //nextInitialization = next initialization segment
    // Else if END_OF_FILE_SEGMENT
    //next = NULL
    //nextInitialization = NULL
    DataSegment* next;                  // next data segment
    DataSegment* nextInitialization;    // next initialization segment
    MseDataType type;

    explicit DataSegment(size_t size)
    {
        mOffset = 0;
        mLength = 0;
        mData = nullptr;
        mPending = true;
        next = nullptr;
        nextInitialization = nullptr;
        if (size > 0)
        {
            mData = new uint8_t[size];
        }
        else
        {
            mData = nullptr;
        }

    }

    ~DataSegment()
    {
        if (mData)
        {
            delete[] mData;
        }

        mData = nullptr;
        next = nullptr;
        nextInitialization = nullptr;
    }
};

class MseSourceBuffer
{
public:
    MseSourceBuffer();
    ~MseSourceBuffer();

    // List Management
    int64_t appendData(void* data, size_t size, MseDataType type);
    movie::Status processAppendedData();

    // Query Management
    bool getEndOfData();
    int getTotalDataBuffered();
    static bool isTypeSupported(const char* in_mimetype, const char* in_codec, const char* in_system, const char* in_url);

    movie::Status Advance();
    movie::Status ReadSampleData(movie::Buffer *buffer);

    size_t CountTracks() { return mTrackFormats.size(); };

    std::vector<movie::MediaData> mTrackFormats;
    std::vector<MseTrackBuffer> mTrackBuffers;
    size_t mCurrentSample;
    size_t mCurrentSampleTrackIndex;
    int64_t mCurrentTimeStampOffset;
    int64_t mTimeStampOffset;

private:
    int64_t mLength;
    int64_t mTimeOffsetUs;

    int64_t mSegmentIndex;
    bool mEOFReached;

    MseSourceBuffer(MseSourceBuffer const&) = delete;
    MseSourceBuffer& operator=(MseSourceBuffer const&) = delete;

    nn::os::Mutex mMutex;

    DataSegment* mSegmentBuffers;                   // head of list
    DataSegment* mCurrentInitializationSegment;     // Current initialization segment that is readat()
    DataSegment* mCurrentDataSegment;               // Current Data Segment that is readat()
    DataSegment* mLastAppendedInitSegment;          // Last Initialization segment appended to buffer
    DataSegment* mLastAppendedSegment;              // Last Data segment appended to buffer
};


}}
