﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LidbeSimple.cpp,PageSampleLidbeSimple}
 *
 * @brief
 *  LIDBE ライブラリのシンプルなサンプルプログラム
 */

/**
 * @page PageSampleLidbeSimple LIDBEライブラリの基本操作
 * @tableofcontents
 *
 * @brief
 *  LIDBE ライブラリを使用したIDBE情報取得サンプルプログラムの解説です。
 *
 * @section PageSampleLidbeSimple_SectionBrief 概要
 *  LIDBE ライブラリを使用して、IDBE情報の取得を行うサンプルです。
 *
 * @section PageSampleLidbeSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/LidbeSimple
 *  Samples/Sources/Applications/LidbeSimple @endlink 以下にあります。
 *
 * @section PageSampleLidbeSimple_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleLidbeSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleLidbeSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleLidbeSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLidbeSimple_SectionDetail 解説
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ネットワーク環境の初期化
 *
 * - IDBEファイルのダウンロードが行えるかの確認
 * - ダウンロード時のキャンセル動作の確認
 * - ダウンロードしたIDBEファイルの正当性の確認
 *     - ファイルサイズの確認
 *     - 復号化の確認
 *     - IDBEファイル情報(ShortTitle、LongTitle、Publisher)の確認
 *
 */

#include <curl/curl.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_UuidApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_ResEndian.h>

#include <nn/lidbe.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/socket.h>


namespace
{
    const size_t ThreadStackSize = 8192;  // スレッド操作スレッドのスタックサイズ
    const nn::TimeSpanType timespan = nn::TimeSpanType::FromMilliSeconds(10);

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;  // 1つ目のスレッドのスタック
    NN_ALIGNAS(4096) uint8_t g_Stack[ThreadStackSize];

    nn::os::ThreadType g_ThreadStack;
    nn::Result ret = nn::Result();
    bool isSocketInitialized = false;
}

bool InitializeNetwork()
{
    nn::nifm::Initialize();

    NN_LOG("Waiting for network interface availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();

    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Network is not available.\n");
        return false;
    }

    ret = nn::socket::Initialize(g_SocketConfigWithMemory);
    isSocketInitialized = true;

    curl_global_init(CURL_GLOBAL_ALL);

    NN_LOG("Connecting to network...\n");

    if (ret.IsFailure())
    {
        NN_LOG("Connection could not be established.\n");
        return false;
    }

    NN_LOG("Connection was established.\n");


    return true;
}

void FinalizeNetwork()
{
    curl_global_cleanup();
    if (isSocketInitialized)
    {
        nn::socket::Finalize();
    }
}

void CancelThreadFunction(void *arg)
{
    nn::os::SleepThread(timespan);

    NN_LOG("* nn::lidbe::Cancel().\n");
    nn::lidbe::Cancel();

    return;
}

void GetIconInfo(void* pIconData, const nn::lidbe::DownloadContext context)
{
    nn::lidbe::ApplicationIdbeAccessor appIdbeAccessor;
    appIdbeAccessor.Initialize(pIconData, (context.readSize - (IdbeHashSize + IdbeFileHeaderSize)));
    nn::lidbe::ApplicationIdbeProperty appIdbeProperty = appIdbeAccessor.GetProperty();
    nn::lidbe::ApplicationTitle appTitle = appIdbeProperty.GetDefaultTitle();

    NN_ALIGNAS(4096) uint8_t utf8Buff[256] = {'\0'};
    nn::util::ConvertStringUtf16NativeToUtf8(reinterpret_cast<char*>(utf8Buff), sizeof(utf8Buff), reinterpret_cast<uint16_t*>(appTitle.shortName));
    NN_LOG("* ApplicationTitle.shortName=%s\n", utf8Buff);
    memset(utf8Buff, '\0', sizeof(utf8Buff));

    nn::util::ConvertStringUtf16NativeToUtf8(reinterpret_cast<char*>(utf8Buff), sizeof(utf8Buff), reinterpret_cast<uint16_t*>(appTitle.longName));
    NN_LOG("* ApplicationTitle.longName=%s\n", utf8Buff);
    memset(utf8Buff, '\0', sizeof(utf8Buff));

    nn::util::ConvertStringUtf16NativeToUtf8(reinterpret_cast<char*>(utf8Buff), sizeof(utf8Buff), reinterpret_cast<uint16_t*>(appTitle.publisher));
    NN_LOG("* ApplicationTitle.publisher=%s\n", utf8Buff);

    return;
}

void Download(uint64_t programId, uint16_t version)
{
    NN_LOG("****************************************************************************************************\n");
    NN_LOG("**** TitleId = %016llX, version = %d\n", programId, version);

    // IDBE_FILE_SIZE_WUP > IDBE_FILE_SIZE_CTR なので WUP 用のバッファサイズでメモリを確保

    NN_ALIGNAS(4096) uint8_t iconFile[IDBE_FILE_SIZE_WUP];
    NN_ALIGNAS(4096) uint8_t iconData[IdbeSizeWup];

    NN_LOG("** CASE-1 (Normal)\n");
    {
        NN_LOG("* Call nn::lidbe::DownloadIconFile().\n");

        if (nn::lidbe::DownloadIconFile(&iconFile, programId, version, true))
        {
            NN_LOG("* Call nn::lidbe::DecryptIconFile().\n");

            if (nn::lidbe::DecryptIconFile(&iconData, &iconFile, programId))
            {
                NN_LOG("  Succeeded!\n");
            }
            else
            {
                NN_LOG("  Failed.\n");
            }
        }
        else
        {
            NN_LOG("  Failed.\n");
        }
    }

    NN_LOG("** CASE-2 (Cancel)\n");
    {
        NN_LOG("* Call nn::lidbe::DownloadIconFile().\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_ThreadStack, CancelThreadFunction,  NULL, &g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_ThreadStack);

        // キャンセルが行われるため、失敗します。
        if (nn::lidbe::DownloadIconFile(&iconFile, programId, version, true))
        {
            NN_LOG("* Call nn::lidbe::DecryptIconFile().\n");

            if (nn::lidbe::DecryptIconFile(&iconData, &iconFile, programId))
            {
                NN_LOG("  Succeeded!\n");
            }
            else
            {
                NN_LOG("  Failed.\n");
            }
        }
        else
        {
            NN_LOG("  Failed. (TEST OK)\n");
        }

        nn::os::WaitThread(&g_ThreadStack);
    }

    NN_LOG("** CASE-3 (Latest)\n");
    {
        NN_LOG("* Call nn::lidbe::DownloadLatestIconFile().\n");

        if (nn::lidbe::DownloadLatestIconFile(&iconFile, programId, true))
        {

        }
        else
        {
            NN_LOG("  Failed.\n");
        }
    }

    NN_LOG("** CASE-4 (User Perform)\n");
    {
        nn::lidbe::DownloadContext context;

        NN_LOG("* Call nn::lidbe::CreateLatestIconDownloadContext().\n");

        if (nn::lidbe::CreateLatestIconDownloadContext(&context, &iconFile, programId, true))
        {
            NN_LOG("* Call curl_easy_perform().\n");

            CURLcode errorCode = curl_easy_perform(context.pCurl);

            if (errorCode == CURLE_OK)
            {
                NN_LOG("  Succeeded!\n");
                NN_LOG("* FileSize=%u\n", context.readSize);
            }
            else
            {
                NN_LOG("  Failed. (errorCode = %d)\n", errorCode);
            }

            NN_LOG("* Call nn::lidbe::CheckDownloadResult().\n");

            if (nn::lidbe::CheckDownloadResult(&context))
            {
                NN_LOG("* Call nn::lidbe::DecryptIconFile().\n");

                if (nn::lidbe::DecryptIconFile(&iconData, &iconFile, programId))
                {
                    GetIconInfo(iconData, context);
                    NN_LOG("  Succeeded!\n");
                }
                else
                {
                   NN_LOG("  Failed.\n");
                }
            }
            else
            {
                NN_LOG("  Failed.\n");
            }

            nn::lidbe::DestroyDownloadContext(&context);
        }
        else
        {
            NN_LOG("  Failed.\n");
        }
    }
} //NOLINT(impl/function_size)

extern "C" void nnMain()
{
    if (!InitializeNetwork())
    {
        FinalizeNetwork();
        return;
    }

    if (nn::lidbe::Initialize())
    {
        // WUP IDBE テスト用タイトル
        Download(0x000500001FBF1000ULL, 0);
        // CTR ホームメニュー
        Download(0x0004001000020000ULL, 0);
    }
    else
    {
        NN_LOG("nn::lidbe::Initialize() failed.\n");
    }

    nn::lidbe::Finalize();

    FinalizeNetwork();

    return;
}
