﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <upnp/upnp.h>
#include <upnp/upnptools.h>
#include <upnp/upnpdebug.h>
#include <upnp/nnlibupnp_VirtualDir.h>

#include <cstdlib>

#include "UpnpDeviceGameState.h"
#include "UpnpDeviceGamePlayer.h"
#include "UpnpDeviceGameUI.h"

/**
 * @examplesource{UpnpDevice.cpp,PageSampleUpnpDevice}
 *
 * @brief
 *  Use libupnp to advertise and play a game on the Nintendo Switch
 */

/**
 * @page PageSampleUpnpDevice UpnpDevice
 * @tableofcontents
 *
 * @brief
 *  Documentation about the Sample program which uses libupnp to advertise and play a game on the Nintendo Switch
 *
 * @section PageSampleUpnpDevice_SectionBrief Overview
 *  First this sample initializes network connectivity by using the NIFM library.  Then the libupnp library is
 *  initialized.  The Sample will then start the "GamePlayer" object which will search the local LAN for
 *  a running "Nintendo Game" using libupnp.  If 5 seconds elapse, and no existing Nintendo Game is found by the
 *  GamePlayer object, then the Sample will start a local copy of the "GameState" object.  The GameState object
 *  will initialize the game state and will advertise the running Nintendo Game to the local LAN so that other
 *  Sample programs can join the running game at a later time.  The Sample will then start the "GameUI" object
 *  which will render the Nintendo Game user interface on the Nintendo Switch.  The User can now begin playing
 *  the game, see below for instructions on how to play the game.  To exit the Nintendo Game, and this Sample,
 *  touch the "E X I T" yellow text located at the right bottom of the Nintendo Switch screen display.
 *
 * @subsection PageUpnpDevice_SectionExpectedOutput Expected Output
 * @image html UpnpDevice.png
 *
 * @section UpnpDevice_SectionFileStructure File Structure
 *  This sample program can be found at
 *  @link ../../../Samples/Sources/Applications/Libupnp/UpnpDevice Samples/Sources/Applications/Libupnp/UpnpDevice @endlink
 *
 * @section PageSampleUpnpDevice_SectionNecessaryEnvironment System Requirements
 *  It is required to import network setting by SettingManager prior to running this sample.
 *  Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 *  The "Nintendo Game" is a simple of game of changing the color of a displayed square box from one color to
 *  new color.  There are three square boxes displayed on the Nintendo Switch screen which are labled: A, B
 *  and C.  To change the color of a square box use your finger to touch or select one of the boxes.  When you
 *  select a box it will briefly change color to WHITE, which indicates that the box was successfully
 *  touched/selected, and then the square box will change to a different color.  Each time a square box changes to a
 *  new color a UPnP SOAP ACTION message is sent across the local network (LAN) to the GameState object using libupnp.
 *  The GameState object will accept the new square box color and will return a success result to the sample.
 *  The GameState will also notify all other Nintendo Switches playing the same Nintendo Game that the selected
 *  square box has a new color and those Nintendo Switches will update the color of the selected square box on their
 *  screens.  Any Nintendo Switch playing the game can update a given square box to a different color and the changed
 *  square box color will be reflected on the other Nintendo Switches.
 *
 *  If you open Windows Network Neighborhood while the sample is running you will find the sample listed in the output
 *  as "UPnP Nintendo Game" under "Other Devices" (Windows 10). Windows Network Neighborhood documentation can
 *  be found here: https://en.wikipedia.org/wiki/My_Network_Places
 *
 * @section PageSampleUpnpDevice_SectionHowToOperate Operation Procedure
 *  The Sample is designed to work on either a single Nintendo Switch or multiple Nintendo Switches.  The first Nintendo
 *  Switch that is started will host the "GameState" for the local LAN.  The other Nintendo Switches running the sample
 *  will access the running GameState using libupnp.
 *
 *  NOTE: By default the sample will use the Nifm negotiated network interface to send and receive libupnp generated
 *  multicast network traffic.  If the sample is running on a host that supports multiple network interfaces the
 *  desired network interface to use can be selected by passing the command line argument "-interface <Ipv4 Address>",
 *  where <Ipv4 Address> is assigned to the host network interface that the sample should use.
 *
 * @section PageSampleUpnpDevice_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleUpnpDevice_SectionHowToExecute Execution Procedure
 *  Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleUpnpDevice_SectionDetail Detail
 *  Here is the sequence of this sample program.
 *
 *  - Sends a request to use the network to the NIFM library
 *  - Initializes the socket library
 *  - The libupnp library is initialized with the host network interface to use for sending and receiving multicast traffic
 *  - Initializes a "virtual filesystem" to support the libupnp "web server".
 *    Enables the libupnp "web server" functionality.
 *  - The Sample then creates the "GamePlayer" object.
 *  - The GamePlayer object will search the local LAN for UPnP devices which support the "Nintendo Game"
 *    deviceType "urn:schemas-upnp-org:device:nintendogame:1.".  This UPnP device and service discovery
 *    step uses libupnp.
 *  - If after 5 seconds no currently running Nintendo Game device is discovered on the local LAN the
 *    Sample will create a GameState object to host the Nintendo Game from this Nintendo Switch.
 *  - The GameState object will initialize the state of the Nintendo Game, in this case it will set the
 *    initial color of each square box, and will register itself to the LAN using libupnp.
 *  - The GamePlayer will then "discover" the newly running GameState using libupnp.
 *  - The Sample then creates the GameUI object.
 *  - The GameUI.MainLoop() method is invoked and the Nintendo Game UI is displayed on the Nintendo Switch
 *    screen.  (For instructions on how to use the Game User Interface -- See above).
 *  - When the user touches/selectes "E X I T" the GameUI will terminate.
 *  - The GamePlayer object is Finalized
 *  - (If applicable) - The GameState is Finalized
 *  - NIFM and nn::socket APIs are Finalized.
 *  - Sample finishes.
 *
 *  Please enable "#define DETAILED_LIBUPNP_DEBUGGING" if you are interested in seeing debugging output from libupnp.
 *  The libupnp library supports the displaying of detailed debugging information to NN_LOG.  By default only CRITICAL
 *  error information from the libupnp library is displayed.
 *
 *  Please invoke the sample with the "-interface <Ipv4 Address>" command line option if you are interested in having the
 *  sample speak only to itself, without accessing the LAN, using the loopback netwoerk interface.  When defined the sample
 *  will initialize the libupnp library on the loopback network interface.  All multicast and unicast network traffic will
 *  be routed through the loopback interface.
 *
 * @subsection PageSampleUpnpDevice_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  UpnpDevice.cpp
 *  @includelineno UpnpDevice.cpp
 *
 * @subsection PageSampleUpnpDevice_SectionSampleDetail Sample Program Description
 *  This application renders a user interface that the user interacts with.
 */

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
// #define DETAILED_LIBUPNP_DEBUGGING

namespace nns {
namespace libupnp {

// Constants
const int                           TimeoutOnNicConfigurationInSec = 15;
const uint16_t                      UpnpServerPort = 49152;

// Socket memory
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

// Nifm values
struct in_addr                      g_NifmIpv4Addr = { 0 };
struct in_addr                      g_NifmIpv4Mask = { 0 };
struct in_addr                      g_NifmIpv4Gateway = { 0 };
struct in_addr                      g_NifmIpv4Dns1 = { 0 };
struct in_addr                      g_NifmIpv4Dns2 = { 0 };
bool                                g_IsNetworkInitialized = false;

// UPNP Initialization
struct UpnpVirtualDirCallbacks      g_Callbacks;
nn::libupnp::VirtualDir             g_VirtualDirectory;
bool                                g_IsUpnpInitialized = false;

// UPNP Device Game State
UpnpDeviceGameState                 g_UpnpDeviceGameState;
bool                                g_IsUpnpDeviceGameStateInitialized = false;

// UPNP Device Game Player
UpnpDeviceGamePlayer                g_UpnpDeviceGamePlayer;
bool                                g_IsUpnpDeviceGamePlayerInitialized = false;

// UPNP Device Game User Interface
UpnpDeviceGameUI                    g_UpnpGameUI;

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
int SetupNetwork() NN_NOEXCEPT
{
    int            ret = 0;
    nn::Result     result = nn::ResultSuccess();
    uint32_t       timeoutSec = 0;
    bool           isSocketLibInitialized = false;
    bool           isNifmLibInitialized = false;

    do
    {
        // If already initialized
        if (g_IsNetworkInitialized == true)
        {
            NN_LOG("Network is already initialized!\n");
            break;
        }

        // Initialize NIFM
        result = nn::nifm::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed calling: 'nn::nifm::Initialize()'\n");
            ret = -1;
            break;
        }

        // Submit the NIFM Network Request
        nn::nifm::SubmitNetworkRequest();

        // Wait for NIFM to go online
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
            {
                NN_LOG("Failed to setup NIFM network interface.\n");
                ret = -1;
                break;
            }
        }

        // Did NIFM request timeout?
        if (ret < 0)
        {
            break;  // Yes
        }

        // Is the NIFM Network Available?
        if(nn::nifm::IsNetworkAvailable() == 0)
        {
            NN_LOG("Network is not available.\n");
            ret = -1;
            break;
        }
        isNifmLibInitialized = true;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &g_NifmIpv4Addr, &g_NifmIpv4Mask, &g_NifmIpv4Gateway, &g_NifmIpv4Dns1, &g_NifmIpv4Dns2 );

        NN_LOG( "Network Interface is UP!\n" );
        NN_LOG( "NIFM Address Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Addr ));
        NN_LOG( "NIFM Network Mask Ipv4 : %s\n", nn::socket::InetNtoa( g_NifmIpv4Mask ));
        NN_LOG( "NIFM Gateway Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Gateway));
        NN_LOG( "NIFM DNS1 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns1 ));
        NN_LOG( "NIFM DNS2 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns2 ));

        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            ret = -1;
            break;
        }
        isSocketLibInitialized = true;

        // Successfully initialized the network
        g_IsNetworkInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // If failed - disable network
    if (ret != 0)
    {
        if (isSocketLibInitialized == true)
        {
            nn::socket::Finalize();
            isSocketLibInitialized  = false;
            ret = -1;
        }

        if (isNifmLibInitialized == true)
        {
            nn::nifm::CancelNetworkRequest();
            isNifmLibInitialized = false;
            ret = -1;
        }

        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Finalize Network
// ------------------------------------------------------------------------------------------------
int FinalizeNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    int            ret = 0;

    if (g_IsNetworkInitialized == true)
    {
        result = nn::socket::Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Failed calling 'nn::socket::Finalize()'!\n");
        }

        nn::nifm::CancelNetworkRequest();
        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Create libupnp virtual filsystem
// ------------------------------------------------------------------------------------------------
int MakeVirtualFileSystem() NN_NOEXCEPT
{
    int rc = 0;

    g_Callbacks.get_info = &g_VirtualDirectory.GetInfo;
    g_Callbacks.open     = &g_VirtualDirectory.Open;
    g_Callbacks.read     = &g_VirtualDirectory.Read;
    g_Callbacks.write    = &g_VirtualDirectory.Write;
    g_Callbacks.seek     = &g_VirtualDirectory.Seek;
    g_Callbacks.close    = &g_VirtualDirectory.Close;

    rc = UpnpSetVirtualDirCallbacks(&g_Callbacks);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpSetVirtualDirCallbacks(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpEnableWebserver(1);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpEnableWebserver(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpAddVirtualDir( "/" );
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpAddVirtualDir(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    return(0);
}

// ------------------------------------------------------------------------------------------------
// Parse Arguments
// ------------------------------------------------------------------------------------------------
bool ParseArgs(char** pInterface) NN_NOEXCEPT
{
    size_t        ifaceSize = 0;
    int           options = 0;
    int           argc = nn::os::GetHostArgc();
    char**        argv = nn::os::GetHostArgv();
    const char*   pChoosenInterface = nullptr;
    bool          ret = true;

    // Initialize passed variables
    *pInterface = nullptr;

    do
    {
        // Look for a passed command line option
        for(options = 0; options < argc; options++)
        {
            if(strcmp(argv[options], "-interface" ) == 0)  // Optional: Network Interface
            {
                options++;
                if(options >= argc)
                {
                    NN_LOG( "Arg: -interface requires an additional option, which is an IPv4 address assigned to this host\n");
                    ret = false;
                    break;
                }

                // Save choosen network interface
                pChoosenInterface = argv[options];
                break;
            }
        }

        // If parsing options failed..
        if (ret == false)
        {
            break;
        }

        // If NO network interface specified
        if (pChoosenInterface == nullptr)
        {
            break;
        }

        // Figure out size of the network Interface
        ifaceSize = strlen(pChoosenInterface) + 1;

        // Allocate Memory to hold network Interface
        *pInterface = new char[ifaceSize];
        if (*pInterface == nullptr)
        {
            NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pChoosenInterface, ifaceSize);
            ret = false;
            break;
        }

        // Copy the network Interface
        memset(*pInterface, 0, ifaceSize);
        memcpy(*pInterface, pChoosenInterface, strlen(pChoosenInterface));

    } while (NN_STATIC_CONDITION(false));

    if (ret == false)
    {
        if (*pInterface != nullptr)
        {
            delete [] *pInterface;
            *pInterface = nullptr;
        }
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Main Loop
// ------------------------------------------------------------------------------------------------
void mainLoop() NN_NOEXCEPT
{
    nn::Result   result;
    int          rc = UPNP_E_SUCCESS;
    int          idx = 0;
    int          len = 0;
    char*        pNetworkIface;
    char*        pTmpNetworkIface = nullptr;
    char*        pUpnpIpAddress = nullptr;
    uint16_t     upnpPort = 0;
    bool         foundGame = false;

    do
    {
       // Determine if caller specified a network interface
       if (ParseArgs(&pNetworkIface) == false)
        {
            NN_LOG("\nError: Invalid arguments.\n");
            NN_LOG("\nUsage:\n");
            NN_LOG("  -interface <ipv4 address> - Network interface to initialize libupnp on. (optional)\n");
            break;
        }

        // Setup network interface
        NN_LOG ( "Initializing Network Interface ... \n");
        rc = SetupNetwork();
        if (rc != 0)
        {
            NN_LOG("Failed to enable network interface\n");
            break;
        }

        // If caller (DID NOT) specify a network interface..
        if (pNetworkIface == nullptr)
        {
            // Turn the Nifm negotiated IPv4 Address into ASCII
            pTmpNetworkIface = nn::socket::InetNtoa(g_NifmIpv4Addr);
            if (pTmpNetworkIface == nullptr)
            {
                NN_LOG( "Failed to translate IPv4 address: [0x%08x] into an ASCII IP Address\n", nn::socket::InetNtohl(g_NifmIpv4Addr.s_addr));
                break;
            }

            // Figure out size of the network Interface
            len = strlen(pTmpNetworkIface) + 1;

            // Allocate Memory to hold network Interface
            pNetworkIface = new char[len];
            if (pNetworkIface == nullptr)
            {
                NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pTmpNetworkIface, len);
                break;
            }

            // Copy the network Interface
            memset(pNetworkIface, 0, len);
            memcpy(pNetworkIface, pTmpNetworkIface, strlen(pTmpNetworkIface));

            NN_LOG( "Sample will use [Nifm] assigned network interface [%s]\n", pNetworkIface);
        }
        else
        {
            NN_LOG( "Sample will use [Command Line] assigned network interface [%s]\n", pNetworkIface);
        }

    // libupnp debugging flags:  UPNP_CRITICAL, UPNP_PACKET, UPNP_INFO, UPNP_ALL
#ifdef DETAILED_LIBUPNP_DEBUGGING
        UpnpSetLogLevel(UPNP_ALL);
#else
        UpnpSetLogLevel(UPNP_CRITICAL);
#endif

        NN_LOG( "Initializing libupnp: (Requesting) IPv4 Network Interface Address:Port: [%s:%d]\n", pNetworkIface, UpnpServerPort);

        rc = UpnpInit(pNetworkIface, UpnpServerPort);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpInit(): %d/%s", rc, UpnpGetErrorMessage(rc));
            break;
        }
        g_IsUpnpInitialized = true;

        pUpnpIpAddress = UpnpGetServerIpAddress();
        if (pUpnpIpAddress == nullptr)
        {
            NN_LOG( "UpnpGetServerIpAddress() returned a (NULL) IPv4 Address - Can't continue\n");
            break;
        }

        upnpPort = UpnpGetServerPort();
        if (upnpPort < 1)
        {
            NN_LOG( "UpnpGetServerPort() returned a (NULL) Port - Can't continue\n");
            break;
        }

        NN_LOG("Successfully Initialized libupnp: (Actual) IPv4 Network Interface Address:Port: [%s:%d]\n", pUpnpIpAddress, upnpPort);

        rc = MakeVirtualFileSystem();
        if(rc != 0)
        {
            NN_LOG ("** ERROR: Failed calling MakeFileSystem()\n" );
            break;
        }

        // Initialize Nintendo Player State
        rc = g_UpnpDeviceGamePlayer.Start();
        if (rc != 0)
        {
            NN_LOG ("** ERROR: Failed to call g_UpnpDeviceGamePlayer.Start()\n" );
            break;
        }
        g_IsUpnpDeviceGamePlayerInitialized = false;

        // Search for an already running Nintendo Game on the local LAN
        g_UpnpDeviceGamePlayer.AddMessage( "[UPnP Discovery] - Searching the Local LAN for a running Nintendo Game\n");
        for(idx = 0; idx < g_UpnpDeviceGamePlayer.UpnpSearchTimeout; idx++)
        {
            if (g_UpnpDeviceGamePlayer.IsNintendoGameDiscovered() == true)
            {
                NN_LOG( "Nintendo Game Discovered!\n");
                g_UpnpDeviceGamePlayer.AddMessage( "[UPnP Discovery] - A Running Nintendo Game was discovered on the local LAN!\n");
                foundGame = true;
                break;
            }

            NN_LOG( "Discovering Nintendo Game using UPnP... Please wait\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        // If Nintendo Game was (NOT) found - Start our own Nintendo Game
        if (foundGame == false)
        {
            NN_LOG( "A running Nintendo Game (WAS NOT) discovered on local LAN!\n");
            g_UpnpDeviceGamePlayer.AddMessage( "[UPnP Discovery] - (DID NOT) discover a Nintendo Game on the local Lan!\n");
            g_UpnpDeviceGamePlayer.AddMessage( "[Sample] Starting a new [Nintendo Game State] to support this Lan!\n");

            // Initialize Nintendo Game State
            rc = g_UpnpDeviceGameState.Start(g_VirtualDirectory);
            if (rc != 0)
            {
                NN_LOG ("** ERROR: Failed to call g_UpnpDeviceGameState.Start()\n" );
                break;
            }
            g_IsUpnpDeviceGameStateInitialized = true;

            // Search for Nintendo Game State on local LAN
            foundGame = false;
            for(idx = 0; idx < g_UpnpDeviceGamePlayer.UpnpSearchTimeout; idx++)
            {
                if (g_UpnpDeviceGamePlayer.IsNintendoGameDiscovered() == true)
                {
                    NN_LOG( "Nintendo Game Discovered!\n");
                    foundGame = true;
                    break;
                }

                NN_LOG( "Discovering Nintendo Game using UPnP... Please wait\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            if (foundGame == false)
            {
                NN_LOG("Did not find Nintendo Game - Giving up!\n");
                break;
            }
        }

        NN_LOG("Nintendo Game Found - Starting as a Player\n");
        g_UpnpDeviceGamePlayer.AddMessage( "[UPnP Discovery] - Nintendo Game Discovered on local LAN!\n");

        rc = g_UpnpDeviceGamePlayer.Subscribe();
        if (rc != 0)
        {
            NN_LOG("Failed to subscribe to the Nintendo Game\n");
            break;
        }

        {
            NN_LOG("***************************** GAME UI STARTING HERE! ********************************\n");

            g_UpnpGameUI.MainLoop(&g_UpnpDeviceGamePlayer);

            NN_LOG("******************************* GAME UI ENDS HERE! **********************************\n");
        }

        rc = g_UpnpDeviceGamePlayer.UnSubscribe();
        if (rc != 0)
        {
            NN_LOG("Failed calling 'UnSubscribe()' from the Nintendo Game\n");
            break;
        }
        NN_LOG("Back from 'UnSubscribe()'\n");

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG ("Exiting!\n");

    if (g_IsUpnpDeviceGamePlayerInitialized == true)
    {
        rc = g_UpnpDeviceGamePlayer.Stop();
        if (rc != 0)
        {
            NN_LOG("** ERROR: Failed to call g_UpnpDeviceGamePlayer.Stop()\n" );
        }
        g_IsUpnpDeviceGamePlayerInitialized = false;
    }

    if (g_IsUpnpDeviceGameStateInitialized == true)
    {
        rc = g_UpnpDeviceGameState.Stop();
        if (rc != 0)
        {
            NN_LOG("** ERROR: Failed to call g_UpnpDeviceGameState.Stop()\n" );
        }
        g_IsUpnpDeviceGameStateInitialized = false;
    }

    if (g_IsUpnpInitialized == true)
    {
        (void) UpnpFinish();
        g_IsUpnpInitialized = false;
    }

    if (pNetworkIface != nullptr)
    {
        delete [] pNetworkIface;
        pNetworkIface = nullptr;
    }

    FinalizeNetwork();

}   // NOLINT(impl/function_size)

}}  // Namespace nns / libupnp

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup() NN_NOEXCEPT
{
    uintptr_t address = 0;
    nn::Result result = nn::ResultSuccess();

    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 96 * 1024 * 1024;
    result = nn::os::SetMemoryHeapSize( MemoryHeapSize );
    NN_ASSERT(result.IsSuccess());

    // Allcate memory block from the heap for malloc
    result = nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);
    NN_ASSERT(result.IsSuccess());

    // Set memory chunk for malloc
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    nns::libupnp::mainLoop();
}
