﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include "TaskRunner.h"

namespace nns { namespace ldn { namespace detail
{
    SequentialVoidTaskRunnerBase::SequentialVoidTaskRunnerBase(
        void* stack, size_t stackSize, int priority,
        IVoidTask** tasks, int capacity) NN_NOEXCEPT
        : m_RunEvent(nn::os::EventClearMode_AutoClear),
          m_StateChangeEvent(nn::os::EventClearMode_AutoClear),
          m_TaskMutex(false),
          m_Tasks(tasks),
          m_State(TaskRunnerState_Paused),
          m_TaskCapacity(static_cast<int16_t>(capacity)),
          m_TaskCount(0),
          m_TaskIndex(0),
          m_IsCanceled(false),
          m_IsDestroyed(false)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_Thread, Thread, this, stack, stackSize, priority));
        nn::os::StartThread(&m_Thread);
    }

    SequentialVoidTaskRunnerBase::~SequentialVoidTaskRunnerBase() NN_NOEXCEPT
    {
        m_IsDestroyed = true;
        Cancel();
        m_RunEvent.Signal();
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
    }

    void SequentialVoidTaskRunnerBase::AddTask(IVoidTask* pTask) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTask);
        NN_ASSERT_EQUAL(m_State, TaskRunnerState_Paused);
        NN_ASSERT(m_TaskCount < m_TaskCapacity);
        NN_ASSERT_EQUAL(pTask->GetState(), TaskState_None);
        m_Tasks[m_TaskCount] = pTask;
        ++m_TaskCount;
    }

    void SequentialVoidTaskRunnerBase::Clear() NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(m_State, TaskRunnerState_Paused);
        m_TaskCount = 0;
        m_TaskIndex = 0;
    }

    void SequentialVoidTaskRunnerBase::Run() NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(m_State, TaskRunnerState_Paused);
        NN_ASSERT(0 < m_TaskCount);
        m_IsCanceled = false;
        SetState(TaskRunnerState_Running);
        m_RunEvent.Signal();
    }

    void SequentialVoidTaskRunnerBase::Cancel() NN_NOEXCEPT
    {
        m_IsCanceled = true;
        std::lock_guard<nn::os::Mutex> lock(m_TaskMutex);
        if (m_TaskIndex < m_TaskCount)
        {
            m_Tasks[m_TaskIndex]->Cancel();
        }
    }

    TaskRunnerState SequentialVoidTaskRunnerBase::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    int SequentialVoidTaskRunnerBase::GetTaskCount() const NN_NOEXCEPT
    {
        return m_TaskCount;
    }

    IVoidTask* SequentialVoidTaskRunnerBase::GetTask(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_TaskCount);
        return m_Tasks[index];
    }

    const IVoidTask* SequentialVoidTaskRunnerBase::GetTask(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_TaskCount);
        return m_Tasks[index];
    }

    void SequentialVoidTaskRunnerBase::SetState(TaskRunnerState newState) NN_NOEXCEPT
    {
        if (m_State != newState)
        {
            m_State = newState;
            m_StateChangeEvent.Signal();
        }
    }

    void SequentialVoidTaskRunnerBase::Thread(void* pArg) NN_NOEXCEPT
    {
        nn::os::SetThreadName(nn::os::GetCurrentThread(), "TaskRunner");
        NN_ASSERT_NOT_NULL(pArg);
        auto& runner = *static_cast<SequentialVoidTaskRunnerBase*>(pArg);
        for (;;)
        {
            // 実行が開始されるまで待機します。
            runner.m_RunEvent.Wait();
            if (runner.m_IsDestroyed)
            {
                break;
            }

            // 全てのタスクを実行します。
            while (runner.m_TaskIndex < runner.m_TaskCount && !runner.m_IsCanceled)
            {
                auto& task = *runner.m_Tasks[runner.m_TaskIndex];
                task.Run();
                {
                    std::lock_guard<nn::os::Mutex> lock(runner.m_TaskMutex);
                    ++runner.m_TaskIndex;
                }
            }

            // 実行を中断します。
            runner.SetState(TaskRunnerState_Paused);
        }
    }

    nn::os::Event& SequentialVoidTaskRunnerBase::GetStateChangeEvent() NN_NOEXCEPT
    {
        return m_StateChangeEvent;
    }

}}} // namespace nns::ldn::detail
