﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include "Npad.h"

namespace nns { namespace ldn { namespace
{
    nn::hid::NpadIdType GetNpadIdType(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, NpadCountMax);
        return static_cast<nn::hid::NpadIdType>(index);
    }

    uint16_t SetIf(uint16_t state, Button button, bool cond)
    {
        return static_cast<uint16_t>(cond ? state | button : state);
    }

    const nn::hid::NpadIdType SupportedNpadIdTypes[] = {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8,
        nn::hid::NpadId::Handheld,
    };

}}} // namespace nns::ldn::<unnamed>

namespace nns { namespace ldn
{
    Npad::Npad() NN_NOEXCEPT
        : Npad(0)
    {
    }

    Npad::Npad(int index) NN_NOEXCEPT
        : m_PreviousButtonState(0),
          m_Index(static_cast<int8_t>(index)),
          m_IsInitialized(false)
    {
        NN_ASSERT_RANGE(index, 0, NpadCountMax);
    }

    Npad::~Npad() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            Finalize();
        }
    }

    void Npad::Initialize() NN_NOEXCEPT
    {
        NN_ASSERT(!m_IsInitialized, "debug pad has already been initialized.");
        m_IsInitialized = true;
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadIdType(
            SupportedNpadIdTypes, sizeof(SupportedNpadIdTypes) / sizeof(nn::hid::NpadIdType));
        nn::hid::SetSupportedNpadStyleSet(
            nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleJoyDual::Mask |
            nn::hid::NpadStyleHandheld::Mask);
    }

    void Npad::Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized, "debug pad has yet to been initialized.");
        m_IsInitialized = false;
    }

    void Npad::Update(PadState* pOutState) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized, "debug pad has yet to been initialized.");
        NN_ASSERT_NOT_NULL(pOutState);

        // NPAD の入力を取得します。
        auto npadId = GetNpadIdType(m_Index);
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadId);
        nn::hid::NpadButtonSet buttons;
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, npadId);
            buttons = state.buttons;
        }
        else if (style.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState state;
            nn::hid::GetNpadState(&state, npadId);
            buttons = state.buttons;
        }
        else
        {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, npadId);
            buttons = state.buttons;
        }

        // 先頭のコントローラには HandHeld の入力もマージします。
        if (m_Index == 0)
        {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            buttons |= state.buttons;
        }

        // 独自の PadState に変換します。
        uint16_t down = 0;
        down = SetIf(down, Button_A, buttons.Test<nn::hid::NpadJoyButton::A>());
        down = SetIf(down, Button_B, buttons.Test<nn::hid::NpadJoyButton::B>());
        down = SetIf(down, Button_X, buttons.Test<nn::hid::NpadJoyButton::X>());
        down = SetIf(down, Button_Y, buttons.Test<nn::hid::NpadJoyButton::Y>());
        down = SetIf(down, Button_L, buttons.Test<nn::hid::NpadJoyButton::L>());
        down = SetIf(down, Button_R, buttons.Test<nn::hid::NpadJoyButton::R>());
        down = SetIf(down, Button_ZL, buttons.Test<nn::hid::NpadJoyButton::ZL>());
        down = SetIf(down, Button_ZR, buttons.Test<nn::hid::NpadJoyButton::ZR>());
        down = SetIf(down, Button_Plus, buttons.Test<nn::hid::NpadJoyButton::Plus>());
        down = SetIf(down, Button_Minus, buttons.Test<nn::hid::NpadJoyButton::Minus>());
        down = SetIf(down, Button_Left, buttons.Test<nn::hid::NpadJoyButton::Left>());
        down = SetIf(down, Button_Up, buttons.Test<nn::hid::NpadJoyButton::Up>());
        down = SetIf(down, Button_Right, buttons.Test<nn::hid::NpadJoyButton::Right>());
        down = SetIf(down, Button_Down, buttons.Test<nn::hid::NpadJoyButton::Down>());
        pOutState->button.down = down;
        pOutState->button.trigger = (~m_PreviousButtonState) & down;
        m_PreviousButtonState = down;
    }

    void Npad::SetIndex(int index) NN_NOEXCEPT
    {
        m_Index = static_cast<int8_t>(index);
        m_PreviousButtonState = 0;
    }

}} // namespace nns::ldn
