﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LblSettingsForVrMode.cpp,PageSampleLblSettingsForVrMode}
 *
 * @brief
 *  VR モード中の本体画面の明るさ設定のサンプルプログラム
 */

/**
 * @page PageSampleLblSettingsForVrMode VR モード中の本体画面の明るさ設定
 * @tableofcontents
 *
 * @brief
 *  VR モード中の本体画面の明るさ設定のサンプルプログラムの解説です。
 *
 * @section PageSampleLblSettingsForVrMode_SectionBrief 概要
 *  ここでは、VR モード中の本体画面の明るさ設定のサンプルプログラムの説明をします。
 *
 * @section PageSampleLblSettingsForVrMode_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/LblSettingsForVrMode Samples/Sources/Applications/LblSettingsForVrMode @endlink 以下にあります。
 *
 * @section PageSampleLblSettingsForVrMode_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleLblSettingsForVrMode_SectionHowToOperate 操作方法
 *  メインループ中は、無線接続したジョイコンまたはデバッグパッドにより以下の操作が可能です。
 *
 *  | 操作                 | 機能 |
 *  | -------------------- | ---- |
 *  | 左キー・左スティック | VR モード中の本体画面の明るさを下げる |
 *  | L/ZL                 | VR モード中の本体画面の明るさを *高速に* 下げる |
 *  | 右キー・右スティック | VR モード中の本体画面の明るさを上げる |
 *  | R/ZR                 | VR モード中の本体画面の明るさを *高速に* 上げる |
 *  | Y                    | 現在の明るさを本体設定に保存し、永続化する |
 *  | A                    | 通常モードと VR モードを切り替える |
 *  | マイナス (-)         | サンプルプログラムを終了する |
 *
 * @section PageSampleLblSettingsForVrMode_SectionPrecaution 注意事項
 *  特になし
 *
 * @section PageSampleLblSettingsForVrMode_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドして実行してください。
 *
 * @section PageSampleLblSettingsForVrMode_SectionDetail 解説
 *
 * @subsection PageSampleLblSettingsForVrMode_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  LblSettingsForVrMode.cpp
 *  @includelineno LblSettingsForVrMode.cpp
 *
 * @subsection PageSampleLblSettingsForVrMode_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - 初期化
 *    - lbl ライブラリを初期化する
 *    - pl ライブラリを使用して VR モードに入れる
 *  - メインループ
 *    - 「操作方法」の通り、コントローラ入力に応じた処理を行う
 *  - 終了
 *    - pl ライブラリを使用して VR モードを抜ける
 *    - lbl ライブラリを終了する
 *
 *  なお、VR モードで動作中に HOME ボタン等でメニュー画面に遷移する場合、
 *  システムからプログラムに対して「VR モードの終了要求」が発生します。
 *  これは、プログラムに対する nn::oe::MessageRequestToEndVrMode メッセージで
 *  通知されます。プログラムはこの要求に対して、内部状態を通常モードに
 *  戻した上で、nn::pl::EndVrMode() を発行するようにして下さい。
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nns/gfxLog.h>
#include <nns/nns_Log.h>
#include <nns/hid.h>

#include <nn/pl/pl_VrModeApi.private.h>

#include <nn/lbl/lbl_Lib.h>
#include <nn/lbl/lbl_SettingsForVrMode.h>
#include <nn/lbl/lbl_SettingsSave.h>

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    // lbl ライブラリを初期化します。
    nn::lbl::Initialize();

    // pl ライブラリを使用して VR モードに入れます。
    nn::pl::BeginVrMode();
    bool isInVrMode = true;

    // ログ描画用スレッドのコアを指定します。
    nns::gfxLog::SetThreadCoreNumber(0);

    NNS_LOG("=============================================\n");
    NNS_LOG("Left key or stick  : Decrease brightness\n");
    NNS_LOG("L/ZL               : Decrease brightness (fast)\n");
    NNS_LOG("Right key or stick : Increase brightness\n");
    NNS_LOG("R/ZR               : Increase brightness (fast)\n");
    NNS_LOG("Y                  : Save current setting\n");
    NNS_LOG("A                  : Enter / Exit VR Mode\n");
    NNS_LOG("Minus              : Quit program\n");
    NNS_LOG("=============================================\n");

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);
    auto& gamePad = *reinterpret_cast<nns::hid::GamePad*>(controllerManager.GetController(nns::hid::ControllerId_GamePad, 0));
    auto& debugPad = *reinterpret_cast<nns::hid::DebugPad*>(controllerManager.GetController(nns::hid::ControllerId_DebugPad, 0));

    auto currentBrightnessInVr = nn::lbl::GetCurrentBrightnessSettingForVrMode();
    NNS_LOG("Current brightness: %f\n", currentBrightnessInVr);

    while ( NN_STATIC_CONDITION(true) )
    {
        controllerManager.Update();

        const auto SmallDecrementInputMask = nns::hid::Button::Left::Mask | nns::hid::Button::LsLeft::Mask | nns::hid::Button::RsLeft::Mask;
        const auto BigDecrementInputMask = nns::hid::Button::L::Mask | nns::hid::Button::ZL::Mask;
        const auto SmallIncrementInputMask = nns::hid::Button::Right::Mask | nns::hid::Button::LsRight::Mask | nns::hid::Button::RsRight::Mask;
        const auto BigIncrementInputMask = nns::hid::Button::R::Mask | nns::hid::Button::ZR::Mask;
        const auto SaveInputMask = nns::hid::Button::Y::Mask;
        const auto SwitchVrModeInputMask = nns::hid::Button::A::Mask;
        const auto EndDemoInputMask = nns::hid::Button::Minus::Mask;

        const float SmallStep = 1.0f / 256.0f;
        const float BigStep = 1.0f / 32.0f;

        auto nextBrightnessInVr = currentBrightnessInVr;
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtons(SmallDecrementInputMask)) ||
             (debugPad.IsConnected() && debugPad.HasAnyButtons(SmallDecrementInputMask)) )
        {
            // 明るさを下げる
            nextBrightnessInVr -= SmallStep;
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtons(BigDecrementInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtons(BigDecrementInputMask)) )
        {
            // 明るさを大きく下げる
            nextBrightnessInVr -= BigStep;
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtons(SmallIncrementInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtons(SmallIncrementInputMask)) )
        {
            // 明るさを上げる
            nextBrightnessInVr += SmallStep;
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtons(BigIncrementInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtons(BigIncrementInputMask)) )
        {
            // 明るさを大きく上げる
            nextBrightnessInVr += BigStep;
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(SaveInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(SaveInputMask)) )
        {
            // 現在の設定を本体設定に保存し、永続化する
            nn::lbl::SaveCurrentSetting();
            NNS_LOG("Settings was saved.\n");
        }
        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(SwitchVrModeInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(SwitchVrModeInputMask)) )
        {
            // VR モードを切り替える
            isInVrMode = !isInVrMode;
            if ( isInVrMode )
            {
                nn::pl::BeginVrMode();
                NNS_LOG("VR Mode Begin.\n");
            }
            else
            {
                nn::pl::EndVrMode();
                NNS_LOG("VR Mode End.\n");
            }
        }
        nn::oe::Message message;
        if ( nn::oe::TryPopNotificationMessage( &message ) )
        {
            if ( nn::oe::MessageRequestToEndVrMode == message && isInVrMode )
            {
                // VR モードを解除する
                isInVrMode = false;
                nn::pl::EndVrMode();
                NNS_LOG("VR Mode End because of system request.\n");
            }
        }

        if ( (gamePad.IsConnected() && gamePad.HasAnyButtonsDown(EndDemoInputMask)) ||
            (debugPad.IsConnected() && debugPad.HasAnyButtonsDown(EndDemoInputMask)) )
        {
            // サンプルプログラムを終了する
            break;
        }

        // 有効な値域に丸める (0.0f <= 有効値 <= 1.0f)
        nextBrightnessInVr =
            (nextBrightnessInVr < 0.0f) ? 0.0f :
            (nextBrightnessInVr > 1.0f) ? 1.0f : nextBrightnessInVr;

        if ( nextBrightnessInVr != currentBrightnessInVr )
        {
            nn::lbl::SetCurrentBrightnessSettingForVrMode(nextBrightnessInVr);
            currentBrightnessInVr = nextBrightnessInVr;
            NNS_LOG("Current brightness: %f\n", currentBrightnessInVr);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // VR モードを終了。
    // なお、本関数を呼ばずに HOME メニュー等から強制的にアプリを終了した場合も自動的に VR モードは終了します。
    nn::pl::EndVrMode();

    // lbl ライブラリを終了。
    nn::lbl::Finalize();

    // プログラムの終了
    NNS_LOG("\nEnd of Demo.\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
} // NOLINT(impl/function_size)
