﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LblAutoBacklightControl.cpp,PageSampleLblAutoBacklightControl}
 *
 * @brief
 *  自動輝度調整のサンプルプログラム
 */

/**
 * @page PageSampleLblAutoBacklightControl 自動輝度調整
 * @tableofcontents
 *
 * @brief
 *  自動輝度調整機能のサンプルプログラムの解説です。
 *
 * @section PageSampleLblAutoBacklightControl_SectionBrief 概要
 *  ここでは、自動輝度調整機能を使ったサンプルプログラムの説明をします。
 *
 *  自動輝度調整機能の使い方については、LBL の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleLblAutoBacklightControl_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/LblAutoBacklightControl Samples/Sources/Applications/LblAutoBacklightControl @endlink 以下にあります。
 *
 * @section PageSampleLblAutoBacklightControl_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleLblAutoBacklightControl_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleLblAutoBacklightControl_SectionPrecaution 注意事項
 *  このデモではLCDのバックライト出力を変化させますが、LCDには何も表示されません。
 *  デモを終了させるまで明るさセンサーの値に応じて延々とバックライト出力が変化します。
 *  バックライトが暗いときにLCDがチラつくときがありますが、
 *  これはLCDの性質によるものです(フリッカーと呼ばれます)。
 *  製品で本ライブラリを用いる場合は、適切な写像の係数を割り当てて
 *  フリッカーを防ぐ必要があります。
 *
 * @section PageSampleLblAutoBacklightControl_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLblAutoBacklightControl_SectionDetail 解説
 *
 * @subsection PageSampleLblAutoBacklightControl_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  LblAutoBacklightControl.cpp
 *  @includelineno LblAutoBacklightControl.cpp
 *
 * @subsection PageSampleLblAutoBacklightControl_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - LBLライブラリを初期化する。
 *  - 自動輝度調整をonにする。
 *  - 簡単な例として、バックライト出力の写像と明るさセンサー値の写像を一次関数とする。
 *  - 現在の輝度設定値の最大値を割り当てる。
 *  - バックライト出力の変化を観察するために無限ループに入る。
 *
 *  本サンプルプログラムを正しく実行できた場合、
 *  LCDのバックライトが変化しながら、ログが表示され続けます。
 *  明るさセンサーの値の取得とバックライト出力への適用は
 *  輝度制御プロセスで行われます。
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/lbl/lbl.h>

const size_t HeapSize = 64 * 1024 * 1024;
const size_t BlockSize = 32 * 1024 * 1024;

extern "C" void nninitStartup()
{
    NN_ASSERT(nn::os::SetMemoryHeapSize(HeapSize).IsSuccess(), "SetMemoryHeapSize failed.\n");

    uintptr_t address;
    nn::Result result = nn::os::AllocateMemoryBlock(&address, BlockSize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), BlockSize);
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    // 初期化する段階でsettingsに保存された設定値が読み込まれる。
    nn::lbl::Initialize();

    // 自動輝度調整をonにする。
    nn::lbl::EnableAutoBrightnessControl();
    bool autoControlEnabled = nn::lbl::IsAutoBrightnessControlEnabled();
    NN_ASSERT(autoControlEnabled, "Auto brightness control is not working.");
    NN_LOG("Auto brightness control is working.\n");

    // 自動輝度調整がonのとき、現在の輝度設定値は明るさセンサー値に影響する。
    // 0.0f だと暗めに、1.0f だと明るめに自動調整される。
    nn::lbl::SetCurrentBrightnessSetting(1.0f);

    // 明るさセンサーの値によってバックライトの輝度が変わる。
    nn::TimeSpan timeSpan = nn::TimeSpan::FromSeconds(10);
    while (1)
    {
        NN_LOG("Apply brightness setting automatically. applied brightness:[%f]\n",
            nn::lbl::GetBrightnessSettingAppliedToBacklight());
        nn::os::SleepThread(timeSpan);
    }
}
