﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/vi.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/nn_Assert.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "GraphicsSystem.h"

const size_t MemoryPoolSize = 64 * 1024 * 1024;
const size_t MemoryPoolSizeInvisible = 32 * 1024 * 1024;

const int DescriptorPoolSlotCount = 2048;
const int DescriptorPoolStartSlot = 256;

const size_t CommandBufferCommandMemoryChunkSize = 512 * 1024;
const size_t CommandBufferControlMemoryChunkSize = 64 * 1024;

const uint32_t ScreenWidth = 1280;
const uint32_t ScreenHeight = 720;

const int SwapInterval = 1;

const char* FontFilePath =
#if defined(NN_BUILD_APISET_NX)
    "Contents:/nintendo_NTLG-DB_002_NX.bffnt";
#else
    "Contents:/nintendo_NTLG-DB_002_Linear.bffnt";
#endif

void* Allocate(size_t size) NN_NOEXCEPT
{
    return std::malloc(size);
}

void Deallocate(void* addr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    std::free(addr);
}

GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
    : m_pApplicationHeap(0)
    , m_pMemoryPool(0)
    , m_pMemoryPoolOffset(0)
    , m_pMemoryPoolCommandOffset(0)
    , m_pMemoryPoolInvisible(0)
    , m_pMemoryPoolInvisibleOffset(0)
    , m_TextureSlotCount(0)
    , m_SamplerSlotCount(0)
    , m_RenderWidth(1280)
    , m_RenderHeight(720)
    , m_DebugFontHeap(0)
    , m_pPrimitiveRenderer(0)
    , m_FrameCount(0)
{
}

GraphicsSystem::~GraphicsSystem() NN_NOEXCEPT
{
}

void GraphicsSystem::SetApplicationHeap(nn::mem::StandardAllocator* pValue) NN_NOEXCEPT
{
    m_pApplicationHeap = pValue;
}

nn::vi::NativeWindowHandle GraphicsSystem::GetNativeWindowHandle() NN_NOEXCEPT
{
    return m_NativeWindowHandle;
}

nn::gfx::Device& GraphicsSystem::GetDevice() NN_NOEXCEPT
{
    return m_Device;
}

nn::gfx::CommandBuffer& GraphicsSystem::GetCommandBuffer() NN_NOEXCEPT
{
    return m_CommandBuffer;
}

nn::gfx::util::DebugFontTextWriter& GraphicsSystem::GetDebugFont() NN_NOEXCEPT
{
    return m_DebugFont;
}

nns::gfx::PrimitiveRenderer::Renderer& GraphicsSystem::GetPrimitiveRenderer() NN_NOEXCEPT
{
    return *m_pPrimitiveRenderer;
}

uint64_t GraphicsSystem::GetFrameCount() NN_NOEXCEPT
{
    return m_FrameCount;
}

void GraphicsSystem::Initialize() NN_NOEXCEPT
{
    nn::vi::Initialize();
    this->InitializeDisplay();
    nn::gfx::Initialize();
    this->InitializeDevice();
    this->InitializeMemoryPool();
    this->InitializeMemoryPoolInvisible();
    this->InitializeSwapChain();
    this->InitializeQueue();
    this->InitializeFence();
    this->InitializeColorTargetTexture();
    this->InitializeColorTargetView();
    this->InitializeDepthStencilTexture();
    this->InitializeDepthStencilView();
    this->InitializeTextureDescriptorPool();
    this->InitializeSamplerDescriptorPool();
    this->InitializeCommandBuffer();
    this->InitializeBlendState();
    this->InitializeDepthStencilState();
    this->InitializeRasterizerState();
    this->InitializeDebugFont();
    this->InitializePrimitiveRenderer();

    m_FrameCount = 0;
}

void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    this->FinalizePrimitiveRenderer();
    this->FinalizeDebugFont();
    m_RasterizerState.Finalize(&m_Device);
    m_DepthStencilState.Finalize(&m_Device);
    this->FinalizeBlendState();
    this->ResetCommandBuffer();
    m_CommandBuffer.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_DepthStencilView.Finalize(&m_Device);
    m_DepthStencilTexture.Finalize(&m_Device);
    m_ColorTargetView.Finalize(&m_Device);
    m_ColorTargetTexture.Finalize(&m_Device);
    m_Fence.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    this->FinalizeMemoryPoolInvisible();
    this->FinalizeMemoryPool();
    m_Device.Finalize();
    nn::gfx::Finalize();
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();
}

void GraphicsSystem::SetDefaultViewportScissor() NN_NOEXCEPT
{
    nn::gfx::ViewportStateInfo viewportStateInfo;
    nn::gfx::ScissorStateInfo scissorStateInfo;

    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth(static_cast<float>(m_RenderWidth));
    viewportStateInfo.SetHeight(static_cast<float>(m_RenderHeight));
    m_CommandBuffer.SetViewports(0, 1, &viewportStateInfo);

    scissorStateInfo.SetDefault();
    scissorStateInfo.SetWidth(m_RenderWidth);
    scissorStateInfo.SetHeight(m_RenderHeight);
    m_CommandBuffer.SetScissors(0, 1, &scissorStateInfo);
}

void GraphicsSystem::BeginDraw() NN_NOEXCEPT
{
    this->ResetCommandBuffer();
    this->AddCommandMemoryToCommandBuffer(&m_CommandBuffer);
    this->AddControlMemoryToCommandBuffer(&m_CommandBuffer);

    m_CommandBuffer.Begin();

    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    m_CommandBuffer.ClearColor(
        &m_ColorTargetView,
        0.0f,
        0.0f,
        0.0f,
        0.0f,
        0);

    m_CommandBuffer.ClearDepthStencil(
        &m_DepthStencilView,
        1.0f,
        0,
        nn::gfx::DepthStencilClearMode_DepthStencil,
        0);

    nn::gfx::ColorTargetView* pColorTargetView = &m_ColorTargetView;
    m_CommandBuffer.SetRenderTargets(
        1,
        &pColorTargetView,
        &m_DepthStencilView);
    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth(static_cast<float>(m_RenderWidth));
    viewportStateInfo.SetHeight(static_cast<float>(m_RenderHeight));
    m_CommandBuffer.SetViewports(0, 1, &viewportStateInfo);

    nn::gfx::ScissorStateInfo scissorStateInfo;
    scissorStateInfo.SetDefault();
    scissorStateInfo.SetWidth(m_RenderWidth);
    scissorStateInfo.SetHeight(m_RenderHeight);
    m_CommandBuffer.SetScissors(0, 1, &scissorStateInfo);

    m_CommandBuffer.SetBlendState(&m_BlendState);
    m_CommandBuffer.SetDepthStencilState(&m_DepthStencilState);
    m_CommandBuffer.SetRasterizerState(&m_RasterizerState);

    SetDefaultViewportScissor();
    m_pPrimitiveRenderer->Update(0);
}

void GraphicsSystem::EndDraw() NN_NOEXCEPT
{
    m_CommandBuffer.End();

    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_Fence);
    m_Queue.Flush();

    m_Queue.CopyToScanBuffer(&m_SwapChain, &m_ColorTargetView);
    m_Queue.Present(&m_SwapChain, SwapInterval);
    m_Queue.Flush();

    ++m_FrameCount;
}

void GraphicsSystem::DrawDone() NN_NOEXCEPT
{
    m_CommandBuffer.End();

    m_Queue.ExecuteCommand(&m_CommandBuffer, NULL);
    m_Queue.Sync();

    m_CommandBuffer.Begin();
}

void GraphicsSystem::Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT
{
    m_Fence.Sync(timeout);
}

void GraphicsSystem::AllocateTexture(nn::gfx::Texture* pTexture, const nn::gfx::Texture::InfoType* pInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTexture);
    NN_ASSERT_NOT_NULL(pInfo);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(m_pMemoryPoolInvisibleOffset, nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, *pInfo));
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, *pInfo);
    pTexture->Initialize(&m_Device, *pInfo, &m_MemoryPoolInvisible, m_pMemoryPoolInvisibleOffset, size);
    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::AllocateBuffer(nn::gfx::Buffer* pBuffer, const nn::gfx::Buffer::InfoType* pInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pInfo);

    m_pMemoryPoolOffset = nn::util::align_up(m_pMemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(&m_Device, *pInfo));
    pBuffer->Initialize(&m_Device, *pInfo, &m_MemoryPool, m_pMemoryPoolOffset, pInfo->GetSize());
    m_pMemoryPoolOffset += pInfo->GetSize();
}

void GraphicsSystem::FreeTexture(nn::gfx::Texture* pTexture) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTexture);

    pTexture->Finalize(&m_Device);
}

void GraphicsSystem::FreeBuffer(nn::gfx::Buffer* pBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);

    pBuffer->Finalize(&m_Device);
}

void GraphicsSystem::RegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::TextureView& textureView ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int slot = m_TextureSlotCount;
    m_TextureDescriptorPool.BeginUpdate();
    m_TextureDescriptorPool.SetTextureView(slot, &textureView);
    m_TextureDescriptorPool.EndUpdate();
    m_TextureDescriptorPool.GetDescriptorSlot(pOutValue, slot);

    ++m_TextureSlotCount;
}

void GraphicsSystem::RegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::Sampler& sampler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int slot = m_SamplerSlotCount;
    m_SamplerDescriptorPool.BeginUpdate();
    m_SamplerDescriptorPool.SetSampler(slot, &sampler);
    m_SamplerDescriptorPool.EndUpdate();
    m_SamplerDescriptorPool.GetDescriptorSlot(pOutValue, slot);

    ++m_SamplerSlotCount;
}

void GraphicsSystem::UnregisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::TextureView& textureView
) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(textureView);
}

void GraphicsSystem::UnregisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::Sampler& sampler) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(sampler);
}

void GraphicsSystem::OutOfCommandMemoryEventCallback(
nn::gfx::CommandBuffer* pCommandBuffer,
const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddCommandMemoryToCommandBuffer(pCommandBuffer);
}

void GraphicsSystem::OutOfControlMemoryEventCallback(
nn::gfx::CommandBuffer* pCommandBuffer,
const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddControlMemoryToCommandBuffer(pCommandBuffer);
}

void* GraphicsSystem::AllocateFunction(size_t size,
size_t alignment,
void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    return reinterpret_cast<
        nn::mem::StandardAllocator*>(pUserData)->Allocate(size, alignment);
}

void GraphicsSystem::FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    reinterpret_cast<nn::mem::StandardAllocator*>(pUserData)->Free(ptr);
}

void GraphicsSystem::InitializeDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::CreateLayer(&m_pLayer,
        m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::SetLayerScalingMode(m_pLayer,
        nn::vi::ScalingMode_FitToLayer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::GetNativeWindow(&m_NativeWindowHandle, m_pLayer));
}

void GraphicsSystem::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();

    m_Device.Initialize(info);
}

void GraphicsSystem::InitializeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached);

    const size_t aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPool = m_pApplicationHeap->Allocate(MemoryPoolSize, aligment);

    info.SetPoolMemory(m_pMemoryPool, MemoryPoolSize);

    m_MemoryPool.Initialize(&m_Device, info);

    m_pMemoryPoolOffset = 0;

    m_pMemoryPoolCommandOffset = 0;
}

void GraphicsSystem::InitializeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
        nn::gfx::MemoryPoolProperty_GpuCached |
        nn::gfx::MemoryPoolProperty_Compressible);

    const size_t aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPoolInvisible =
        m_pApplicationHeap->Allocate(MemoryPoolSizeInvisible, aligment);

    info.SetPoolMemory(m_pMemoryPoolInvisible, MemoryPoolSizeInvisible);

    m_MemoryPoolInvisible.Initialize(&m_Device, info);

    m_pMemoryPoolInvisibleOffset = 0;
}

void GraphicsSystem::InitializeSwapChain() NN_NOEXCEPT
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(m_RenderWidth);
    info.SetHeight(m_RenderHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);

    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        m_pMemoryPoolInvisibleOffset = nn::util::align_up(
            m_pMemoryPoolInvisibleOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));

        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);

        m_SwapChain.Initialize(&m_Device,
            info,
            &m_MemoryPoolInvisible,
            m_pMemoryPoolInvisibleOffset,
            size);

        m_pMemoryPoolInvisibleOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, NULL, 0, 0);
    }
}

void GraphicsSystem::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);

    m_Queue.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();

    m_Fence.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeColorTargetTexture() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_RenderWidth);
    info.SetHeight(m_RenderHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(
        m_pMemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_ColorTargetTexture.Initialize(&m_Device,
        info,
        &m_MemoryPoolInvisible,
        m_pMemoryPoolInvisibleOffset,
        size);

    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeColorTargetView() NN_NOEXCEPT
{
    nn::gfx::ColorTargetView::InfoType info;
    info.SetDefault();
    info.SetImageDimension(nn::gfx::ImageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    info.SetTexturePtr(&m_ColorTargetTexture);

    m_ColorTargetView.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilTexture() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_RenderWidth);
    info.SetHeight(m_RenderHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(
        m_pMemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_DepthStencilTexture.Initialize(&m_Device,
        info,
        &m_MemoryPoolInvisible,
        m_pMemoryPoolInvisibleOffset,
        size);

    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeDepthStencilView() NN_NOEXCEPT
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension(nn::gfx::ImageDimension_2d);
    info.SetTexturePtr(&m_DepthStencilTexture);

    m_DepthStencilView.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(DescriptorPoolSlotCount);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_TextureDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_pMemoryPoolOffset,
        size);

    m_pMemoryPoolOffset += size;

    m_TextureSlotCount = DescriptorPoolStartSlot;
}

void GraphicsSystem::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(DescriptorPoolSlotCount);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_SamplerDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_pMemoryPoolOffset,
        size);

    m_pMemoryPoolOffset += size;

    m_SamplerSlotCount = DescriptorPoolStartSlot;
}

void GraphicsSystem::InitializeCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    m_CommandBuffer.Initialize(&m_Device, info);
    m_CommandBuffer.SetUserPtr(this);
    m_CommandBuffer.SetOutOfCommandMemoryEventCallback(
        OutOfCommandMemoryEventCallback);
    m_CommandBuffer.SetOutOfControlMemoryEventCallback(
        OutOfControlMemoryEventCallback);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    m_pMemoryPoolCommandOffset = m_pMemoryPoolOffset;
    m_pMemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
}

void GraphicsSystem::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red |
        nn::gfx::ChannelMask_Green |
        nn::gfx::ChannelMask_Blue);
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetDestinationColorBlendFactor(
        nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
    m_BlendState.SetMemory(m_pApplicationHeap->Allocate(memorySize),
        memorySize);

    m_BlendState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(true);

    m_DepthStencilState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);

    m_RasterizerState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDebugFont() NN_NOEXCEPT
{
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        &m_Device,
        info
        );
    m_DebugFontHeap = nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    m_DebugFont.Initialize(
        &m_Device,
        info,
        m_DebugFontHeap.Get(),
        debugFontHeapSize,
        NULL,
        0,
        0
        );
    m_DebugFont.SetDisplayWidth(ScreenWidth);
    m_DebugFont.SetDisplayHeight(ScreenHeight);

    m_DebugFont.SetTextureDescriptor(&m_TextureDescriptorPool, m_TextureSlotCount);
    ++m_TextureSlotCount;
    m_DebugFont.SetSamplerDescriptor(&m_SamplerDescriptorPool, m_SamplerSlotCount);
    ++m_SamplerSlotCount;
}

void GraphicsSystem::InitializePrimitiveRenderer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity(2);
    info.SetAllocator(AllocateFunction, m_pApplicationHeap);

    // PrimitiveRendererのインスタンス
    m_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(&m_Device, info);

    m_pPrimitiveRenderer->SetScreenWidth(ScreenWidth);
    m_pPrimitiveRenderer->SetScreenHeight(ScreenHeight);

}

void GraphicsSystem::FinalizeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPool.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPool);
}

void GraphicsSystem::FinalizeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPoolInvisible.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPoolInvisible);
}

void GraphicsSystem::FinalizeBlendState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_BlendState.GetMemory();

    m_BlendState.Finalize(&m_Device);

    m_pApplicationHeap->Free(ptr);
}

void GraphicsSystem::FinalizeDebugFont() NN_NOEXCEPT
{
    m_DebugFont.Finalize();
    delete[] reinterpret_cast<uint8_t*>(m_DebugFontHeap.Get());
    m_DebugFontHeap.Reset(nullptr);
}

void GraphicsSystem::FinalizePrimitiveRenderer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nns::gfx::PrimitiveRenderer::DestroyRenderer(m_pPrimitiveRenderer, &m_Device, FreeFunction, m_pApplicationHeap);
}

void GraphicsSystem::ResetCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_CommandBuffer.Reset();

    m_pMemoryPoolOffset = m_pMemoryPoolCommandOffset;

    const size_t size = m_CommandBufferControlMemoryChunks.size();

    for (size_t i = 0; i < size; ++i)
    {
        m_pApplicationHeap->Free(
            m_CommandBufferControlMemoryChunks[i]);
    }

    m_CommandBufferControlMemoryChunks.clear();
}

void GraphicsSystem::AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    pCommandBuffer->AddCommandMemory(&m_MemoryPool,
        m_pMemoryPoolOffset,
        CommandBufferCommandMemoryChunkSize);

    m_pMemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
}

void GraphicsSystem::AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* pControlMemoryChunk = m_pApplicationHeap->Allocate(
        CommandBufferControlMemoryChunkSize,
        nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device));

    pCommandBuffer->AddControlMemory(pControlMemoryChunk,
        CommandBufferControlMemoryChunkSize);

    m_CommandBufferControlMemoryChunks.push_back(pControlMemoryChunk);
}
