﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{IrSensorHeartRateViewer.cpp,PageSampleIrSensorHeartRateView}

    @brief
    nn::irsensor ライブラリの使い方を示すサンプルプログラム
*/

/**
    @page PageSampleIrSensorHeartRateViewer モーションIRカメラを使った脈拍取得のサンプルプログラム
    @tableofcontents

    @brief
    nn::irsensor ライブラリを用いて Joy-Con の モーションIRカメラを使って脈拍を取得し、それを表示するサンプルプログラムの解説です。

    @section PageSampleIrSensorHeartRateView_SectionBrief 概要
    モーションIRカメラから取得したデータを可視化します。

    @section PageSampleIrSensorHeartRateView_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/IrSensorHeartRateView
    Samples/Sources/Applications/IrSensorHeartRateView @endlink 以下にあります。

    @section PageSampleIrSensorHeartRateView_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。

    @section PageSampleIrSensorHeartRateView_SectionHowToOperate 操作方法
    サンプルプログラムを実行するとウィンドウが立ち上がります。
    ウインドウ上には モーションIRカメラの状態が表示されており、
    コントローラを操作すると状態の変化が反映されます。

    サンプルプログラムを終了させるには + ボタンまたは - ボタンを押してください。

    @section PageSampleIrSensorHeartRateView_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    コントローラのスタイルとして、JoyRight にのみ対応しています。

    @section PageSampleIrSensorHeartRateView_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleIrSensorHeartRateView_SectionDetail 解説
    - Npad を取得
    - Npad から モーションIRカメラのハンドルを取得
    - 脈拍測定モードのデフォルト設定を取得します
    - モーションIRカメラを脈拍測定モードとして起動する
    - ループ開始
    - Npad から入力を取得
    - Npad の入力に合わせて Config 設定を更新する
    - Npad から R ボタンが押されていれば、モード切替及び Config 設定の反映を モーションIRカメラに適応させる
    - モーションIRカメラから現在有効になっているモードのデータを取得する
    - 取得したデータの可視化を行う
    - ループ開始に戻る
*/

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
//#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/irsensor.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/init/init_Malloc.h>

#include "GraphicsSystem.h"
#include "IrSensorDemo.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

namespace {

const int FrameRate = 60;

}

extern "C" void nnMain()
{
    /////////////////////////////////////////////////////////////////
    // Initialization
    /////////////////////////////////////////////////////////////////

    nn::hid::NpadIdType npadIds[] =
    {
        ::nn::hid::NpadId::No1,
        ::nn::hid::NpadId::No2,
    };

    //
    // Memory
    //

    nn::mem::StandardAllocator appAllocator;
    void* workMemoryBuffers[NN_ARRAY_SIZE(npadIds)];
    const size_t appMemorySize  = 128 * 1024 * 1014;
    nn::Bit8* pAppMemory        = new nn::Bit8[appMemorySize];

    appAllocator.Initialize(pAppMemory, appMemorySize);


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    const size_t graphicsMemorySize = 8 * 1024 * 1024;
    void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
    nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif

    //
    // Graphics
    //

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    pGraphicsSystem->SetApplicationHeap(&appAllocator);
    pGraphicsSystem->Initialize();

    //
    // Initialize Hid
    //

    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleJoyRight::Mask
        | nn::hid::NpadStyleJoyLeft::Mask
        | nn::hid::NpadStyleJoyDual::Mask
        | nn::hid::NpadStyleHandheld::Mask
        | nn::hid::NpadStyleFullKey::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));

    //
    // Initialize Demo
    //

    IrSensorDemo demoStates[NN_ARRAY_SIZE(npadIds)];
    for (int i = 0; i < NN_ARRAY_SIZE(npadIds); i++)
    {
        demoStates[i].Initialize(npadIds[i], pGraphicsSystem, nullptr);
    }

    /////////////////////////////////////////////////////////////////
    // Main Loop
    /////////////////////////////////////////////////////////////////

    bool exit = false;
    while (!exit)
    {
        nn::gfx::util::DebugFontTextWriter* pTextWriter = &pGraphicsSystem->GetDebugFont();
        pGraphicsSystem->BeginDraw();

        for (int i = 0; i < NN_ARRAY_SIZE(npadIds); ++i)
        {
            if (!demoStates[i].Run(i, pGraphicsSystem, pTextWriter))
            {
                exit = true;
                break;
            }
        }

        pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    /////////////////////////////////////////////////////////////////
    // Finalization
    /////////////////////////////////////////////////////////////////

    //
    // Finalize Demo
    //
    for (int i = 0; i < NN_ARRAY_SIZE(npadIds); ++i)
    {
        demoStates[i].Finalize();
        appAllocator.Free(workMemoryBuffers[i]);
    }

    //
    // Finalize Graphics
    //

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    //
    // Finalize Memory
    //
    appAllocator.Finalize();
    delete[] pAppMemory;
}
