﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AdaptiveClusteringModeState.h"

#include <algorithm>
#include <sstream>
#include <iostream>
#include <iomanip>

#include <nn/nn_Assert.h>

namespace {

const char* AccuracyLevel[] =
{
    "Low",
    "Middle",
    "High",
};

const char* AdaptiveMode[] =
{
    "StaticFov",
    "DynamicFov",
};

const char* TargetDistance[] =
{
    "Near",
    "Middle",
    "Far",
};

}

AdaptiveClusteringModeState::AdaptiveClusteringModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_AdaptiveClusteringProcessorState()
    , m_AdaptiveClusteringProcessorConfig()
    , m_lastSamplingNumber(0)
    , m_Allocator()
    , m_pWorkBuffer(nullptr)
    , m_pMarkerPlaneEstimation(nullptr)
{
    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_AdaptiveClusteringProcessorState.samplingNumber, &m_AdaptiveClusteringProcessorState.timeStamp, &m_AdaptiveClusteringProcessorState.ambientNoiseLevel);
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);

    m_AdaptiveClusteringProcessorConfig.mode = nn::irsensor::AdaptiveClusteringMode_StaticFov;
    auto* pAdaptiveClusteringConfig = &m_AdaptiveClusteringProcessorConfig;

    m_ReadWriteMenu.emplace_back("Mode",
        [pAdaptiveClusteringConfig](std::stringstream& sstr) {
            sstr << AdaptiveMode[pAdaptiveClusteringConfig->mode];
        },
        [pAdaptiveClusteringConfig](int8_t delta) {
            int8_t mode = static_cast<int8_t>(pAdaptiveClusteringConfig->mode) + delta;
            mode = std::min(std::max(mode, static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_StaticFov)), static_cast<int8_t>(nn::irsensor::AdaptiveClusteringMode_DynamicFov));
            pAdaptiveClusteringConfig->mode = static_cast<nn::irsensor::AdaptiveClusteringMode>(mode);
        }
    );

    m_ReadWriteMenu.emplace_back("TargetDistance",
        [pAdaptiveClusteringConfig](std::stringstream& sstr) {
            sstr << TargetDistance[pAdaptiveClusteringConfig->distance];
        },
        [pAdaptiveClusteringConfig](int8_t delta) {
            int8_t distance = static_cast<int8_t>(pAdaptiveClusteringConfig->distance) + delta;
            distance = std::min(std::max(distance, static_cast<int8_t>(nn::irsensor::AdaptiveClusteringTargetDistance_Near)), static_cast<int8_t>(nn::irsensor::AdaptiveClusteringTargetDistance_Far));
            pAdaptiveClusteringConfig->distance = static_cast<nn::irsensor::AdaptiveClusteringTargetDistance>(distance);
        }
    );

    // Prepare for MarkerPlaneEstimation
    std::memset(m_Heap, 0, MemoryHeapSize);
    m_Allocator.Initialize(reinterpret_cast<void*>(m_Heap), MemoryHeapSize);
    m_pWorkBuffer = reinterpret_cast<unsigned char*>(m_Allocator.Allocate(nn::irsensor::MarkerPlaneEstimationWorkBufferSize));
    m_pMarkerPlaneEstimation = new MarkerPlaneEstimation(m_pWorkBuffer, sizeof(m_pWorkBuffer));

    int8_t* pObjectCount = &m_AdaptiveClusteringProcessorState.objectCount;
    m_ReadOnlyMenu.emplace_back("Object Count",
        [pObjectCount](std::stringstream& sstr) {
            sstr << static_cast<int>(*pObjectCount);
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    auto* pLevel = &m_AdaptiveClusteringProcessorState.accuracyLevel;
    m_ReadOnlyMenu.emplace_back("Accuracy Level",
        [pLevel](std::stringstream& sstr) {
            sstr << AccuracyLevel[*pLevel];
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    uint8_t* pBackgroundIntensity = &m_AdaptiveClusteringProcessorState.backgroundIntensity;
    m_ReadOnlyMenu.emplace_back("Background Intensity",
        [pBackgroundIntensity](std::stringstream& sstr) {
            sstr << static_cast<int>(*pBackgroundIntensity);
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );

    nn::irsensor::MarkerPlaneState* pState;
    m_pMarkerPlaneEstimation->GetState(&pState);
    m_ReadOnlyMenu.emplace_back("Transfer Matrix",
        [pState](std::stringstream& sstr) {
            sstr << std::fixed << std::setprecision(3) << "Confidence:" << pState->confidence << std::endl;
            sstr << std::fixed << std::setprecision(3) << pState->m[0][0] << " " << pState->m[1][0] << " " << pState->m[2][0] << " " << pState->m[3][0] << std::endl;
            sstr << std::fixed << std::setprecision(3) << pState->m[0][1] << " " << pState->m[1][1] << " " << pState->m[2][1] << " " << pState->m[3][1] << std::endl;
            sstr << std::fixed << std::setprecision(3) << pState->m[0][2] << " " << pState->m[1][2] << " " << pState->m[2][2] << " " << pState->m[3][2] << std::endl;
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

AdaptiveClusteringModeState::~AdaptiveClusteringModeState()
{
    delete m_pMarkerPlaneEstimation;
    m_pMarkerPlaneEstimation = nullptr;
    m_Allocator.Free(m_pWorkBuffer);
    m_Allocator.Finalize();
}

void AdaptiveClusteringModeState::Start()
{
    nn::irsensor::RunAdaptiveClusteringProcessor(m_IrCameraHandle, m_AdaptiveClusteringProcessorConfig);
}

void AdaptiveClusteringModeState::Update()
{
    int count;
    nn::Result result = nn::irsensor::GetAdaptiveClusteringProcessorStates(&m_AdaptiveClusteringProcessorState, &count, 1, m_lastSamplingNumber + 1, m_IrCameraHandle);
    HandleResult(result);
    if(count > 0)
    {
        NN_ASSERT(m_AdaptiveClusteringProcessorState.objectCount <= nn::irsensor::AdaptiveClusteringProcessorObjectCountMax);

        m_Statistics.Update(&m_AdaptiveClusteringProcessorState.samplingNumber, IrSensorModeStatistics::ExpectedClusteringModeFramerate);
        m_lastSamplingNumber = m_AdaptiveClusteringProcessorState.samplingNumber;

        nn::irsensor::PointList list = {};
        list.count = m_AdaptiveClusteringProcessorState.objectCount;
        for (auto i = 0; i < list.count; i++)
        {
            list.p[i].area = m_AdaptiveClusteringProcessorState.objects[i].area;
            list.p[i].v[0] = m_AdaptiveClusteringProcessorState.objects[i].centroid.x;
            list.p[i].v[1] = m_AdaptiveClusteringProcessorState.objects[i].centroid.y;
        }
        m_pMarkerPlaneEstimation->Update(list);
    }
}

namespace
{

void WriteAdaptiveClusteringProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::AdaptiveClusteringProcessorState* pAdaptiveClusteringState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pAdaptiveClusteringState);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green ={ { 0, 255, 0, 255 } };
    const nn::util::Uint8x4 Orange ={ { 255, 128, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);


     pPrimitiveRenderer->SetLineWidth(5.f);

    // 各面を描画
    for (auto i = 0; i < pAdaptiveClusteringState->objectCount; ++i)
    {
        nn::irsensor::AdaptiveClusteringData const& pObject = pAdaptiveClusteringState->objects[i];
        pPrimitiveRenderer->SetColor(pObject.isIncomplete ? Orange : Green);

        ::nn::util::Float2 centroid = pObject.centroid;
        float radius = 5.f * sqrt(pObject.area / float(nn::util::FloatPi));

        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (centroid.x - radius) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * centroid.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * (centroid.x + radius) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * centroid.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * centroid.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (centroid.y - radius) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * centroid.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (centroid.y + radius) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * centroid.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * centroid.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
    }
}


}

void AdaptiveClusteringModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WriteAdaptiveClusteringProcessorState(pPrimitiveRenderer, pCommandBuffer, &m_AdaptiveClusteringProcessorState, index);
}
