﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @examplesource{ImguiDemo.cpp,PageSampleImguiDemo}
 *
 * @brief
 *  nns::gfx::dbgui ライブラリの使い方を示すサンプル
 */

/**
* @page PageSampleImguiDemo ImguiDemo
* @tableofcontents
*
* @brief
*  nns::gfx::dbgui ライブラリの使い方を示すサンプル
*
* @section PageSampleImguiDemo_SectionBrief 概要
*  dbgui ライブラリを使いながら、シンプルな ImGui の GUI を作ります。
*
* @section PageSampleImguiDemo_SectionFileStructure ファイル構成
*  本サンプルプログラムは @link ../../../Samples/Sources/Applications/ImguiDemo Samples/Sources/Applications/ImguiDemo @endlink 以下にあります。
*
* @section PageSampleImguiDemo_SectionNecessaryEnvironment 必要な環境
* とくになし
*
* @section PageSampleImguiDemo_SectionHowToOperate 操作方法
* 以下のコントローラを対応しています。\n
* DebugPad　(NXのみ）\n
* Handheld　(NXのみ）\n
* JoyDual　(NXのみ）\n
* Keyboard　(InputDirector でパソコンのキーボードが使えます）\n
* Mouse　(InputDirector でパソコンのマウスが使えます）\n
* TouchScreen　(Handheld モードのみ）\n
* \n
* \n
* 文字入力はソフトキーボード、USB 差し込まれるキーボード、あるいはInputDirectorを利用できます。\n
* WindowsOS で日本語文字の基本的な実装があります。\n
* \n
* \n
* デバッグパッド・DualJoy の操作は、以下の通りです：\n
* D-Pad 上/下/左/右: ナビゲート, 値いの手直し\n
* A ボタン: アイテムのアクティブ, 押しながら値いの手直し, チャイルドに入る, 等\n
* B ボタン: ポップアップメニューを閉じる, チャイルドから出る, 選択をクリア, 等\n
* Y ボタン(TAP): メニューのアクセス, ウインドウの崩壊, ウインドウの設定, 等\n
* Y ボタン(押し)+ Dpad ：ウインドウのサイズの変換\n
* Y ボタン(押し)+アナログスティック:ウインドウを移動する\n
* Y ボタン(押し)+ ZL/LR ：ウインドウのフォカスの変換　(ALT-TABと似ている)\n
* X ボタン: 文字入力\n
* L/R Trigger: 手直しの速さを変換\n
* アナログスティック: スクローリング\n
* \n
* \n
* デバッグパッドで Start ボタンを押すと、マウスの操作ができます。\n
* アナログスティック：マウスポインタの移動\n
* A・D-Pad 右：マウス左クリック\n
* B・D-Pad 下：マウス右クリック\n
* X・D-Pad 上：マウス中クリック\n
* \n
* \n
* DualJoy で「＋」ボタンを押すと、右 JoyCon の SixAxisSensor ポインタでマウス操作できます。\n
* A・D-Pad 右：マウス左クリック\n
* B・D-Pad 下：マウス右クリック\n
* X・D-Pad 上：マウス中クリック\n
* \n
* \n
* 「R」と「＋」か「スタート」ボタンを同時に押すと、デバッグGUIを有効・無効にすることができます。\n
*
* @section PageSampleImguiDemo_SectionPrecaution 注意事項
*  とくになし
*
* @section PageSampleImguiDemo_SectionHowToExecute 実行手順
*  サンプルプログラムをビルドし、実行してください。
*/


#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include <nn/gfx/util/gfx_ScalableViewport.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/dbgui/dbgui_Interface.h>
#include <nns/dbgui/dbgui_DefaultUpdateScissorCallbacks.h>

#include <imgui.h>

#include "ImguiDemo_IrSensor.h"
#include "ImguiDemo_UserInputs.h"


#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
#define NNS_IMGUIDEMO_ENABLE_IR_CAMERA
#endif  // NN_BUILD_TARGET_PLATFORM_OS_NN

#define CONTENTS_MOUNT_POINT "contents"
#define HOST_MOUNT_POINT "host"

void MountContents()
{
    static const size_t MountRomCacheBufferSize = 4 * 1024;
    static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
    NN_ASSERT(mountRomCacheUseSize <= MountRomCacheBufferSize);

    NN_LOG("Mount Contents\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(CONTENTS_MOUNT_POINT, s_MountRomCacheBuffer, MountRomCacheBufferSize));
    nn::fs::DirectoryEntry entryList[100];
    nn::fs::DirectoryHandle h = {};
    nn::fs::OpenDirectory(&h, CONTENTS_MOUNT_POINT":/", nn::fs::OpenDirectoryMode_All);
    int64_t n = 0;
    nn::fs::ReadDirectory(&n, entryList, h, 100);
    NN_LOG("%d entry found.\n", static_cast<int>(n));
    for (int64_t i = 0; i < n; i++)
    {
        auto& e = entryList[i];
        NN_LOG("  %s%s\n", e.name, (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "/" : ""));
    }
    nn::fs::CloseDirectory(h);

    nn::fs::MountHost(HOST_MOUNT_POINT, ".");
}

void UnmountContents()
{
    nn::fs::Unmount(HOST_MOUNT_POINT);
    nn::fs::Unmount(CONTENTS_MOUNT_POINT);
}

void InitializeGraphicsFramework(
    nns::gfx::GraphicsFramework* pGfw,
    nn::gfx::util::ScalableViewport::OriginMode windowOriginMode,
    int displayWidth, int displayHeight, int bufferCount)
{
    const int ScreenBpp = 4;

    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;

    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(displayWidth);
    fwInfo.SetDisplayHeight(displayHeight);
    fwInfo.SetBufferCount(bufferCount);
    fwInfo.SetSwapChainBufferCount(bufferCount);
    fwInfo.SetDepthStencilBufferEnabled(false);
    fwInfo.SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);

    nn::gfx::util::ScalableViewport::WindowCoordinate virtualWindowCoordinate, physicalWindowCoord;
    {
        float windowWidth, windowHeight;
        windowWidth = static_cast<float>(displayWidth);
        windowHeight = static_cast<float>(displayHeight);

        virtualWindowCoordinate.SetWidth(windowWidth);
        virtualWindowCoordinate.SetHeight(windowHeight);
        virtualWindowCoordinate.SetOriginMode(nn::gfx::util::ScalableViewport::OriginMode_UpperLeft);

        physicalWindowCoord.SetWidth(windowWidth);
        physicalWindowCoord.SetHeight(windowHeight);
        physicalWindowCoord.SetOriginMode(windowOriginMode);
    }
    fwInfo.SetPhysicalWindowCoordinate(physicalWindowCoord);
    fwInfo.SetVirtualWindowCoordinate(virtualWindowCoordinate);

    fwInfo.SetMemoryPoolSize(
        nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer,
        4 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(
        nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
        nn::util::align_up(displayWidth * displayHeight * ScreenBpp, 4096) * (bufferCount * 2));
    fwInfo.SetMemoryPoolSize(
        nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
        1024 * 1024);

    fwInfo.SetRootCommandBufferCommandMemorySize(1 * 1024 * 1024);
#if defined(NN_SDK_BUILD_DEBUG)
    fwInfo.SetDebugMode(nn::gfx::DebugMode_Full);
#endif
    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_BufferView, 16);
    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_TextureView, 1024);
    fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_Sampler, 1024);

    pGfw->Initialize(fwInfo);
}

nns::dbgui::ViewportYAxisDirection GetViewportYAxisDirection(
    nn::gfx::util::ScalableViewport::OriginMode windowOriginMode)
{
    nns::dbgui::ViewportYAxisDirection result = nns::dbgui::ViewportYAxisDirection_Up;

#if NN_GFX_IS_TARGET_GL
    result =
        (windowOriginMode == nn::gfx::util::ScalableViewport::OriginMode_UpperLeft)
        ? nns::dbgui::ViewportYAxisDirection_Down
        : nns::dbgui::ViewportYAxisDirection_Up;
#elif NN_GFX_IS_TARGET_NVN
    NN_UNUSED(windowOriginMode);
    result = nns::dbgui::ViewportYAxisDirection_Up;
#else
#error unsupported underlying api
#endif

    return result;
}

void ConfigureScissorScalableViewportPhysicalWindowCoord(
    nn::gfx::util::ScalableViewport::WindowCoordinate* pScissorScalableViewportPhysicalWindowCoord,
    int viewportWidth, int viewportHeight,
    nn::gfx::util::ScalableViewport::OriginMode originMode)
{
    pScissorScalableViewportPhysicalWindowCoord->SetWidth(static_cast<float>(viewportWidth));
    pScissorScalableViewportPhysicalWindowCoord->SetHeight(static_cast<float>(viewportHeight));

#if NN_GFX_IS_TARGET_GL
    NN_UNUSED(originMode);
    pScissorScalableViewportPhysicalWindowCoord->SetOriginMode(nn::gfx::util::ScalableViewport::OriginMode_LowerLeft);
#elif NN_GFX_IS_TARGET_NVN
    pScissorScalableViewportPhysicalWindowCoord->SetOriginMode(originMode);
#else
    NN_UNUSED(originMode);
#error unsupported underlying api
#endif
}

void InitializeScissorScalableViewport(
    nn::gfx::util::ScalableViewport* pScissorScalableViewport,
    int imguiDisplayWidth, int imguiDisplayHeight,
    int viewportWidth, int viewportHeight,
    nn::gfx::util::ScalableViewport::OriginMode originMode)
{
    nn::gfx::util::ScalableViewport::WindowCoordinate virtualWindowCoordinate;
    virtualWindowCoordinate.SetWidth(static_cast<float>(imguiDisplayWidth));
    virtualWindowCoordinate.SetHeight(static_cast<float>(imguiDisplayHeight));
    virtualWindowCoordinate.SetOriginMode(nn::gfx::util::ScalableViewport::OriginMode_UpperLeft);

    nn::gfx::util::ScalableViewport::WindowCoordinate physicalWindowCoordinate;
    ConfigureScissorScalableViewportPhysicalWindowCoord(
        &physicalWindowCoordinate,
        viewportWidth, viewportHeight,
        originMode);
    pScissorScalableViewport->Initialize(virtualWindowCoordinate, physicalWindowCoordinate);
}

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
struct IrCameraTexture
{
    static const nns::gfx::GraphicsFramework::MemoryPoolType MemoryPoolType =
        nns::gfx::GraphicsFramework::MemoryPoolType_Data;

    nn::gfx::Texture                texture;
    nn::gfx::TextureView            textureView;
    nn::gfx::DescriptorSlot         descriptorSlot;
    ptrdiff_t                       memoryPoolOffset = 0;
    ptrdiff_t                       texturePitch = 0;
    size_t                          textureBufferSize = 0;
    int                             textureDescriptorSlotIndex = -1;
};

void InitializeIrCameraTexture(
    IrCameraTexture* pIrCameraTexture,
    nns::gfx::GraphicsFramework* pGfw)
{
    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetWidth(nn::irsensor::IrHandAnalysisImageWidth);
    textureInfo.SetHeight(nn::irsensor::IrHandAnalysisImageHeight);
    textureInfo.SetImageFormat(nn::gfx::ImageFormat_R16_Unorm);
    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetTileMode(nn::gfx::TileMode_Linear);

    size_t size =
        nn::gfx::Texture::CalculateMipDataSize(pGfw->GetDevice(), textureInfo);
    size_t alignment =
        nn::gfx::Texture::CalculateMipDataAlignment(pGfw->GetDevice(), textureInfo);
    ptrdiff_t pitch =
        nn::gfx::Texture::GetRowPitch(pGfw->GetDevice(), textureInfo);

    ptrdiff_t memoryPoolOffset = pGfw->AllocatePoolMemory(
        IrCameraTexture::MemoryPoolType, size, alignment);

    void* pBuffer = pGfw->GetMemoryPool(IrCameraTexture::MemoryPoolType)->Map<uint8_t>() + memoryPoolOffset;
    memset(pBuffer, 0xFF, size);
    pGfw->GetMemoryPool(IrCameraTexture::MemoryPoolType)->Unmap();
    pGfw->GetMemoryPool(IrCameraTexture::MemoryPoolType)->FlushMappedRange(memoryPoolOffset, size);

    pIrCameraTexture->texture.Initialize(
        pGfw->GetDevice(), textureInfo, pGfw->GetMemoryPool(IrCameraTexture::MemoryPoolType),
        memoryPoolOffset, size);

    nn::gfx::TextureView::InfoType textureViewInfo;
    textureViewInfo.SetDefault();
    textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    textureViewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_Unorm);
    textureViewInfo.SetTexturePtr(&pIrCameraTexture->texture);
    textureViewInfo.SetChannelMapping(
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_One);
    pIrCameraTexture->textureView.Initialize(pGfw->GetDevice(), textureViewInfo);

    int textureViewDescriptorSlotIndex =
        pGfw->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);

    nn::gfx::DescriptorPool* pDescriptorPool =
        pGfw->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView);
    pDescriptorPool->BeginUpdate();
    pDescriptorPool->SetTextureView(textureViewDescriptorSlotIndex, &pIrCameraTexture->textureView);
    pDescriptorPool->EndUpdate();
    pDescriptorPool->GetDescriptorSlot(&pIrCameraTexture->descriptorSlot, textureViewDescriptorSlotIndex);

    pIrCameraTexture->memoryPoolOffset = memoryPoolOffset;
    pIrCameraTexture->texturePitch = static_cast<ptrdiff_t>(pitch);
    pIrCameraTexture->textureBufferSize = size;
    pIrCameraTexture->textureDescriptorSlotIndex = textureViewDescriptorSlotIndex;
}

void FinalizeIrCameraTexture(
    IrCameraTexture* pIrCameraTexture,
    nns::gfx::GraphicsFramework* pGfw)
{
    pIrCameraTexture->texture.Finalize(pGfw->GetDevice());
    pIrCameraTexture->textureView.Finalize(pGfw->GetDevice());
    pGfw->FreePoolMemory(IrCameraTexture::MemoryPoolType, pIrCameraTexture->memoryPoolOffset);
    pGfw->FreeDescriptorSlot(
        nn::gfx::DescriptorPoolType_TextureView,
        pIrCameraTexture->textureDescriptorSlotIndex);
}

void UpdateIrCameraTextureContent(
    nns::gfx::GraphicsFramework* pGfw,
    nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType,
    ptrdiff_t memoryPoolOffset, ptrdiff_t pitch, size_t bufferSize)
{
    nn::gfx::MemoryPool* pMemoryPool = pGfw->GetMemoryPool(memoryPoolType);
    void* pBuffer = pMemoryPool->Map<uint8_t>() + memoryPoolOffset;

    CopyIrSensorImageR16(pBuffer, pitch, bufferSize);

    pMemoryPool->Unmap();
    pMemoryPool->FlushMappedRange(memoryPoolOffset, bufferSize);
}
#endif // defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)

void* SetupApplicationCustomFont(
    nns::gfx::GraphicsFramework* pGfw,
    nns::dbgui::InterfaceInfo& dbguiInterfaceInfo)
{
    void* pFontBuffer = nullptr;
    size_t fontBufferSize = 0;

    nn::Result result;
    nn::fs::FileHandle hTtfFile;
    result = nn::fs::OpenFile(
        &hTtfFile, CONTENTS_MOUNT_POINT":/NotoSansCJKjp-Regular.otf",
        nn::fs::OpenMode_Read);

    float characterSize = 22.0f;

    if (result.IsSuccess())
    {
        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, hTtfFile);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        fontBufferSize = static_cast<size_t>(fileSize);
        pFontBuffer = pGfw->AllocateMemory(fontBufferSize, 8);
        size_t readSize = 0;
        result = nn::fs::ReadFile(&readSize, hTtfFile, 0, pFontBuffer, fontBufferSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_ASSERT(fontBufferSize == readSize);

        dbguiInterfaceInfo.SetCustomTtf(
            pFontBuffer, fontBufferSize,
            false, characterSize);

        nn::fs::CloseFile(hTtfFile);
    }

    return pFontBuffer;
}

void* InitializeDbgui(
    nns::gfx::GraphicsFramework* pGfw,
    nns::dbgui::ViewportYAxisDirection viewportYAxisDirection,
    nn::gfx::util::ScalableViewport* pScissorUpdateScalableViewport,
    int imguiDisplayWidth, int imguiDisplayHeight, int bufferCount)
{
    nns::dbgui::InterfaceInfo dbguiInterfaceInfo;
    dbguiInterfaceInfo.SetDefault();
    dbguiInterfaceInfo.SetBufferCount(bufferCount);
    dbguiInterfaceInfo.SetImguiDisplaySize(imguiDisplayWidth, imguiDisplayHeight);
    dbguiInterfaceInfo.SetAllocator(
        pGfw->GetAllocateFunction(),
        pGfw->GetFreeFunction(),
        pGfw->GetAllocateFunctionUserData());
    dbguiInterfaceInfo.SetDescriptorPoolData(
        pGfw->GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
        pGfw->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1),
        pGfw->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
        pGfw->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1),
        pGfw->GetDescriptorPool(nn::gfx::DescriptorPoolType_BufferView),
        pGfw->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_BufferView, 1));
    dbguiInterfaceInfo.SetUpdateScissorCallback(
        nns::dbgui::UpdateScissorScalableViewport,
        pScissorUpdateScalableViewport);
    dbguiInterfaceInfo.SetViewportYAxisDirection(viewportYAxisDirection);
    dbguiInterfaceInfo.SetMouseMoveFunction(OnMouseMoveFunction, nullptr);
    dbguiInterfaceInfo.SetIniFilePath(HOST_MOUNT_POINT":/imgui.ini");

    void* pCustomFontBuffer = SetupApplicationCustomFont(pGfw, dbguiInterfaceInfo);

    nns::dbgui::Initialize(pGfw->GetDevice(), dbguiInterfaceInfo);

    return pCustomFontBuffer;
}

struct GuiState
{
    bool isEnabled;
    bool showDemoWindow;
    bool showAnotherWindow;
    ImVec4 clearColor;
    float sliderValue;
    static const int testBufferSize = 128;
    char testBufferSingleLine[testBufferSize];
    char testBufferMuliline[testBufferSize];
    bool exitRequested;
    bool showDisplaySizeWidow;
    int renderTargetWidth;
    int renderTargetHeight;
    int imguiDisplayWidth;
    int imguiDisplayHeight;
    int viewportWidth;
    int viewportHeight;
    float displayScale;
    bool blitToFullScreen;
    bool displayChangeRequested;
    nn::gfx::ViewportScissorState viewportScissorState;
    nn::gfx::util::ScalableViewport scissorScalableViewport;

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
    bool showIrCameraWindow;
    nn::gfx::DescriptorSlot* pIrCameraTextureDescriptorSlot;
#endif
};

void InitializeViewportScissorState(GuiState* pGuiState, nn::gfx::Device* pDevice)
{
    NN_ASSERT(pGuiState->viewportWidth <= pGuiState->renderTargetWidth);
    NN_ASSERT(pGuiState->viewportHeight <= pGuiState->renderTargetHeight);


    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetOriginX(static_cast<float>(0));
    viewportStateInfo.SetOriginY(static_cast<float>(0));
    viewportStateInfo.SetWidth(static_cast<float>(pGuiState->viewportWidth));
    viewportStateInfo.SetHeight(static_cast<float>(pGuiState->viewportHeight));

    nn::gfx::ScissorStateInfo scissorInfo;
    scissorInfo.SetDefault();
    scissorInfo.SetWidth(pGuiState->viewportWidth);
    scissorInfo.SetHeight(pGuiState->viewportHeight);

    nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;
    viewportScissorInfo.SetDefault();
    viewportScissorInfo.SetScissorEnabled(true);
    viewportScissorInfo.SetViewportStateInfoArray(&viewportStateInfo, 1);
    viewportScissorInfo.SetScissorStateInfoArray(&scissorInfo, 1);

    NN_SDK_ASSERT(nn::gfx::ViewportScissorState::GetRequiredMemorySize(viewportScissorInfo) == 0);
    pGuiState->viewportScissorState.Initialize(pDevice, viewportScissorInfo);
}



void ImGuiDisplaySizeWidow(GuiState* pGuiState)
{
    ImGui::SetNextWindowPos(ImVec2(0, 0), ImGuiCond_Appearing);
    ImGui::SetNextWindowSize(
        ImVec2(
            static_cast<float>(pGuiState->imguiDisplayWidth),
            static_cast<float>(pGuiState->imguiDisplayHeight)),
        ImGuiCond_Appearing);
    ImGui::Begin("Display size debug", &pGuiState->showDisplaySizeWidow);

    if (ImGui::SliderInt(
        "display width", &pGuiState->viewportWidth,
        pGuiState->imguiDisplayWidth / 10, pGuiState->imguiDisplayWidth))
    {
        pGuiState->displayChangeRequested = true;
    }
    if (ImGui::SliderInt(
        "display height", &pGuiState->viewportHeight,
        pGuiState->imguiDisplayHeight / 10, pGuiState->imguiDisplayHeight))
    {
        pGuiState->displayChangeRequested = true;
    }
    if (ImGui::SliderFloat("display scale", &pGuiState->displayScale, 0.1f, 1.0f))
    {
        pGuiState->viewportWidth = static_cast<int>(static_cast<float>(pGuiState->imguiDisplayWidth) * pGuiState->displayScale);
        pGuiState->viewportHeight = static_cast<int>(static_cast<float>(pGuiState->imguiDisplayHeight) * pGuiState->displayScale);
        pGuiState->displayChangeRequested = true;
    }

    ImGui::Checkbox("Blit to fullscreen", &pGuiState->blitToFullScreen);

    ImGui::End();
}

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
void ImGuiDisplayIrCameraWindow(GuiState* pGuiState)
{
    ImGui::SetNextWindowSize(
        ImVec2(
            static_cast<float>(500),
            static_cast<float>(500)),
        ImGuiCond_FirstUseEver);

    ImGui::Begin("Ir Camera", &pGuiState->showIrCameraWindow);

    int handCount = GetIrSensorHandCount();
    int handFingerCount[nn::irsensor::IrHandAnalysisHandCountMax] = { 0 };
    IrSensorHandInput irSensorHandInput = GetIrSensorHandInput();

    for (int handIndex = 0; handIndex < handCount; ++handIndex)
    {
        handFingerCount[handIndex] = GetIrSensorHandFingerCount(handIndex);
    }

    NN_ASSERT(nn::irsensor::IrHandAnalysisHandCountMax >= 2);
    ImGui::Text("HandCount:%d Hand0FingerCount:%d Hand1FingerCount:%d",
        handCount,
        handFingerCount[0], handFingerCount[1]);

    ImVec2 contentRegionMin = ImGui::GetWindowContentRegionMin();
    ImVec2 contentRegionMax = ImGui::GetWindowContentRegionMax();
    ImVec2 imageSize(
        contentRegionMax.x - contentRegionMin.x,
        contentRegionMax.y - contentRegionMin.y);
    ImTextureID textureId = reinterpret_cast<ImTextureID>(pGuiState->pIrCameraTextureDescriptorSlot);
    ImColor tintColor = 0xFFFFFFFF;
    if (irSensorHandInput == IrSensorHandInput_Validate)
    {
        tintColor = 0xFF0000FF;
    }
    else if (irSensorHandInput == IrSensorHandInput_Cancel)
    {
        tintColor = 0xFF00FF00;
    }
    ImGui::Image(
        textureId, imageSize,
        ImVec2(0, 0), ImVec2(1, 1),
        tintColor);
    ImGui::End();
}
#endif

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    GuiState* pGuiState = reinterpret_cast<GuiState*>(pUserData);

    if (pGuiState->isEnabled)
    {

        nns::dbgui::NewFrame();

        // 1. Show a simple window.
        // Tip: if we don't call ImGui::Begin()/ImGui::End() the widgets appears in a window automatically
        // called "Debug".
        {
            ImGui::Text("Hello, world!");
            ImGui::Text(u8"日本語も表示できます");
            ImGui::Text("viewportWidth %d viewportHeight %d", pGuiState->viewportWidth, pGuiState->viewportHeight);
            ImGui::InputText("input text", pGuiState->testBufferSingleLine, pGuiState->testBufferSize);
            ImGui::InputTextMultiline("input text multiline", pGuiState->testBufferMuliline, pGuiState->testBufferSize);
            ImGui::SliderFloat("transparency", &pGuiState->sliderValue, 0.0f, 1.0f);
            ImGui::ColorEdit3("clear color", (float*)&pGuiState->clearColor);
            if (ImGui::Button("Demo Window"))
            {
                pGuiState->showDemoWindow ^= 1;
            }
            if (ImGui::Button("Another Window"))
            {
                pGuiState->showAnotherWindow ^= 1;
            }
            if (ImGui::Button("Display size debug Window"))
            {
                pGuiState->showDisplaySizeWidow ^= 1;
            }
#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
            if (ImGui::Button("Display IrCamera window"))
            {
                pGuiState->showIrCameraWindow ^= 1;
            }
#endif
            ImGui::Text("Application average %.3f ms/frame (%.1f FPS)",
                1000.0f / ImGui::GetIO().Framerate, ImGui::GetIO().Framerate);
            if (ImGui::Button("Exit application")) pGuiState->exitRequested ^= 1;
        }

        // 2. Show another simple window. In most cases you will use an explicit Begin/End pair to name the window.
        if (pGuiState->showAnotherWindow)
        {
            ImGui::Begin("Another Window", &pGuiState->showAnotherWindow);
            ImGui::Text("Hello from another window!");
            ImGui::End();
        }

        // 3. Show the ImGui test window. Most of the sample code is in ImGui::ShowDemoWindow().
        if (pGuiState->showDemoWindow)
        {
            // Normally user code doesn't need/want to call it because positions are saved in .ini file anyway.
            // Here we just want to make the demo initial state a bit more friendly!
            ImGui::SetNextWindowPos(ImVec2(650, 20), ImGuiCond_FirstUseEver);
            ImGui::ShowDemoWindow(&pGuiState->showDemoWindow);
        }

        // 4. Test viewport resize
        if (pGuiState->showDisplaySizeWidow)
        {
            ImGuiDisplaySizeWidow(pGuiState);
        }

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
        // 4. Ir Camera information
        if (pGuiState->showIrCameraWindow)
        {
            ImGuiDisplayIrCameraWindow(pGuiState);
        }
#endif
    }
}

void MakeCommandCallback(nns::gfx::GraphicsFramework* pGfw, int bufferIndex, void* pUserData)
{
    GuiState* pGuiState = static_cast<GuiState*>(pUserData);

    pGfw->BeginFrame(bufferIndex);

    nn::gfx::CommandBuffer* pCommandBuffer = pGfw->GetRootCommandBuffer(bufferIndex);

    nn::gfx::ColorTargetView* pScreenTarget = pGfw->GetColorTargetView();
    pCommandBuffer->ClearColor(
        pScreenTarget,
        pGuiState->clearColor.x,
        pGuiState->clearColor.y,
        pGuiState->clearColor.z,
        pGuiState->clearColor.w,
        nullptr);
    pCommandBuffer->SetRenderTargets(1, &pScreenTarget, nullptr);

    if (pGuiState->displayChangeRequested)
    {
        nn::gfx::Device* pDevice = pGfw->GetDevice();
        pGuiState->viewportScissorState.Finalize(pDevice);
        InitializeViewportScissorState(pGuiState, pDevice);

        nn::gfx::util::ScalableViewport::WindowCoordinate physicalWindowCoordinate = *pGfw->GetScalableViewport()->GetPhysicalWindowCoordinate();
        physicalWindowCoordinate.SetWidth(static_cast<float>(pGuiState->viewportWidth));
        physicalWindowCoordinate.SetHeight(static_cast<float>(pGuiState->viewportHeight));
        pGfw->SetPhysicalWindowCoordinate(physicalWindowCoordinate);

        nn::gfx::util::ScalableViewport::WindowCoordinate scissorScalableViewportPhysicalWindowCoord;
        ConfigureScissorScalableViewportPhysicalWindowCoord(
            &scissorScalableViewportPhysicalWindowCoord,
            pGuiState->viewportWidth, pGuiState->viewportHeight,
            physicalWindowCoordinate.GetOriginMode());
        pGuiState->scissorScalableViewport.SetPhysicalWindowCoordinate(scissorScalableViewportPhysicalWindowCoord);


        pGuiState->displayChangeRequested = false;
    }
    pCommandBuffer->SetViewportScissorState(&pGuiState->viewportScissorState);

    if (pGuiState->isEnabled)
    {
        nns::dbgui::Render(pCommandBuffer);
    }

    pCommandBuffer->ClearColor(
        pGfw->GetScanBufferView(pGfw->GetNextScanBufferIndex()),
        0, 0, 0, 0,
        nullptr);

    nn::gfx::Texture* pScanBuffer = pGfw->GetScanBuffer(pGfw->GetNextScanBufferIndex());

    nn::gfx::Texture* pColorBuffer = pGfw->GetColorBuffer();

    nn::gfx::TextureCopyRegion srcRegion;
    srcRegion.SetDefault();
    srcRegion.SetOffsetU(0);
    srcRegion.SetOffsetV(0);
    srcRegion.SetWidth(pGuiState->viewportWidth);
    srcRegion.SetHeight(pGuiState->viewportHeight);

    nn::gfx::TextureCopyRegion dstRegion;
    dstRegion.SetDefault();

    if (pGuiState->blitToFullScreen)
    {
        dstRegion.SetOffsetU(0);
        dstRegion.SetOffsetV(0);
        dstRegion.SetWidth(pGuiState->renderTargetWidth);
        dstRegion.SetHeight(pGuiState->renderTargetHeight);
    }
    else
    {
        dstRegion.SetOffsetU(0);
        dstRegion.SetOffsetV(0);
        dstRegion.SetWidth(pGuiState->viewportWidth);
        dstRegion.SetHeight(pGuiState->viewportHeight);
    }
    pCommandBuffer->BlitImage(pScanBuffer, dstRegion, pColorBuffer, srcRegion, 0);

    pGfw->EndFrame(bufferIndex, false);
}

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    const int ImguiDisplayWidth = 1920;
    const int ImguiDisplayHeight = 1080;
#else
    const int ImguiDisplayWidth = 1280;
    const int ImguiDisplayHeight = 720;
#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

    const int BufferCount = 2;

    MountContents();

    nn::gfx::util::ScalableViewport::OriginMode windowOriginMode =
        nn::gfx::util::ScalableViewport::OriginMode::OriginMode_UpperLeft;

    nns::gfx::GraphicsFramework gfw;
    InitializeGraphicsFramework(
        &gfw, windowOriginMode,
        ImguiDisplayWidth, ImguiDisplayHeight, BufferCount);

    ApplicationInputs userInputs = {};
    InitializeApplicationInputs(gfw.GetLayer(), &userInputs);

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
    IrCameraTexture irCameraTexture = {};
    InitializeIrCameraTexture(&irCameraTexture, &gfw);
#endif

    GuiState guiState = {};
    guiState.isEnabled = true;
    guiState.clearColor = ImVec4(0.45f, 0.55f, 0.60f, 1.00f);
    guiState.exitRequested = false;
    guiState.imguiDisplayWidth = ImguiDisplayWidth;
    guiState.imguiDisplayHeight = ImguiDisplayHeight;
    guiState.viewportWidth = ImguiDisplayWidth;
    guiState.viewportHeight = ImguiDisplayHeight;
    guiState.renderTargetWidth = gfw.GetDisplayWidth();
    guiState.renderTargetHeight = gfw.GetDisplayHeight();
    guiState.displayScale = 1.0f;
    guiState.blitToFullScreen = true;
#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
    guiState.showIrCameraWindow = false;
    guiState.pIrCameraTextureDescriptorSlot = &irCameraTexture.descriptorSlot;
#endif

    InitializeScissorScalableViewport(
        &guiState.scissorScalableViewport,
        guiState.imguiDisplayWidth, guiState.imguiDisplayHeight,
        guiState.viewportWidth, guiState.viewportHeight,
        windowOriginMode);

    void* pCustomFontBuffer = InitializeDbgui(
        &gfw,
        GetViewportYAxisDirection(windowOriginMode),
        &guiState.scissorScalableViewport,
        ImguiDisplayWidth, ImguiDisplayHeight, BufferCount);

    InitializeViewportScissorState(&guiState, gfw.GetDevice());

    gfw.SetCalculateCallback(&CalculateCallback, &guiState);
    gfw.SetMakeCommandCallback(&MakeCommandCallback, &guiState);

    while (!guiState.exitRequested)
    {
        GetApplicationInputs(gfw.GetLayer(), &userInputs);

        if (TestToggleDebugGui(&userInputs))
        {
            guiState.isEnabled = !guiState.isEnabled;
        }

        if (guiState.isEnabled)
        {
            UpdateDdguiInputs(&userInputs);
        }

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
        UpdateIrCameraTextureContent(
            &gfw,
            irCameraTexture.MemoryPoolType, irCameraTexture.memoryPoolOffset,
            irCameraTexture.texturePitch, irCameraTexture.textureBufferSize);
#endif

        gfw.ProcessFrame();
    }

    guiState.viewportScissorState.Finalize(gfw.GetDevice());
    nns::dbgui::Finalize(gfw.GetDevice());
    guiState.scissorScalableViewport.Finalize();

#if defined(NNS_IMGUIDEMO_ENABLE_IR_CAMERA)
    FinalizeIrCameraTexture(&irCameraTexture, &gfw);
#endif

    if (pCustomFontBuffer != nullptr)
    {
        gfw.FreeMemory(pCustomFontBuffer);
        pCustomFontBuffer = nullptr;
    }

    gfw.Finalize();

    FinalizeApplicationInputs(&userInputs);

    UnmountContents();
}
