﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{ImageJpegEncoding.cpp,PageSampleImageJpegEncoding}

    @brief
    ピクセルデータにサムネイル画像と Exif 情報を付与して JPEG エンコードするためのサンプルプログラム
 */

/**
    @page PageSampleImageJpegEncoding JPEG エンコードと Exif 情報の設定
    @tableofcontents

    @brief
    JPEG エンコードと Exif 情報の設定のサンプルプログラムの解説です。

    @section PageSampleImageJpegEncoding_SectionBrief 概要
    ピクセルデータを生成し、 JPEG データにエンコードします。 併せて、サムネイル画像と Exif 情報を設定します。

    @section PageSampleImageJpegEncoding_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/ImageJpegEncoding
    Samples/Sources/Applications/ImageJpegEncoding @endlink 以下にあります。

    @section PageSampleImageJpegEncoding_SectionNecessaryEnvironment 必要な環境
    このサンプルプログラムはホスト PC のファイルシステムをマウント可能な環境でのみ動作します。

    @section PageSampleImageJpegEncoding_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、ピクセルデータを作成し解析を行い、サムネイル画像と Exif 情報を付与して JPEG エンコードを行います。
    得られた JEPG データはファイルとしてホスト PC のストレージに保存されます。

    @section PageSampleImageJpegEncoding_SectionPrecaution 注意事項
    出力される JPEG ファイルは、ホスト PC の "C:\JpegInputDir\output.jpg" として保存されます。
    ディレクトリ "C:\JpegInputDir" はあらかじめ作成されている必要があります。

    @section PageSampleImageJpegEncoding_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleImageJpegEncoding_SectionDetail 解説
    このサンプルプログラムは、ピクセルデータを JPEG データにエンコードする方法を示します。
    JPEG データへのエンコードには、「サムネイル画像の付与」と「Exif 情報の設定」が含まれます。

    サンプルプログラムの処理の流れは以下の通りです。

    - ピクセルデータの生成
    - 出力用バッファの確保
    - Exif 情報の設定
        - メタデータの設定
        - サムネイル画像の生成と設定
    - JPEG エンコード
        - エンコードのパラメータを設定
        - ピクセルデータとエンコードパラメータの解析
        - JPEG エンコードに要するワークメモリ量を計算
        - Exif 情報を指定したエンコード処理
    - JPEG データをファイルに出力

    エンコードの対象となるピクセルデータを生成します。
    これは適当な大きさで行アラインメントされたバッファに格納します。
    次に、出力される JPEG データを格納するバッファを用意します。
    出力される JPEG データの大きさは厳密に予測することができませんので、あらかじめ十分大きなバッファを予約してください。

    Exif 情報は、 JPEG エンコードと独立し、 nn::image::ExifBuilder を使用して構築することができます。
    サポートしているメタデータのうち任意の組み合わせを設定できますが、サムネイル画像のサイズと合わせて、最終的な Exif 情報の大きさが
    65527 バイトを超えないように注意してください。
    サムネイル画像は任意のピクセルデータを JPEG エンコードすることで取得できます。
    Exif 情報の大きさを超えない範囲であれば、任意のピクセル数やバイト数のサムネイル画像を指定できます。
    ただし、サムネイル画像には Exif 情報を含めないでください。

    これまでで得られた「ピクセルデータ」、「サムネイル画像」、「Exif 情報」を使用して JPEG データを生成します。
    nn::image::JpegEncoder にピクセルデータとエンコードパラメータを設定し、解析を行うと、エンコードに必要なワークメモリ量を取得できます。
    適切なサイズのワークメモリを確保したら、 Exif 情報を指定して JPEG エンコードを行います。
    サムネイル画像と各メタデータ、 nn::image::ExifBuilder のワークメモリは JPEG エンコード後に解放できます。
    得られた JPEG データは、そのままファイルとしてホスト PC のファイルシステム上に保存されます。

    このサンプルプログラムを実行して得られる結果を以下に示します。
    また、サンプルプログラムの実行で得られるJPEG データは、このサンプルと同じフォルダから ExampleOutput.jpg として参照できます。

    @verbinclude ExampleOutput.txt

    このアプリケーションは次の OSS を使用して作成されています。

    - IJG libjpeg
        - this software is based in part on the work of the Independent JPEG Group
 */

#define NN_LOG_USE_DEFAULT_LOCALE_CHARSET

#include <cstdlib>
#include <cmath>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/image/image_ExifBuilder.h>
#include <nn/image/image_JpegEncoder.h>

#include "Utilities.h"

namespace
{
static const size_t BytesPerPixel = sizeof(nn::Bit32); // 1px あたりのバイト数定義

/// @brief JPEG ライブラリに設定可能なメタデータ
struct MetaData
{
    // 0th TIFF IFD カテゴリのメタデータ
    nn::image::ExifOrientation orientation; // 画像方向
    const char *software;   // ソフトウェア名
    size_t softwareSize;    // 終端文字を含めたソフトウェア名のバイト数
    const char *dateTime;   // 撮影日時 (主単文字を含め20 バイト)

    // 0th Exif IFD カテゴリのメタデータ
    const void *makerNote;  // メーカーノート
    size_t makerNoteSize;   // メーカーノートのバイト数
    nn::image::Dimension effectiveDim;  // 実効画像サイズ
    const char *uniqueId;   // 画像のユニーク ID (終端文字を含め 33 バイト)
};

/// @brief 与えられた Exif 情報の内容をコンソールに出力します。
void PrintMetaData(const MetaData &metaData)
{
    const char *orientationStr;
    switch (metaData.orientation)
    {
    case nn::image::ExifOrientation_Normal:
        orientationStr = "正立";
        break;
    case nn::image::ExifOrientation_FlipHorizontal:
        orientationStr = "水平方向に反転";
        break;
    case nn::image::ExifOrientation_Rotate180:
        orientationStr = "時計回りに 180 度回転";
        break;
    case nn::image::ExifOrientation_FlipVertical:
        orientationStr = "垂直方向に反転";
        break;
    case nn::image::ExifOrientation_FlipTopRightToLeftBottom:
        orientationStr = "右下がりの対角線を軸に、上辺右端が左辺下端に来るように反転";
        break;
    case nn::image::ExifOrientation_Rotate270:
        orientationStr = "時計回りに 270 度回転";
        break;
    case nn::image::ExifOrientation_FlipTopLeftToRightBottom:
        orientationStr = "左下がりの対角線を軸に、上辺左端が右辺下端に来るように反転";
        break;
    case nn::image::ExifOrientation_Rotate90:
        orientationStr = "時計回りに 90 度回転";
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    LogIfNotNull(" - 画像方向", orientationStr);
    LogIfNotNull(" - ソフトウェア名", metaData.software);
    LogIfNotNull(" - 撮影日時", metaData.dateTime);

    if (metaData.makerNote != nullptr)
    {
        NN_LOG(" - メーカーノート: %zu バイト\n", metaData.makerNoteSize);
    }
}

/// @brief 縮小されたピクセルデータを取得する関数
void GetDownsampledPixelData(
    Buffer *pOutPixelData,
    const nn::image::Dimension pixelDim,
    const Buffer &sourceData,
    const nn::image::Dimension sourceDim)
{
    // ピクセルデータの縮小 (本来であれば GPU ですべきですが、ここでは CPU で行っています。)
    uint8_t *sourceBuf = reinterpret_cast<uint8_t*>(sourceData.GetDataPtr());
    size_t sourceWidthAligned = sourceData.GetDataSize() / sourceDim.height / BytesPerPixel;
    uint8_t *pixelBuf = reinterpret_cast<uint8_t*>(pOutPixelData->GetDataPtr());
    size_t dstWidthAligned = pOutPixelData->GetDataSize() / pixelDim.height / BytesPerPixel;
    int scale = sourceDim.width / pixelDim.width;
    for (int y = 0; y < pixelDim.height; y++)
    {
        for (int x = 0; x < pixelDim.width; x++)
        {
            uint8_t *source = sourceBuf + (y * scale * sourceWidthAligned + x * scale) * BytesPerPixel;
            uint8_t *pixel = pixelBuf + (y * dstWidthAligned + x) * BytesPerPixel;
            // [入力例]
            // サムネイルだということを明示するためにチャンネルを変えて作成
            pixel[0] = source[2]; // R <- B
            pixel[1] = source[0]; // G <- R
            pixel[2] = source[1]; // B <- G
            pixel[3] = source[3]; // A
        }
    }
}

/// @brief 指定された品質設定でエンコードを試す関数
bool TryEncodeWithGivenQuality(
    size_t *pOutActualSize,
    Buffer *pOutJpeg,
    nn::image::JpegEncoder *pEncoder,
    const int quality)
{
    // エンコード品質の設定
    pEncoder->SetQuality(quality);

    // ピクセルデータの解析
    nn::image::JpegStatus jpegStatus = pEncoder->Analyze();
    NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok);

    // エンコード用のワークバッファ。 このワークバッファはサムネイルのエンコードが終われば解放可能です。
    Buffer workBuf(pEncoder->GetAnalyzedWorkBufferSize());

    // エンコード処理
    jpegStatus = pEncoder->Encode(
        pOutActualSize,
        pOutJpeg->GetDataPtr(), pOutJpeg->GetDataSize(),
        workBuf.GetDataPtr(), workBuf.GetDataSize());
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_ShortOutput:
        return false;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラー
    }
    return true;
}

/// @brief サムネイル JPEG を生成する関数
bool EncodeThumbnailData(
    size_t *pOutActualSize,
    Buffer *pOutJpeg,
    const Buffer &sourceData,
    const nn::image::Dimension &sourceDim,
    const int scale,
    const int alignmentSize)
{
    // サムネイル画像のサイズを計算
    nn::image::Dimension dim = {sourceDim.width / scale, sourceDim.height / scale};
    nn::image::Dimension dimAligned = {
        dim.width % alignmentSize == 0? dim.width: dim.width - dim.width % alignmentSize + alignmentSize,
        dim.height};

    // サムネイル用ピクセルデータ。このワークバッファはサムネイルのエンコードが終われば解放可能です。
    Buffer pixelData(dimAligned.width * dimAligned.height * BytesPerPixel);

    // ピクセルデータの縮小
    GetDownsampledPixelData(&pixelData, dim, sourceData, sourceDim);

    // エンコーダのインスタンス
    nn::image::JpegEncoder encoder;

    // ピクセルデータの設定
    encoder.SetPixelData(pixelData.GetDataPtr(), dimAligned, alignmentSize);

    // 例えば、バッファに収まるまで画質パラメータをいじりながらリトライできます。
    static const int LowestQuality = 1;

    // [入力例]
    // 画質設定 (100 からスタート, サンプリング比はデフォルトの 4:2:0 で固定)
    int lastQuality = 100;
    while (!TryEncodeWithGivenQuality(pOutActualSize, pOutJpeg, &encoder, lastQuality))
    {
        if (lastQuality <= LowestQuality)
        {
            // これ以上品質を下げることができない場合、エラーとします。
            NN_LOG("JPEG データを書き込むバッファが不足しました。 サムネイル画像の作成を中止します。\n");
            return false;
        }
        // LowestQuality (=1) を下限として、品質を下げてリトライします。
        lastQuality = std::max(lastQuality - 7, LowestQuality);
        NN_LOG("JPEG データを書き込むバッファが不足しました。 画質を %d に下げてリトライします。\n", lastQuality);
    }
    // pixelData はここ以降で解放することができます。

    NN_LOG(" - サムネイル画像: 幅 %zu px, 高さ %zu px, 画質 %d, %zu バイト\n", dim.width, dim.height, lastQuality, *pOutActualSize);
    return true;
}

/// @brief 適当なメタデータを設定します。
void SetupExifBuilder(
    nn::image::ExifBuilder *pOutExifBuilder,
    Buffer *pThumbnail,
    const Buffer &sourceData,
    const nn::image::Dimension &sourceDim)
{
    NN_LOG("Exif 情報を設定します。\n");

    // [入力例]
    // メタデータを設定します。 文字列やバイナリ列はコピーされずポインタが設定されるだけなので、誤って領域を解放しないよう注意します。
    static const char software[] = "Nintendo SDK Sample Program for JPEG library";
    static char dateTime[20] = "2015:08:07 09:00:00";
    static const nn::Bit8 makerNote[] = {0xBE, 0xEF};
    static const char uniqueId[] = "0123456789ABCDEF0123456789abcdef";

    MetaData metaData = {};
    metaData.orientation = nn::image::ExifOrientation_FlipVertical; // 上下反転
    metaData.software = software; // 撮影ソフトウェア名
    metaData.softwareSize = sizeof(software);
    metaData.dateTime = dateTime; // 撮影日時
    metaData.makerNote = makerNote; // メーカーノート
    metaData.makerNoteSize = sizeof(makerNote);
    metaData.uniqueId = uniqueId; // ユニークID
    PrintMetaData(metaData);

    pOutExifBuilder->SetOrientation(metaData.orientation);
    pOutExifBuilder->SetSoftware(metaData.software, metaData.softwareSize);
    pOutExifBuilder->SetDateTime(metaData.dateTime, 20);
    pOutExifBuilder->SetMakerNote(metaData.makerNote, metaData.makerNoteSize);
    pOutExifBuilder->SetUniqueId(metaData.uniqueId, 33);

    // [入力例]
    // サムネイル画像の生成 (アラインメントなし (1))
    size_t thumbSize;
    // アプリケーションがサムネイルとして表示するサイズに合わせます。(例として長辺 120 px)
    int scale = std::max(1, std::max(sourceDim.width / 120, sourceDim.height / 120));
    if (!EncodeThumbnailData(&thumbSize, pThumbnail, sourceData, sourceDim, scale, 1))
    {
        NN_LOG("サムネイル画像を生成できませんでした。省略します。\n");
    }
    else
    {
        pOutExifBuilder->SetThumbnail(pThumbnail->GetDataPtr(), thumbSize);
    }

    // サムネイルピクセルデータの解析
    nn::image::JpegStatus exifStatus = pOutExifBuilder->Analyze();
    switch (exifStatus)
    {
    case nn::image::JpegStatus_OutputOverabounds:
        // この場合、ExifBuilder.IsBuildable() が false となります。
        NN_LOG("メタデータの設定しすぎで Exif 情報のサイズ上限 (65527 バイト) を超過しています。\n");
        return;
    default:
        NN_ASSERT(exifStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }
}

/// @brief メタデータとサムネイルを付与しつつ、ピクセルデータを JPEG エンコードする関数
void EncodePixelData(
    size_t *pOutActualSize,
    Buffer *pOutJpeg,
    const Buffer &pixelData,
    const nn::image::Dimension dim,
    const int alignmentSize)
{
    /* ----------------------------------------------------------------------------------------
        Exif 情報構築フェーズ
        (メタデータとサムネイルのどちらも使用しない場合は読み飛ばしてください。)
     */
    Buffer thumbJpeg(32 * 1024); // サムネイル用に適当な大きさのメモリを予約します。 このバッファは、元データの JPEG エンコードが終わるまで解放してはいけません。
    Buffer exifWorkBuf(nn::image::ExifBuilder::GetWorkBufferSize()); // このバッファは、元データの JPEG エンコードが終わるまで解放してはいけません。
    nn::image::ExifBuilder builder(exifWorkBuf.GetDataPtr(), exifWorkBuf.GetDataSize());
    // ExifBuilder を適当なメタデータでセットアップします。
    SetupExifBuilder(&builder, &thumbJpeg, pixelData, dim);

    /* ----------------------------------------------------------------------------------------
        エンコード設定フェーズ
     */
    // エンコーダのインスタンス
    nn::image::JpegEncoder encoder;

    // ピクセルデータを設定します。
    encoder.SetPixelData(pixelData.GetDataPtr(), dim, alignmentSize);

    // [入力例]
    // 画質関係を設定します。(最高画質 100, サンプリング比 4:4:4)
    NN_LOG("エンコード設定\n");
    NN_LOG(" - 画質パラメータ: 100\n");
    NN_LOG(" - YCbCrサンプリング比: 4:4:4\n");
    encoder.SetQuality(100);
    encoder.SetSamplingRatio(nn::image::JpegSamplingRatio_444);

    /* ----------------------------------------------------------------------------------------
        ピクセルデータ解析フェーズ
     */
    // ピクセルデータを解析します。
    nn::image::JpegStatus jpegStatus = encoder.Analyze();
    NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 現時点では OK のみ返ります。

    /* ----------------------------------------------------------------------------------------
        JPEG エンコードフェーズ
     */
    NN_LOG("JPEG データにエンコードします。\n");

    // エンコード用のワークバッファの確保。このバッファは JPEG エンコードが終われば解放可能です。
    Buffer workBuf(encoder.GetAnalyzedWorkBufferSize());
    NN_LOG(" - 必要な作業メモリ量: %zu bytes\n", workBuf.GetDataSize());

    // エンコード処理
    if (builder.IsBuildable())
    {
        // もし IsBuildable() == true な ExifBuilder があれば、Exif 情報を設定してエンコードします。
        jpegStatus = encoder.Encode(
            pOutActualSize,
            pOutJpeg->GetDataPtr(), pOutJpeg->GetDataSize(),
            workBuf.GetDataPtr(), workBuf.GetDataSize(),
            &builder); // Exif 情報を設定
    }
    else
    {
        // Exif 情報構築フェーズでエラーが起きていれば、 IsBuildable() は false となります。
        // その場合は Exif 情報を設定せずエンコードします。
        jpegStatus = encoder.Encode(
            pOutActualSize,
            pOutJpeg->GetDataPtr(), pOutJpeg->GetDataSize(),
            workBuf.GetDataPtr(), workBuf.GetDataSize());
    }
    // サムネイルと ExifBuilder 用のワークメモリ、 JpegEncoder 用のワークメモリはここで解放できます。
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_ShortOutput:
        NN_LOG("JPEG データを書き込むバッファが不足しました。\n");
        return;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラー
        NN_LOG("出力された JPEG データ: 幅 %zu px, 高さ %zu px, %zu バイト\n", dim.width, dim.height, *pOutActualSize);
    }
}
}

extern "C" void nnMain()
{
    // ファイルシステムの初期化
    nn::fs::SetAllocator(Allocate, Deallocate);

    // PC 上のファイルシステムをマウント
    // NOTE: host としてマウントするディレクトリは適宜設定してください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("Asset", "C:\\JpegInputDir"));

    // 例: 幅の 4 byte アラインが必要なテクスチャをエンコードする場合
    static const int AlignmentSize = 4; // アラインメント
    static const nn::image::Dimension dim = {639, 480}; // パディングの発生のためにキリの悪い 639 にしています。
    static const nn::image::Dimension dimAligned = {
        (dim.width % AlignmentSize == 0)? dim.width: dim.width - dim.width % AlignmentSize + AlignmentSize,
        dim.height}; // アライン済みサイズ

    // このバッファは Jpeg エンコードが終われば解放可能。
    Buffer pixelData(dimAligned.width * dimAligned.height * BytesPerPixel);

    // カラーバッファの初期化
    uint8_t *pixelBuf = reinterpret_cast<uint8_t*>(pixelData.GetDataPtr());
    for (int y = 0; y < dim.height; y++)
    {
        const uint8_t R = static_cast<uint8_t>(((float)y / dim.height * 0xFF) + 0.5f);
        for (int x = 0; x < dim.width; x++)
        {
            const uint8_t G = static_cast<uint8_t>(((float)x / dim.width) * 0xFF + 0.5f);
            const uint8_t B = static_cast<uint8_t>(
                (std::sqrt(x * x + y * y) / sqrt(dim.width * dim.width + dim.height * dim.height))
                * 0xFF + 0.5f);

            // エンディアンに関係なく、R, G, B, A で並んでいること。
            uint8_t *pixel = pixelBuf + (x + y * dimAligned.width) * BytesPerPixel;
            pixel[0] = R;
            pixel[1] = G;
            pixel[2] = B;
            pixel[3] = 0xFF; // A
        }
    }

    // JPEG エンコード処理の実体
    size_t actualSize;
    Buffer jpegData(1024 * 1024); // JPEG データ用に (適当に) 1 MiB 予約する。
    EncodePixelData(&actualSize, &jpegData, pixelData, dim, AlignmentSize);
    pixelData.Release(); // サンプルのため明示的に解放。

    FileData::Save("Asset:/output.jpg", jpegData.GetDataPtr(), actualSize);
    jpegData.Release(); // サンプルのため明示的に解放。

    // FS 終了処理
    nn::fs::Unmount("Asset");
    return;
}
