﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HtcConnectionEvent.cpp,PageSampleHtcConnectionEvent}
 *
 * @brief
 *  ホストとの通信状態を取得するサンプルプログラム
 */

/**
 * @page PageSampleHtcConnectionEvent Event サンプルプログラム
 * @tableofcontents
 *
 * @brief
 *  ホストとの通信状態を取得するサンプルプログラムの解説です。
 *
 * @section PageSampleHtcConnectionEvent_SectionBrief 概要
 *  ここでは、ホストとの通信状態を取得する API を使ったサンプルプログラムの説明をします。
 *
 *  API の使い方については、HTC ライブラリマニュアル および
 *  @ref nn::htc "HTC の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleHtcConnectionEvent_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HtcConnectionEvent Samples/Sources/Applications/HtcConnectionEvent @endlink 以下にあります。
 *
 * @section PageSampleHtcConnectionEvent_SectionNecessaryEnvironment 必要な環境
 *  Target Manager がインストールされている必要があります。
 *
 * @section PageSampleHtcConnectionEvent_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleHtcConnectionEvent_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleHtcConnectionEvent_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleHtcConnectionEvent_SectionDetail 解説
 *
 * @subsection PageSampleHtcConnectionEvent_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HtcConnectionEvent.cpp
 *  @includelineno HtcConnectionEvent.cpp
 *
 * @subsection PageSampleHtcConnectionEvent_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nn::htc::BindHostConnectionEvent(), nn::htc::BindHostDisconnectionEvent() で Targat Manager との接続通知イベントと切断通知イベントを取得
 *  - 接続通知イベントがシグナルされるまで待機
 *  - ホストと接続中に nn::htc::GetEnvironmentVariableLength() でホストから環境変数のサイズを取得
 *  - 切断通知イベントがシグナルされるまで待機
 *
 *  本サンプルでは Target Manager との接続状態に従って動作するようになっていますので、必要に応じて Target Manager から接続／切断操作を行って確認してください。
 *
 *  このサンプルの実行結果の例を以下に示します。
 *
 *  @verbinclude  HtcConnectionEvent_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/htc.h>

//-----------------------------------------------------------------------------

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result      result;
    size_t length;

    nn::htc::Initialize();

    // Target Manager と接続されたことを通知するイベントを取得します
    nn::os::SystemEvent connectionEvent;
    nn::htc::BindHostConnectionEvent(&connectionEvent);

    // Target Manager から切断されたことを通知するイベントを取得します
    nn::os::SystemEvent disconnectionEvent;
    nn::htc::BindHostDisconnectionEvent(&disconnectionEvent);

    // Target Manager との接続を待機します
    NN_LOG("Waiting for connection from Host.\n");
    connectionEvent.Wait();

    // ホスト PC から環境変数「NINTENDO_SDK_ROOT」のサイズを取得します
    result = nn::htc::GetEnvironmentVariableLength(&length, "NINTENDO_SDK_ROOT");
    if (result.IsSuccess())
    {
        NN_LOG("length = %d\n", length);
    }
    else
    {
        NN_LOG("GetEnvironmentVariableLength failed (%08x)\n", result.GetInnerValueForDebug());
    }

    // Target Manager からの切断を待機します
    NN_LOG("Waiting for disconnection from Host.\n");
    disconnectionEvent.Wait();

    nn::htc::Finalize();
}

