﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hidbus/hidbus.h>

#include <nns/hidbus/hidbus_Ronde.h>

#include "HidbusRondeHardwareTestTool_Ronde.h"

namespace {

    HidbusState state = HidbusState_NotGotHandle;

    RondeInfo g_RondeInfo;

    struct Npad
    {
        typedef nn::hid::NpadIdType Id;
        typedef nn::hid::NpadButton Button;
    };

    Npad::Id npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
    };

    nn::hidbus::BusHandle rightJoyHandle;

    // PollingMode 時に渡すバッファ (handle ごとに別のバッファが必要です。)
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_Buffer[nn::os::MemoryPageSize];

    HidbusState HandleCommunicationErrorResult(nn::Result result, HidbusState currentState)
    {
        if (nn::hidbus::ResultExternalDeviceTimeout::Includes(result))
        {
            // 拡張バスに接続されたデバイスとの通信中にタイムアウトが発生しました。
            // リトライしてください。
            NN_LOG("Timeout Occur. Retry\n");
            return currentState;
        }
        else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(result))
        {
            // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
            // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
            // 再度 Enable からやり直してください。
            NN_LOG("NotEnabled. \n");
            return HidbusState_GotHandle;
        }
        else if (nn::hidbus::ResultInvalidHandle::Includes(result))
        {
            // handle に対応したコントローラーが切断されており、 handle が無効化されています。
            // 再度 handle の取得からやり直してください。
            NN_LOG("Invalid Handle. \n");
            return HidbusState_NotGotHandle;
        }
        else
        {
            // 通信コマンド送受信時は他の Result は返りません。
            NN_ABORT("Unexpected Error\n");
        }
    }

    HidbusState ViewRondeInfomation(nn::hidbus::BusHandle handle, HidbusState currentState)
    {
        // 各種情報を取得し、表示します。
        uint8_t uniqueId[nns::hidbus::RondeUniqueIdSize];
        nns::hidbus::RondeManufactureCalibrationData calData;
        nns::hidbus::RondeFwVersion fwVersion;
        HidbusState tempState;
        while (NN_STATIC_CONDITION(true))
        {
            // UniqueID の取得
            auto uniqueIdResult = nns::hidbus::GetRondeUniqueId(uniqueId, sizeof(uniqueId), handle);
            if (uniqueIdResult.IsSuccess())
            {
                break;
            }
            else
            {
                tempState = HandleCommunicationErrorResult(uniqueIdResult, currentState);
                if (tempState != currentState)
                {
                    return tempState;
                }
            }
        }

        while (NN_STATIC_CONDITION(true))
        {
            // 工程キャルの取得
            auto calibrationResult = nns::hidbus::GetRondeManufactureCalibrationData(&calData, handle);
            if (calibrationResult.IsSuccess())
            {
                break;
            }
            else
            {
                tempState = HandleCommunicationErrorResult(calibrationResult, currentState);
                if (tempState != currentState)
                {
                    return tempState;
                }
            }
        }

        while (NN_STATIC_CONDITION(true))
        {
            // ファームウェアバージョンの取得
            auto firmwareVersionResult = nns::hidbus::GetRondeFirmwareVersion(&fwVersion, handle);
            if (firmwareVersionResult.IsSuccess())
            {
                break;
            }
            else
            {
                tempState = HandleCommunicationErrorResult(firmwareVersionResult, currentState);
                if (tempState != currentState)
                {
                    return tempState;
                }
            }
        }

        NN_LOG("-------------- Ronde Info ---------------\n");
        NN_LOG("FW Version = %x.%x\n", fwVersion.mainVersion, fwVersion.subVersion);
        NN_LOG("Unique ID = ");
        for (int i = nns::hidbus::RondeUniqueIdSize - 1; i >= 0; --i)
        {
            NN_LOG("%02x", uniqueId[i]);
        }
        NN_LOG("\n");

        NN_LOG("Manufacture Cal : \n");
        NN_LOG(" osMax   = %d\n", calData.osMax);
        NN_LOG(" hkMax   = %d\n", calData.hkMax);
        NN_LOG(" zeroMax = %d\n", calData.zeroMax);
        NN_LOG(" zeroMin = %d\n", calData.zeroMin);
        NN_LOG("-----------------------------------------\n");

        memcpy(g_RondeInfo.uniqueId, uniqueId, 12);
        g_RondeInfo.maxPush = calData.osMax;
        g_RondeInfo.maxPull = calData.hkMax;
        g_RondeInfo.zeroMax = calData.zeroMax;
        g_RondeInfo.zeroMin = calData.zeroMin;
        g_RondeInfo.fwVer = fwVersion.mainVersion << 8 | fwVersion.subVersion;

        return currentState;
    }


    HidbusState WaitSupportedJoyConConnected(nn::hidbus::BusHandle* pOutHandle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            // RightJoyRail を持つコントローラーが接続されるまで待ちます。
            while (!nn::hidbus::GetBusHandle(pOutHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
            {
                NN_LOG("Please connect supported controller.\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            // 対応するコントローラーが接続されたら、ライブラリを初期化します。
            auto result = nn::hidbus::Initialize(*pOutHandle);
            if (result.IsSuccess())
            {
                return HidbusState_GotHandle;
            }
            // Initialize に失敗した時はコントローラーが切断される等によって handle が無効化されているため、
            // 再度 handle 取得からやり直します。
        }
    }

    HidbusState EnableDevice(nn::hidbus::BusHandle handle, char* pBuffer, size_t bufferSize)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto enableResult = nns::hidbus::EnableRonde(handle, pBuffer, bufferSize);
            if (enableResult.IsSuccess())
            {
                // Unique ID とキャル値を表示します。
                auto handledState = ViewRondeInfomation(handle, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }

                return HidbusState_Enabled;
            }
            else if (nn::hidbus::ResultFWUpdateFailed::Includes(enableResult))
            {
                // コントローラーの Firmware Update が発生したが失敗しました。
                // リトライするか、アプリ側で Firmware Update アプレットを立ち上げてください。
                // 本サンプルではリトライします。
                NN_LOG("Firmware Update Failed. retry.\n");
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(enableResult))
            {
                // コントローラの切断等が発生し、handle が無効になっています。handle 取得からリトライしてください。
                NN_LOG("Handle is invalid.\n");
                return HidbusState_NotGotHandle;
            }
            else if (nn::hidbus::ResultNotInForcus::Includes(enableResult))
            {
                // アプリケーションが InForcus 状態でないため、拡張デバイスを有効化できません。
                // アプリケーション側で nn::oe のライブラリ群を使用してアプリケーションが InForcus になるのを待ってください。
                // 本サンプルではリトライします。
                NN_LOG("Not InForcus.\n");
            }
            else if (nn::hidbus::ResultActivateFailureNpadBusy::Includes(enableResult))
            {
                // 指定された Npad が拡張バス機能と排他で使用されるべき機能を使用しているため、有効化できませんでした。
                // アプリケーション側で拡張バスと排他で使用されるべき機能を終了して、再度本関数を実行してください。
                NN_LOG("Please deactivate the dependency function\n");
            }
            else if (nn::hidbus::ResultExternalDeviceNotAttached::Includes(enableResult))
            {
                // 拡張バスにデバイスが接続されていません。
                // 拡張バスにデバイスが接続されるまでリトライします。
                NN_LOG("Waiting External Device Connected..\n");
            }
            else if (nn::hidbus::ResultExternalDeviceReadyTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信準備中にタイムアウトしました。
                // 本サンプルではリトライします。
                NN_LOG("Communication Ready Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信中にタイムアウトしました。
                // ハンドリング方法はデバイスごとに異なりますが、本サンプルではリトライします。
                NN_LOG("Communication Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceInvalidId::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスが RONDE ではありません。
                // ユーザーに分かりやすいような表示を行ってください。本サンプルではリトライします。
                NN_LOG("Attached device is not RONDE. Please attach RONDE.\n");
            }
            else if(nn::hidbus::ResultExternalDeviceNotEnabled::Includes(enableResult))
            {
                // nns::hidbus::EnableRonde では内部で PollingMode の Enable をしているため、
                // NotEnabled が返る可能性がある。
                // この時はリトライする
                NN_LOG("NotEnabled. Please retry.\n");
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(enableResult);
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    HidbusState PollingReadRondeData(nn::hidbus::BusHandle handle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            nns::hidbus::RondeSensorData sensorData;
            auto result = nns::hidbus::GetRondeSensorData(&sensorData, handle);
            if (result.IsSuccess())
            {
                NN_LOG("Ronde Sensor Value = %d ", sensorData.data);
                g_RondeInfo.sensor = sensorData.data;
                g_RondeInfo.samplingNumber = sensorData.samplingNumber;
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(result, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }
            }

            int16_t thermistor;
            auto thermistorResult = nns::hidbus::GetRondeThermisterData(&thermistor, handle);
            if (thermistorResult.IsSuccess())
            {
                NN_LOG("Thermistor = %d\n", thermistor);
                g_RondeInfo.thermistor = thermistor;
            }
            else
            {
                auto handledState = HandleCommunicationErrorResult(thermistorResult, HidbusState_Enabled);
                if (handledState != HidbusState_Enabled)
                {
                    return handledState;
                }
            }

            // エラー情報の取得
            nns::hidbus::GetRondeInternalErrorInfo(&g_RondeInfo.errorInfo);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
    }

}//namespace

void RondeThread(void* arg)
{
    NN_UNUSED(arg);
    g_RondeInfo.state = HidbusState_NotGotHandle;

    nn::hid::InitializeNpad();

    // Npadのスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleJoyLeft::Mask |
        ::nn::hid::NpadStyleJoyRight::Mask
    );

    // 使用するNpadId の設定
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    // 全体のループ
    while (NN_STATIC_CONDITION(true))
    {
        switch (state)
        {
        case HidbusState_NotGotHandle:
            state = WaitSupportedJoyConConnected(&rightJoyHandle);
            g_RondeInfo.state = state;
            break;

        case HidbusState_GotHandle:
            state = EnableDevice(rightJoyHandle, s_Buffer, sizeof(s_Buffer));
            g_RondeInfo.state = state;
            break;

        case HidbusState_Enabled:
            state = PollingReadRondeData(rightJoyHandle);
            g_RondeInfo.state = state;
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // RONDE の無効化
    nns::hidbus::DisableRonde(rightJoyHandle);

    // 拡張デバイス機能の終了
    nn::hidbus::Finalize(rightJoyHandle);
}

RondeInfo GetRondeInfo()
{
    return g_RondeInfo;
}
