﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
    @examplesource{HidXpadBasic.cpp,PageSampleHidXpadBasic}

    @brief
    Xpadを使ってボタン入力を取得するためのサンプルプログラム
*/

/**
    @page PageSampleHidXpadBasic Xpadを使ったボタン入力の取得
    @tableofcontents

    @brief
    Xpadを使ってボタン入力を取得するためのサンプルプログラムの解説です。

    @section PageSampleHidXpadBasic_SectionBrief 概要
    コントローラの最新のボタン入力を取得する方法について説明します。

    @section PageSampleHidXpadBasic_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidXpadBasic @endlink 以下にあります。

    @section PageSampleHidXpadBasic_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV 環境で動作させる場合は、事前に SDEV とコントローラをペアリングしてください。

    @section PageSampleHidXpadBasic_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    接続後、A,B,X,Y,L,R,ZL,ZR,StickL,StickR,上下左右,+,-,SL,SRのいずれかのボタンを押すと、コンソール（Nintendo Target Manager）にログが出力されます。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。
    Joy-Con (L) だけ、もしくは Joy-Con (R) だけが接続された場合はVisualStudioかNintendo Target Manager経由で終了させてください。

    @section PageSampleHidXpadBasic_SectionPrecaution 注意事項
    本サンプルプログラムでは画面描画は行いません。

    @section PageSampleHidXpadBasic_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    Joy-Con (L) だけ、もしくは Joy-Con (R) だけが接続された場合、JoyXpadを使って入力状態を取得します。
    Joy-Con (L) および Joy-Con (R) がセットで接続された場合、JoyXpadとBasicXpadを使って入力状態を取得します。

    @section PageSampleHidXpadBasic_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - XpadID を取得
    - Xpad を初期化
    - 現在のボタン入力状態を取得
*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid.h>
#include <nn/hid/hid_JoyXpad.h>

namespace
{
    bool    waits = true;

    int basicXpadIdCount = 0;
    nn::hid::BasicXpadId basicXpadIds[nn::hid::XpadIdCountMax];
    nn::hid::BasicXpadState oldBasicXpadState[nn::hid::XpadIdCountMax];
    nn::hid::BasicXpadState currentBasicXpadState[nn::hid::XpadIdCountMax];

    int joyXpadIdCount = 0;
    nn::hid::JoyXpadId joyXpadIds[nn::hid::XpadIdCountMax];
    nn::hid::JoyXpadState oldJoyXpadState[nn::hid::XpadIdCountMax * 2];
    nn::hid::JoyXpadState currentJoyXpadState[nn::hid::XpadIdCountMax * 2];

    void PrintBasicXpadState(const nn::hid::BasicXpadState& state)
    {
        char buttons[21];
        buttons[0]   = (state.buttons.Test<nn::hid::BasicXpadButton::A>())         ? 'A' : '_';
        buttons[1]   = (state.buttons.Test<nn::hid::BasicXpadButton::B>())         ? 'B' : '_';
        buttons[2]   = (state.buttons.Test<nn::hid::BasicXpadButton::X>())         ? 'X' : '_';
        buttons[3]   = (state.buttons.Test<nn::hid::BasicXpadButton::Y>())         ? 'Y' : '_';
        buttons[4]   = (state.buttons.Test<nn::hid::BasicXpadButton::StickL>())    ? 'L' : '_';
        buttons[5]   = (state.buttons.Test<nn::hid::BasicXpadButton::StickL>())    ? 'S' : '_';
        buttons[6]   = (state.buttons.Test<nn::hid::BasicXpadButton::StickR>())    ? 'R' : '_';
        buttons[7]   = (state.buttons.Test<nn::hid::BasicXpadButton::StickR>())    ? 'S' : '_';
        buttons[8]   = (state.buttons.Test<nn::hid::BasicXpadButton::L>())         ? 'L' : '_';
        buttons[9]   = (state.buttons.Test<nn::hid::BasicXpadButton::R>())         ? 'R' : '_';
        buttons[10]  = (state.buttons.Test<nn::hid::BasicXpadButton::ZL>())        ? 'Z' : '_';
        buttons[11]  = (state.buttons.Test<nn::hid::BasicXpadButton::ZL>())        ? 'L' : '_';
        buttons[12]  = (state.buttons.Test<nn::hid::BasicXpadButton::ZR>())        ? 'Z' : '_';
        buttons[13]  = (state.buttons.Test<nn::hid::BasicXpadButton::ZR>())        ? 'R' : '_';
        buttons[14]  = (state.buttons.Test<nn::hid::BasicXpadButton::Start>())     ? '+' : '_';
        buttons[15]  = (state.buttons.Test<nn::hid::BasicXpadButton::Select>())    ? '-' : '_';
        buttons[16]  = (state.buttons.Test<nn::hid::BasicXpadButton::Left>())      ? '<' : '_';
        buttons[17]  = (state.buttons.Test<nn::hid::BasicXpadButton::Up>())        ? '^' : '_';
        buttons[18]  = (state.buttons.Test<nn::hid::BasicXpadButton::Right>())     ? '>' : '_';
        buttons[19]  = (state.buttons.Test<nn::hid::BasicXpadButton::Down>())      ? 'v' : '_';
        buttons[20]  = '\0';

        NN_LOG("[%6lld] %s\n", state.samplingNumber, buttons);
    }

    void PrintJoyXpadState(const nn::hid::JoyXpadState& state)
    {
        char buttons[29];
        buttons[0]   = (state.buttons.Test<nn::hid::JoyXpadButton::A>())         ? 'A' : '_';
        buttons[1]   = (state.buttons.Test<nn::hid::JoyXpadButton::B>())         ? 'B' : '_';
        buttons[2]   = (state.buttons.Test<nn::hid::JoyXpadButton::X>())         ? 'X' : '_';
        buttons[3]   = (state.buttons.Test<nn::hid::JoyXpadButton::Y>())         ? 'Y' : '_';
        buttons[4]   = (state.buttons.Test<nn::hid::JoyXpadButton::StickL>())    ? 'L' : '_';
        buttons[5]   = (state.buttons.Test<nn::hid::JoyXpadButton::StickL>())    ? 'S' : '_';
        buttons[6]   = (state.buttons.Test<nn::hid::JoyXpadButton::StickR>())    ? 'R' : '_';
        buttons[7]   = (state.buttons.Test<nn::hid::JoyXpadButton::StickR>())    ? 'S' : '_';
        buttons[8]   = (state.buttons.Test<nn::hid::JoyXpadButton::L>())         ? 'L' : '_';
        buttons[9]   = (state.buttons.Test<nn::hid::JoyXpadButton::R>())         ? 'R' : '_';
        buttons[10]  = (state.buttons.Test<nn::hid::JoyXpadButton::ZL>())        ? 'Z' : '_';
        buttons[11]  = (state.buttons.Test<nn::hid::JoyXpadButton::ZL>())        ? 'L' : '_';
        buttons[12]  = (state.buttons.Test<nn::hid::JoyXpadButton::ZR>())        ? 'Z' : '_';
        buttons[13]  = (state.buttons.Test<nn::hid::JoyXpadButton::ZR>())        ? 'R' : '_';
        buttons[14]  = (state.buttons.Test<nn::hid::JoyXpadButton::Plus>())      ? '+' : '_';
        buttons[15]  = (state.buttons.Test<nn::hid::JoyXpadButton::Minus>())     ? '-' : '_';
        buttons[16]  = (state.buttons.Test<nn::hid::JoyXpadButton::Left>())      ? '<' : '_';
        buttons[17]  = (state.buttons.Test<nn::hid::JoyXpadButton::Up>())        ? '^' : '_';
        buttons[18]  = (state.buttons.Test<nn::hid::JoyXpadButton::Right>())     ? '>' : '_';
        buttons[19]  = (state.buttons.Test<nn::hid::JoyXpadButton::Down>())      ? 'v' : '_';
        buttons[20]  = (state.buttons.Test<nn::hid::JoyXpadButton::LeftSL>())    ? 'S' : '_';
        buttons[21]  = (state.buttons.Test<nn::hid::JoyXpadButton::LeftSL>())    ? 'L' : '_';
        buttons[22]  = (state.buttons.Test<nn::hid::JoyXpadButton::LeftSR>())    ? 'S' : '_';
        buttons[23]  = (state.buttons.Test<nn::hid::JoyXpadButton::LeftSR>())    ? 'R' : '_';
        buttons[24]  = (state.buttons.Test<nn::hid::JoyXpadButton::RightSL>())   ? 'S' : '_';
        buttons[25]  = (state.buttons.Test<nn::hid::JoyXpadButton::RightSL>())   ? 'L' : '_';
        buttons[26]  = (state.buttons.Test<nn::hid::JoyXpadButton::RightSR>())   ? 'S' : '_';
        buttons[27]  = (state.buttons.Test<nn::hid::JoyXpadButton::RightSR>())   ? 'R' : '_';
        buttons[28]  = '\0';

        NN_LOG("[%6lld] %s\n", state.samplingNumber, buttons);
    }

    void BasicXpadInit()
    {
        //BasicXpadの取得
        basicXpadIdCount = nn::hid::GetXpadIds(basicXpadIds, sizeof(basicXpadIds) / sizeof(basicXpadIds[0]));
        for (int i = 0; i < basicXpadIdCount; ++i)
        {
            //Xpadの初期化
            nn::hid::InitializeXpad(basicXpadIds[i]);
        }
    }

    void JoyXpadInit()
    {
        //JoyXpadIDの取得
        joyXpadIdCount = nn::hid::GetXpadIds(joyXpadIds, sizeof(joyXpadIds) / sizeof(joyXpadIds[0]));

        for (int i = 0; i < joyXpadIdCount; ++i)
        {
            //Xpadの初期化
            nn::hid::InitializeXpad(joyXpadIds[i]);
        }
    }

    void BasicXpadUpdate()
    {
        for(int i = 0; i < basicXpadIdCount; i++)
        {
            oldBasicXpadState[i] = currentBasicXpadState[i];

            //最新のXpadのステートを取得
            nn::hid::GetXpadState(&(currentBasicXpadState[i]), basicXpadIds[i]);

            //ボタンが押された
            if((currentBasicXpadState[i].buttons ^ oldBasicXpadState[i].buttons & currentBasicXpadState[i].buttons).IsAnyOn())
            {
                //ボタンのクリック状態を表示
                NN_LOG("BasicXpad (%d)     ",i);
                PrintBasicXpadState(currentBasicXpadState[i]);
            }

            // ＋ と - ボタンを同時に押されたら終了
            if (currentBasicXpadState[i].buttons.Test<nn::hid::BasicXpadButton::Start>() && currentBasicXpadState[i].buttons.Test<nn::hid::BasicXpadButton::Select>() )
            {
                waits = false;
            }
        }
    }

    void JoyXpadUpdate()
    {
        for(int i = 0; i < joyXpadIdCount; i++)
        {
            int id = 2 * i;
            oldJoyXpadState[id] = currentJoyXpadState[id];

            //最新の Joy-Con (R) のステートを取得
            nn::hid::GetJoyXpadRightState(&(currentJoyXpadState[id]), joyXpadIds[i]);

            //ボタンが押された
            if((currentJoyXpadState[id].buttons ^ oldJoyXpadState[id].buttons & currentJoyXpadState[id].buttons).IsAnyOn())
            {
                //ボタンのクリック状態を表示
                NN_LOG("JoyXpad Right (%d) ",i);
                PrintJoyXpadState(currentJoyXpadState[id]);
            }

            id++;
            oldJoyXpadState[id] = currentJoyXpadState[id];

            //最新の Joy-Con (L) のステートを取得
            nn::hid::GetJoyXpadLeftState(&(currentJoyXpadState[id]), joyXpadIds[i]);

            //ボタンが押された
            if((currentJoyXpadState[id].buttons ^ oldJoyXpadState[id].buttons & currentJoyXpadState[id].buttons).IsAnyOn())
            {
                //ボタンのクリック状態を表示
                NN_LOG("JoyXpad Left  (%d) ",i);
                PrintJoyXpadState(currentJoyXpadState[id]);
            }

            id = 2 * i;
            // ＋ と - ボタンが同時に押されたら終了
            if (currentJoyXpadState[id].buttons.Test<nn::hid::JoyXpadButton::Plus>() && currentJoyXpadState[id + 1].buttons.Test<nn::hid::JoyXpadButton::Minus>())
            {
                waits = false;
            }

        }
    }

    void Init()
    {
        BasicXpadInit();
        JoyXpadInit();
    }

    void Update()
    {
        BasicXpadUpdate();
        JoyXpadUpdate();
    }

}//namespace

extern "C" void nnMain()
{
    NN_LOG("Xpad Sample Start.\n");

    Init();

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");

    while(waits)
    {
        Update();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    NN_LOG("Xpad Sample Done.\n");
}
