﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Vibration.h>

#include "Sphere.h"
#include "File.h"

struct VirtualBallDescriptor
{
    // 物体の特性
    float radius;               //!< 大きさ（球の半径）
    float weight;               //!< 重さ
    float coefStaticFriction;   //!< 静摩擦係数
    float coefDynamicFriction;  //!< 動摩擦係数
    float coefRestitutionWall;  //!< 反発係数（壁）
    float coefRestitutionBall;  //!< 反発係数（物体）

    //物体の表示設定
    bool isRolling;                                 //!< 回転しているかどうか
    nn::util::Unorm8x4 baseColor;                   //!< 物体の色
    Sphere::VertexColorPattern colorPattern;        //!< 物体の模様

    // 物体の床との振動特性
    float fieldVibrationAmplifierLimit;             //!< 振幅制限
    nn::hid::VibrationModulation fieldModulation;   //!< 変調設定
    FileLabel fieldVibrationPattern;                //!< 使用する振動ファイル

    // 物体の壁との振動特性
    float wallVibrationAmplifierLimit;              //!< 振幅制限
    nn::hid::VibrationModulation wallModulation;    //!< 変調設定
    FileLabel wallVibrationPattern;                 //!< 使用する振動ファイル
};

class VirtualBall
{
    NN_DISALLOW_COPY(VirtualBall);
    NN_DISALLOW_MOVE(VirtualBall);

public:
    VirtualBall() NN_NOEXCEPT;
    ~VirtualBall() NN_NOEXCEPT;

    nn::hid::VibrationTarget* GetVibrationTarget(int idx) NN_NOEXCEPT
    {
        if (idx == 0)
        {
            return &m_VibrationTargetL;
        }
        else
        {
            return &m_VibrationTargetR;
        }
    }

    void SetVibrationMixMode(nn::hid::VibrationMixMode mode) NN_NOEXCEPT;

    // 三軸の加速度情報を元に物体の状態を更新します。
    void Update(float gx, float gy, float gz) NN_NOEXCEPT;

    void Reset() NN_NOEXCEPT
    {
        m_Position = 0.0f;
        m_Velocity = 0.0f;
        m_StaticFriction = 0.0f;
        m_DynamicFriction = 0.0f;
        m_IsMoving = false;
        m_IsHitWall = false;
        m_HitPowerWall = 0.0f;

        m_VibrationHitWall.Stop();
        m_VibrationRolling.Stop();
    }

    void SetDescriptor(const VirtualBallDescriptor* desc) NN_NOEXCEPT
    {
        m_pDescriptor = desc;

        // 衝突範囲の更新
        m_FieldHalfWidthEfficient = m_FieldHalfWidth - m_pDescriptor->radius;

        m_VibrationNodeConnectionRollingL.SetModulation(m_pDescriptor->fieldModulation);
        m_VibrationNodeConnectionRollingR.SetModulation(m_pDescriptor->fieldModulation);
        m_VibrationNodeConnectionHitWallL.SetModulation(m_pDescriptor->wallModulation);
        m_VibrationNodeConnectionHitWallR.SetModulation(m_pDescriptor->wallModulation);

        // 振動ファイル設定
        auto wallData = File::GetInstance().GetBnvibFile(m_pDescriptor->wallVibrationPattern);
        auto fieldData = File::GetInstance().GetBnvibFile(m_pDescriptor->fieldVibrationPattern);
        m_VibrationHitWall.Load(wallData->data, wallData->size);
        m_VibrationRolling.Load(fieldData->data, fieldData->size);
    }

    const VirtualBallDescriptor* GetDesc() const NN_NOEXCEPT
    {
        return m_pDescriptor;
    }

    bool IsMoving() const NN_NOEXCEPT
    {
        return m_IsMoving;
    }

    bool IsHitWall() const NN_NOEXCEPT
    {
        return m_IsHitWall;
    }

    float GetHitPowerWall() const NN_NOEXCEPT
    {
        return m_HitPowerWall;
    }

    float GetPosition() const NN_NOEXCEPT
    {
        return m_Position;
    }

    float GetVelocity() const NN_NOEXCEPT
    {
        return m_Velocity;
    }

    // フィールドの特性
    void SetFieldWidth(float width) NN_NOEXCEPT
    {
        m_FieldWidth     = width;
        m_FieldHalfWidth = width / 2.0f;
    }

protected:
    void UpdateMoving(float force) NN_NOEXCEPT;
    void UpdateHitWall() NN_NOEXCEPT;

protected:
    // 物体の固定値
    const VirtualBallDescriptor* m_pDescriptor;

    // 物体の位置、速度など
    float m_Position        = 0.0f;  //!< 位置
    float m_Velocity        = 0.0f;  //!< 速度
    float m_StaticFriction  = 0.0f;  //!< 静摩擦
    float m_DynamicFriction = 0.0f;  //!< 動摩擦
    float m_HitPowerWall    = 0.0f;  //!< 衝突時のパワー
    bool  m_IsMoving        = false; //!< 移動中
    bool  m_IsHitWall       = false; //!< 壁に衝突

    // フィールドの特性
    float m_FieldWidth      = 0.0f;
    float m_FieldHalfWidth  = 0.0f;
    float m_FieldHalfWidthEfficient = 0.0f;

    // 振動関連
    nn::hid::VibrationPlayer m_VibrationHitWall;                        //!< 物体が壁に衝突したときの振動源
    nn::hid::VibrationPlayer m_VibrationRolling;                        //!< 物体が転がっているときの振動源
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionHitWallL; //!< 衝突時の振動をVibrationMixerへ出力する経路(左)
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionHitWallR; //!< 衝突時の振動をVibrationMixerへ出力する経路(右)
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionRollingL; //!< 転がりの振動をVibrationMixerへ出力する経路(左)
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionRollingR; //!< 転がりの振動をVibrationMixerへ出力する経路(右)
    nn::hid::VibrationMixer m_VibrationMixerL;                          //!< VibrationMixerは物体から発生する複数の振動源を合成する(左)
    nn::hid::VibrationMixer m_VibrationMixerR;                          //!< VibrationMixerは物体から発生する複数の振動源を合成する(右)
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionOutputL;  //!< VibrationMixerからVibrationTargetへ出力する経路(左)
    nn::hid::VibrationNodeConnection m_VibrationNodeConnectionOutputR;  //!< VibrationMixerからVibrationTargetへ出力する経路(右)
    nn::hid::VibrationTarget m_VibrationTargetL;                        //!< 左振動子への出力先
    nn::hid::VibrationTarget m_VibrationTargetR;                        //!< 右振動子への出力先
};
