﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Constant.h>
#include "Sphere.h"

namespace
{
    int   g_Counter = 0;
    float g_CosArray[Sphere::HorizontalDivisionNumber + 1];
    float g_SinArray[Sphere::HorizontalDivisionNumber + 1];
}

void Prepare() NN_NOEXCEPT
{
    for (int i = 0; i <= Sphere::HorizontalDivisionNumber; i++)
    {
        g_CosArray[i] = cosf((i * 1.0f / Sphere::HorizontalDivisionNumber) * 2.0f * nn::util::FloatPi);
        g_SinArray[i] = sinf((i * 1.0f / Sphere::HorizontalDivisionNumber) * 2.0f * nn::util::FloatPi);
    }
}

float CalcVertexColor(int pattern, int i, int j, int k) NN_NOEXCEPT
{
    switch (pattern)
    {
    case Sphere::VertexColorPattern_MonoStripe:
        return (((j % 2) == 0) ? 1.0f : 0.5f);
    case Sphere::VertexColorPattern_VerticalGradient:
        return (g_CosArray[i] + 1.0f) / 2.0f;
    case Sphere::VertexColorPattern_HorizontalGradient:
        return (g_CosArray[j] + 1.0f) / 2.0f;
    case Sphere::VertexColorPattern_GradientDoubleSided:
        if (k == 0)
        {
            return (g_CosArray[i] + 1.0f) / 2.0f;
        }
        if (k == 2)
        {
            return (g_CosArray[Sphere::VerticalDivisionNumber - i] + 1.0f) / 2.0f;
        }
        return 0.0f;
    case Sphere::VertexColorPattern_RgbStripe:
        return ((j % 3) == k) ? 1.0f : 0.0f;
    case Sphere::VertexColorPattern_RgbStripeGradient:
        return ((j % 3) == k) ? (g_CosArray[i] + 1.0f) / 2.0f : 0.0f;
    case Sphere::VertexColorPattern_RgbStripeSqueeze:
        return (((j + i) % 3) == k) ? 1.0f : 0.0f;
    case Sphere::VertexColorPattern_RgbStripeSqueezeGradient:
        return (((j + i) % 3) == k) ? (g_CosArray[i] + 1.0f) / 2.0f : 0.0f;
    case Sphere::VertexColorPattern_ConstantValue:
    default:
        return 1.0f;
    }
}

void Sphere::Make() NN_NOEXCEPT
{
    if (g_Counter == 0)
    {
        Prepare();
        g_Counter++;
    }

    int base = 0;
    int face = 0;

    for (int i = 0; i <= VerticalDivisionNumber; i++)
    {
        if (i == 0)
        {
            // 北極
            for (int j = 0; j < HorizontalDivisionNumber; j++)
            {
                VertexPos[base + j].x = 0.0f;
                VertexPos[base + j].y = 0.0f;
                VertexPos[base + j].z = 1.0f;
                for (int k = 0; k < VertexColorPattern_CountMax; k++)
                {
                    VertexColor[k][base + j].x = CalcVertexColor(k, i, j, 0);
                    VertexColor[k][base + j].y = CalcVertexColor(k, i, j, 1);
                    VertexColor[k][base + j].z = CalcVertexColor(k, i, j, 2);
                    VertexColor[k][base + j].w = 1.0f;
                }
                VertexUv[base + j].x = 1.0f - ((j + 0.5f) * 1.0f / HorizontalDivisionNumber);
                VertexUv[base + j].y = 0.0f;
            }
            base += HorizontalDivisionNumber;
        }
        else if (i < VerticalDivisionNumber)
        {
            for (int j = 0; j <= HorizontalDivisionNumber; j++)
            {
                VertexPos[base + j].x = g_SinArray[i] * g_CosArray[j];
                VertexPos[base + j].y = g_SinArray[i] * g_SinArray[j];
                VertexPos[base + j].z = g_CosArray[i];

                for (int k = 0; k < VertexColorPattern_CountMax; k++)
                {
                    VertexColor[k][base + j].x = CalcVertexColor(k, i, j, 0);
                    VertexColor[k][base + j].y = CalcVertexColor(k, i, j, 1);
                    VertexColor[k][base + j].z = CalcVertexColor(k, i, j, 2);
                    VertexColor[k][base + j].w = 1.0f;
                }

                VertexUv[base + j].x = 1.0f - (j * 1.0f / HorizontalDivisionNumber);
                VertexUv[base + j].y = (i * 1.0f / VerticalDivisionNumber);

                if (i == 1)
                {
                    FaceTable[face++] = base - HorizontalDivisionNumber + j;
                    FaceTable[face++] = base + j + 0;
                    FaceTable[face++] = base + j + 1;
                }
                else
                {
                    FaceTable[face++] = base - (HorizontalDivisionNumber + 1) + j + 1;
                    FaceTable[face++] = base - (HorizontalDivisionNumber + 1) + j + 0;
                    FaceTable[face++] = base + j + 0;
                    FaceTable[face++] = base - (HorizontalDivisionNumber + 1) + j + 1;
                    FaceTable[face++] = base + j + 0;
                    FaceTable[face++] = base + j + 1;
                }
            }
            base += (HorizontalDivisionNumber + 1);
        }
        else
        {
            // 南極
            for (int j = 0; j < HorizontalDivisionNumber; j++)
            {
                VertexPos[base + j].x = 0.0f;
                VertexPos[base + j].y = 0.0f;
                VertexPos[base + j].z = -1.0f;
                for (int k = 0; k < VertexColorPattern_CountMax; k++)
                {
                    VertexColor[k][base + j].x = CalcVertexColor(k, i, j, 0);
                    VertexColor[k][base + j].y = CalcVertexColor(k, i, j, 1);
                    VertexColor[k][base + j].z = CalcVertexColor(k, i, j, 2);
                    VertexColor[k][base + j].w = 1.0f;
                }
                VertexUv[base + j].x = 1.0f - ((j + 0.5f) * (1.0f / HorizontalDivisionNumber));
                VertexUv[base + j].y = 1.0f;
            }
            // Index
            for (int j = 0; j < HorizontalDivisionNumber; j++)
            {
                FaceTable[face++] = base + j;
                FaceTable[face++] = base - (HorizontalDivisionNumber + 1) + j + 0;
                FaceTable[face++] = base - (HorizontalDivisionNumber + 1) + j + 1;
            }
            base += HorizontalDivisionNumber;
        }
    }
}
