﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include "ControllerManager.h"

namespace
{
    // コントローラの管理用 vector
    std::vector<NpadControllerBase*> g_Controllers;

    // 使用する NpadId および NpadStyle の設定
    NpadHandheldController g_HandheldController(nn::hid::NpadId::Handheld, "Id:Handheld Handheld");
    NpadJoyDualController g_JoyDualControllerNo1(nn::hid::NpadId::No1, "Id:1 JoyDual");
    NpadFullKeyController g_FullKeyControllerNo1(nn::hid::NpadId::No1, "Id:1 FullKey");

    // 振動処理用のスレッド
    nn::os::ThreadType  thread;
    nn::os::TimerEventType  g_TimerEvent;
    const size_t            StackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack1[StackSize];

    //
    bool g_IsQuitRequired = false;
}

void VibrationNodeThreadFunction(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    while (!g_IsQuitRequired)
    {
        nn::os::WaitTimerEvent(&g_TimerEvent);
        nn::hid::VibrationNode::Update();
        ControllerManager::GetInstance().UpdateVibrationBuffer();
    }
    nn::os::StopTimerEvent(&g_TimerEvent);
}

void StartVibrationThread() NN_NOEXCEPT
{
    // タイマーイベントを初期化する
    nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_AutoClear);
    // スレッドを生成する
    nn::Result result = nn::os::CreateThread(&thread, VibrationNodeThreadFunction, NULL, g_ThreadStack1, StackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    // スレッドの実行を開始する
    nn::os::StartThread(&thread);
    // タイマーイベントは周期タイマーイベントとして開始する
    const nn::TimeSpan interval = nn::hid::VibrationNode::DefaultVibrationSampleInterval;
    nn::os::StartPeriodicTimerEvent(&g_TimerEvent, interval, interval);

    g_IsQuitRequired = false;
}

void StopVibrationThread() NN_NOEXCEPT
{
    g_IsQuitRequired = true;
    // スレッドが終了するのを待機する
    nn::os::WaitThread(&thread);
    // スレッドを破棄する
    nn::os::DestroyThread(&thread);
    // タイマーイベントをファイナライズする
    nn::os::FinalizeTimerEvent(&g_TimerEvent);
}

ControllerManager& ControllerManager::GetInstance() NN_NOEXCEPT
{
    static ControllerManager instance;
    return instance;
}

const VibrationValueBuffer& ControllerManager::GetVibrationBuffer(int idx) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(idx >= 0);
    NN_ABORT_UNLESS(idx <= VibrationTargetCountMax);
    return m_VibrationBuffer[idx];
}

void ControllerManager::UpdateVibrationBuffer() NN_NOEXCEPT
{
    for (int i = 0; i < VibrationTargetCountMax; i++)
    {
        // 実際の振動値
        nn::hid::VibrationValue actualValue;
        // 設定した振動値
        nn::hid::VibrationValue currentValue;

        if (m_pVibrationTargetArray[i] != nullptr)
        {
            m_pVibrationTargetArray[i]->GetActualVibration(&actualValue);
            currentValue = m_pVibrationTargetArray[i]->GetCurrentVibration();
        }

        m_VibrationBuffer[i].AddVibrationValues(currentValue, actualValue);
    }
}

void ControllerManager::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定します
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask
    );

    // 使用する Npad を設定します
    nn::hid::NpadIdType npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld
    };
    nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));

    // 管理対象の操作形態を追加
    g_Controllers.push_back(&g_HandheldController);
    g_Controllers.push_back(&g_JoyDualControllerNo1);
    g_Controllers.push_back(&g_FullKeyControllerNo1);

    // 管理対象の操作形態に関する初期化
    for (auto it = g_Controllers.begin();
        it != g_Controllers.end();
        ++it)
    {
        (*it)->Initialize();
    }

    // 管理対象が１つもないときは終了
    NN_ASSERT(g_Controllers.size() > 0);

    // 現在有効なコントローラの設定
    m_pActiveController = *(g_Controllers.begin());
}

void ControllerManager::Finalize() NN_NOEXCEPT
{
    g_Controllers.clear();
}

void ControllerManager::Update() NN_NOEXCEPT
{
    for (auto it = g_Controllers.begin();
        it != g_Controllers.end();
        ++it)
    {
        // 各コントローラの状態を更新
        (*it)->Update();

        // 接続しているコントローラで、最後に操作しているものを現在有効なものとする
        if ((*it)->IsConnected())
        {
            const auto& button = (*it)->GetNpadButtonSet();
            auto sensor = (*it)->IsSixAxisSensorAtRest();

            if (button.IsAnyOn() || sensor == false)
            {
                if (m_pActiveController != *it)
                {
                    //有効なコントローラを変更する
                    m_pActiveController = *it;

                    //有効なコントローラの振動ハンドルを VibrationTarget に設定する
                    for (int i = 0; i < VibrationTargetCountMax; i++)
                    {
                        if (m_pVibrationTargetArray[i] != nullptr)
                        {
                            auto handle = m_pActiveController->GetVibrationState(i).deviceHandle;
                            m_pVibrationTargetArray[i]->UnsetVibrationDeviceHandle();
                            m_pVibrationTargetArray[i]->SetVibrationDeviceHandle(handle);
                        }
                    }
                }
            }
        }
    }
}

void ControllerManager::SetActiveVibrationTarget(nn::hid::VibrationTarget* left, nn::hid::VibrationTarget* right) NN_NOEXCEPT
{
    // 現在の VibrationTarget の設定を解放する
    for (int i = 0; i < VibrationTargetCountMax; i++)
    {
        if (m_pVibrationTargetArray[i] != nullptr)
        {
            m_pVibrationTargetArray[i]->UnsetVibrationDeviceHandle();
        }
    }

    m_pVibrationTargetArray[0] = left;
    m_pVibrationTargetArray[1] = right;

    for (int i = 0; i < VibrationTargetCountMax; i++)
    {
        if (m_pVibrationTargetArray[i] != nullptr)
        {
            auto handle = m_pActiveController->GetVibrationState(i).deviceHandle;
            m_pVibrationTargetArray[i]->UnsetVibrationDeviceHandle();
            m_pVibrationTargetArray[i]->SetVibrationDeviceHandle(handle);
        }
    }
}

