﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidUniquePadSimple.cpp,PageSampleHidUniquePadSimple}
 *
 * @brief
 *  UniquePad でのデバイス列挙を示したシンプルのサンプルプログラムです
 */

/**
 * @page PageSampleHidUniquePadSimple 本体に物理接続されているコントローラーを列挙します
 * @tableofcontents
 *
 * @brief
 *  本サンプルプログラムは、本体に接続されているデバイスを Npad 毎に出力します。
 *  接続されているコントローラーについては、コントローラーの種類、接続インタフェース、シリアルナンバーを表示します
 *
 * @section PageSampleConnectionTrigger_SectionBrief 概要
 *  ここでは、UniquePad でのデバイス列挙を示したシンプルなサンプルプログラムの説明をします。
 *
 *  UniquePad の使い方については、Hid の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleConnectionTrigger_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidUniquePadSimple Samples/Sources/Applications/HidUniquePadSimple @endlink 以下にあります。
 *
 * @section PageSampleConnectionTrigger_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionPrecaution 注意事項
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleConnectionTrigger_SectionDetail 解説
 *
 * @subsection PageSampleConnectionTrigger_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HidUniquePadSimple.cpp
 *  @includelineno HidUniquePadSimple.cpp
 *
 * @subsection PageSampleConnectionTrigger_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nn::hid::system::BindUniquePadConnectionEvent() に渡したイベントでデバイスの接続状態の監視を行う
 *  - デバイスの接続状態が更新された際に Npad 毎に関連付けられた UniquePad の Id を取得する
 *  - UniquePad 毎にコントローラーの情報を出力する
 *
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace
{
    const nn::hid::NpadIdType NpadIds[] = { nn::hid::NpadId::No1,
                                             nn::hid::NpadId::No2,
                                             nn::hid::NpadId::No3,
                                             nn::hid::NpadId::No4,
                                             nn::hid::NpadId::No5,
                                             nn::hid::NpadId::No6,
                                             nn::hid::NpadId::No7,
                                             nn::hid::NpadId::No8,
                                             nn::hid::NpadId::Handheld };

void PrintNpadId(nn::hid::NpadIdType id) NN_NOEXCEPT;
void PrintUniquePadInfo(nn::hid::system::UniquePadId id) NN_NOEXCEPT;

}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、自前で nninitStartup() を用意しています。
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));

    // Set supported NpadStyle
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                                        nn::hid::NpadStyleJoyDual::Mask |
                                        nn::hid::NpadStyleJoyLeft::Mask |
                                        nn::hid::NpadStyleJoyRight::Mask);

    nn::os::MultiWaitType multiWait;
    nn::os::SystemEventType connectionEvent;
    nn::os::MultiWaitHolderType connectionEventHolder;

    nn::os::InitializeMultiWait(&multiWait);

    nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

    bool runs = true;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        if(pHolder == &connectionEventHolder)
        {
            nn::os::ClearSystemEvent(&connectionEvent);

            NN_LOG("\n------------------------------------------------\n");
            NN_LOG("Device List Updated\n");
            nn::hid::system::UniquePadId uniquePadIds[nn::hid::system::UniquePadIdCountMax];

            // Npad毎に UniquePadId を取得
            for (auto& npadId : NpadIds)
            {
                auto uniquePadCount = nn::hid::system::GetUniquePadsFromNpad(uniquePadIds, NN_ARRAY_SIZE(uniquePadIds), npadId);
                if (uniquePadCount > 0)
                {
                    NN_LOG("\n\n");
                    PrintNpadId(npadId);
                    NN_LOG("\n");
                    for (int uniquePadIndex = 0; uniquePadIndex < uniquePadCount; uniquePadIndex++)
                    {
                        PrintUniquePadInfo(uniquePadIds[uniquePadIndex]);
                    }
                }
            }
            NN_LOG("\n------------------------------------------------\n");
        }
    }
}

namespace
{

void PrintNpadId(nn::hid::NpadIdType npadId) NN_NOEXCEPT
{
    switch (npadId)
    {
    case nn::hid::NpadId::No1:
        NN_LOG("Npad01");
        break;
    case nn::hid::NpadId::No2:
        NN_LOG("Npad02");
        break;
    case nn::hid::NpadId::No3:
        NN_LOG("Npad03");
        break;
    case nn::hid::NpadId::No4:
        NN_LOG("Npad04");
        break;
    case nn::hid::NpadId::No5:
        NN_LOG("Npad05");
        break;
    case nn::hid::NpadId::No6:
        NN_LOG("Npad06");
        break;
    case nn::hid::NpadId::No7:
        NN_LOG("Npad07");
        break;
    case nn::hid::NpadId::No8:
        NN_LOG("Npad08");
        break;
    case nn::hid::NpadId::Handheld:
        NN_LOG("NpadHandheld");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void PrintUniquePadInfo(nn::hid::system::UniquePadId uniquePadId) NN_NOEXCEPT
{
    ::nn::hid::system::UniquePadInterface uniquePadInterface;
    ::nn::hid::system::UniquePadType uniquePadType;
    ::nn::hid::system::UniquePadSerialNumber uniquePadSerialNumber;

    NN_LOG("    UniquePad Id=0x%llx ", uniquePadId._storage);

    uniquePadType = ::nn::hid::system::GetUniquePadType(uniquePadId);
    switch (uniquePadType)
    {
    case nn::hid::system::UniquePadType_LeftController:
        NN_LOG("Joy-Con(L)");
        break;
    case nn::hid::system::UniquePadType_RightController:
        NN_LOG("Joy-Con(R)");
        break;
    case nn::hid::system::UniquePadType_FullKeyController:
        NN_LOG("Switch Pro Controller");
        break;
    case nn::hid::system::UniquePadType_Embedded:
        NN_LOG("Embedded");
        break;
    default:
        ;
    }


    if (::nn::hid::system::GetUniquePadInterface(&uniquePadInterface, uniquePadId).IsSuccess())
    {
        NN_LOG("\n        Interface:");
        switch (uniquePadInterface)
        {
        case nn::hid::system::UniquePadInterface_Bluetooth:
            NN_LOG("Bluetooth");
            break;
        case nn::hid::system::UniquePadInterface_Usb:
            NN_LOG("Usb");
            break;
        case nn::hid::system::UniquePadInterface_Rail:
            NN_LOG("Rail");
            break;
        case nn::hid::system::UniquePadInterface_Embedded:
            NN_LOG("Embedded");
            break;
        default:
            ;
        }
    }

    if (::nn::hid::system::GetUniquePadSerialNumber(&uniquePadSerialNumber, uniquePadId).IsSuccess())
    {
        NN_LOG("\n        SerialNumber: ");
        for (auto& c : uniquePadSerialNumber._storage)
        {
            NN_LOG("%c", c);
        }
    }
    NN_LOG("\n");
}

}
