﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidSixAxisSensorAccurateCalibration_Main.cpp,PageSampleHidSixAxisSensorAccurateCalibration}
 *
 * @brief
 *  6 軸センサーの高精度ユーザーキャリブレーションのサンプルプログラム
 */

/**
 * @page PageSampleHidSixAxisSensorAccurateCalibration 本体 6 軸センサーのユーザーキャリブレーション
 * @tableofcontents
 *
 * @brief
 *  6軸センサーのユーザーキャリブレーションのサンプルプログラムの解説です。
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionBrief 概要
 *  ここでは、6軸センサーのユーザーキャリブレーションのサンプルプログラムの説明をします。
 *
 *  6軸センサーのユーザーキャリブレーションの使い方については、Hid の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidSixAxisSensorAccurateCalibration Samples/Sources/Applications/HidSixAxisSensorAccurateCalibration @endlink 以下にあります。
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionPrecaution 注意事項
 *  とくになし
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleHidSixAxisSensorAccurateCalibration_SectionDetail 解説
 *
 * @subsection PageSampleHidSixAxisSensorAccurateCalibration_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HidSixAxisSensorAccurateCalibration_Main.cpp
 *  @includelineno HidSixAxisSensorAccurateCalibration_Main.cpp
 *
 * @subsection PageSampleHidSixAxisSensorAccurateCalibration_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - キャリブレーション処理を開始する
 *  - キャリブレーション中の状態をプリントする
 *  - キャリブレーション関連の API 呼び出しをする
 *
 *  A ボタンを押すと本体 6 軸センサーのキャリブレーションが開始されます。
 *  B ボタンを押すと本体 6 軸センサーのキャリブレーションがキャンセルされます。
 *  X ボタンを押すと本体 6 軸センサーのキャリブレーション値を工程出荷状態に戻します。
 *
 *  キャリブレーションが開始されるとキャリブレーションの段階と進捗状況が出力されます。
 *
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_TimerEvent.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>

#include "HidSixAxisSensorAccurateCalibration_Context.h"

extern "C" void nnMain()
{
    // Npad の初期化
    ::nn::hid::InitializeNpad();

    // 本体 6 軸センサーの初期化
    nn::mem::StandardAllocator appAllocator;
    const size_t ApplicationHeapSize = 128 * 1024 * 1024;
    nn::Bit8* pAppMemory = new nn::Bit8[ApplicationHeapSize];
    appAllocator.Initialize(pAppMemory, ApplicationHeapSize);
    void* pWorkBuffer = appAllocator.Allocate(::nn::hid::SevenSixAxisSensorWorkBufferSize, 4096);
    ::nn::hid::InitializeSevenSixAxisSensor(pWorkBuffer, ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    // タイマーイベントを登録
    ::nn::os::MultiWaitType multiWait;
    ::nn::os::TimerEventType timerEvent;
    ::nn::os::MultiWaitHolderType timerEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    ::nn::os::InitializeTimerEvent(&timerEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timerEventHolder,
                                        &timerEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait,
                                  &timerEventHolder);
    const int64_t IntervalMilliSeconds = 15;
    ::nn::os::StartPeriodicTimerEvent(&timerEvent,
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSeconds),
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSeconds));

    bool runs = true;
    Context context;
    context.Initialize();

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        // タイマーイベントがシグナルされた
        if (pHolder == &timerEventHolder)
        {
            ::nn::os::ClearTimerEvent(&timerEvent);

            context.Update();
        }
    }

    ::nn::hid::FinalizeSevenSixAxisSensor();
}
