﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_Vibration.h"
#include "HidNpadIntegrate_VibrationValueDrawer.h"

namespace
{

    SET_PLUGIN(VibrationDemoScene, "Vibration Demo");

    uint32_t g_LoopCount = 0;
    const int VibrationDeviceCountMax = 2;
    const int VibrationNpadIdCountMax = 5;
    const int VibrationNpadStyleMax = 5;

    // 振動子の状態を表す構造体です。
    struct VibrationState
    {
        nn::hid::VibrationDeviceHandle deviceHandle;    // 振動子のハンドル
        nn::hid::VibrationDeviceInfo deviceInfo;        // 振動子のデバイス情報
        int vibrationPatternId;                         // 振動パターン番号
        nn::hid::VibrationValue currentVibration;       // 現在発生している振動
    };

    class VibrationDemo
    {
        NN_DISALLOW_COPY(VibrationDemo);
        NN_DISALLOW_MOVE(VibrationDemo);

    public:
        VibrationDemo() NN_NOEXCEPT {}

        const char* GetName() NN_NOEXCEPT
        {
            return m_Name.c_str();
        }

        nn::hid::NpadIdType GetNpadId() NN_NOEXCEPT
        {
            return *m_pId;
        }

        void Initialize(const nn::hid::NpadIdType& Id) NN_NOEXCEPT
        {
            m_pId = &Id;
            nn::hid::VibrationDeviceHandle handles[VibrationNpadStyleMax][VibrationDeviceCountMax];

            for(int i = 0; i < GetPluginManager().GetNpadStylePluginCount(); i++)
            {
                // 振動子のハンドルを取得します
                nn::hid::NpadStyleSet NpadStyle =
                                        GetPluginManager().GetNpadStyle(i)->GetNpadStyleSet();
                m_VibrationDeviceCount[i] = nn::hid::GetVibrationDeviceHandles(
                    handles[i], VibrationDeviceCountMax, *m_pId, NpadStyle);

                for (int j = 0; j < m_VibrationDeviceCount[i]; j++)
                {
                    VibrationState& state = m_VibrationStateArray[i][j];
                    state.vibrationPatternId = 0;
                    state.currentVibration = nn::hid::VibrationValue::Make();
                    state.deviceHandle = handles[i][j];

                    // 振動子を初期化します
                    nn::hid::InitializeVibrationDevice(state.deviceHandle);

                    // 振動子の情報を取得します
                    nn::hid::GetVibrationDeviceInfo(&state.deviceInfo, state.deviceHandle);
                }
            }
        }

        void Update() NN_NOEXCEPT
        {
            m_Style = nn::hid::GetNpadStyleSet(*m_pId);
            NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(m_Style);
            if(pNpad != NULL)
            {
                m_Name = pNpad->GetName();
                m_StyleIndex = pNpad->GetStyleIndex();
                m_IsDeviceConnected = true;
            }
            else
            {
                m_Name = "---";
                m_IsDeviceConnected = false;
            }

            UpdateButtons();
            UpdateVibrationPattern();
            UpdateVibrationValue();
        }

        int GetVibrationDeviceCount() NN_NOEXCEPT
        {
            return m_VibrationDeviceCount[m_StyleIndex];
        }

        bool GetDeviceConnected() NN_NOEXCEPT
        {
            return m_IsDeviceConnected;
        }

        const VibrationState& GetVibrationState(int id) NN_NOEXCEPT
        {
            return m_VibrationStateArray[m_StyleIndex][id];
        }

    private:
        void UpdateButtons() NN_NOEXCEPT
        {
            NpadCommonState buttonState;
            buttonState = GetNpadButtonSet(*m_pId);
            m_Buttons = buttonState.buttons;
        }

        void UpdateVibrationPattern() NN_NOEXCEPT
        {
            const nn::hid::NpadButtonSet LeftHandButtonMask =
                nn::hid::NpadButton::Up::Mask   | nn::hid::NpadButton::Right::Mask |
                nn::hid::NpadButton::Down::Mask | nn::hid::NpadButton::Left::Mask |
                nn::hid::NpadButton::L::Mask    | nn::hid::NpadButton::ZL::Mask;
            const nn::hid::NpadButtonSet RightHandButtonMask =
                nn::hid::NpadButton::A::Mask | nn::hid::NpadButton::B::Mask |
                nn::hid::NpadButton::X::Mask | nn::hid::NpadButton::Y::Mask |
                nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask;

            // 押されているボタンの個数に応じて振動パターンを更新します
            for(int i = 0; i < m_VibrationDeviceCount[m_StyleIndex]; i++)
            {
                VibrationState& state = m_VibrationStateArray[m_StyleIndex][i];
                switch(state.deviceInfo.position)
                {
                case nn::hid::VibrationDevicePosition_Left:
                    state.vibrationPatternId =
                        (m_Buttons & LeftHandButtonMask).CountPopulation();
                    break;
                case nn::hid::VibrationDevicePosition_Right:
                    state.vibrationPatternId =
                        (m_Buttons & RightHandButtonMask).CountPopulation();
                    break;
                default:
                    state.vibrationPatternId = 0;
                    break;
                }
            }
        }

        void UpdateVibrationValue() NN_NOEXCEPT
        {
            for (int i = 0; i < m_VibrationDeviceCount[m_StyleIndex]; i++)
            {
                VibrationState& state = m_VibrationStateArray[m_StyleIndex][i];

                // 振動パターンに応じた振動値を生成します
                nn::hid::VibrationValue vibrationValue = nn::hid::VibrationValue::Make();
                switch (state.vibrationPatternId)
                {
                case 0:
                    // 振動を発生させない
                    break;
                case 1:
                    // 周波数 160 Hz で振幅が 1.00 と 0.00 に交互に切り替わる振動を発生します
                    vibrationValue.amplitudeLow = (g_LoopCount % 20 < 1) ? 1.0f : 0.0f;
                    vibrationValue.frequencyLow = 160.0f;
                    vibrationValue.amplitudeHigh = 0.0f;
                    vibrationValue.frequencyHigh = 320.0f;
                    break;
                case 2:
                    // 周波数 160 Hz で振幅が動的に変化する振動を発生します
                    vibrationValue.amplitudeLow = 0.5f * ( 1.0f + static_cast<float>(
                        sin(2 * nn::util::FloatPi * (g_LoopCount % 60 / 60.0 ))));
                    vibrationValue.frequencyLow = 160.0f;
                    vibrationValue.amplitudeHigh = 0.0f;
                    vibrationValue.frequencyHigh = 320.0f;
                    break;
                case 3:
                    // 振幅 0.50 で周波数が 160 Hz と 320 Hz に交互に切り替わる振動を発生します
                    vibrationValue.amplitudeLow = (g_LoopCount % 60 < 30) ? 0.5f : 0.0f;
                    vibrationValue.frequencyLow = 160.0f;
                    vibrationValue.amplitudeHigh = (g_LoopCount % 60 < 30) ? 0.0f : 0.5f;
                    vibrationValue.frequencyHigh = 320.0f;
                    break;
                default:
                    // 振幅が 0.50 で周波数が動的に変化する振動を発生します
                    vibrationValue.amplitudeLow = 0.5f;
                    vibrationValue.frequencyLow = 160.0f * (1.5f + 0.5f * static_cast<float>(
                        sin(2 * nn::util::FloatPi * (g_LoopCount % 120 / 120.0))));
                    vibrationValue.amplitudeHigh = 0.0f;
                    vibrationValue.frequencyHigh = 320.0f;
                    break;
                }

                // 振動値を送信します
                nn::hid::SendVibrationValue(state.deviceHandle, vibrationValue);

                // 現在の振動を取得します
                nn::hid::GetActualVibrationValue(&state.currentVibration, state.deviceHandle);
            }
        }

    private:
        const nn::hid::NpadIdType*  m_pId;
        int                         m_StyleIndex;
        bool                        m_IsDeviceConnected;
        std::string                 m_Name;
        nn::hid::NpadButtonSet      m_Buttons;
        nn::hid::NpadStyleSet       m_Style;
        int                         m_VibrationDeviceCount[VibrationNpadStyleMax];
        VibrationState     m_VibrationStateArray[VibrationNpadStyleMax][VibrationDeviceCountMax];
    };

    VibrationDemo& GetVibrationDevice(int i) NN_NOEXCEPT
    {
        static VibrationDemo s_VibrationDevice[VibrationNpadIdCountMax];
        return s_VibrationDevice[i % VibrationNpadIdCountMax];
    }

    // Vibration Demoの操作説明を描画します。
    void WriteVibrationDemoGuide(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);
        const float OffsetX = 650;
        const float OffsetY = 350;

        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("Corresponding buttons are below.");

        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 30);
        pTextWriter->Print("Right");
        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 55);
        pTextWriter->Print("Left");
        pTextWriter->SetCursor(OffsetX + 70, OffsetY + 30);
        pTextWriter->Print(":");
        pTextWriter->SetCursor(OffsetX + 70, OffsetY + 55);
        pTextWriter->Print(":");
        pTextWriter->SetCursor(OffsetX + 80, OffsetY + 30);
        pTextWriter->Print("A, B, X, Y, R, ZR");
        pTextWriter->SetCursor(OffsetX + 80, OffsetY + 55);
        pTextWriter->Print("Up, Right, Left, Down, L, ZL");

        pTextWriter->SetCursor(OffsetX, OffsetY + 100);
        pTextWriter->Print("Vibration pattern will change depending on the number of buttons");
        pTextWriter->SetCursor(OffsetX, OffsetY + 120);
        pTextWriter->Print("pressed at the same time.");

        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 150);
        pTextWriter->Print("0 : Not vibrate");
        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 175);
        pTextWriter->Print("1 : Frequency[160Hz], Amplitude[alternately 1.00 and 0.00]");
        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 200);
        pTextWriter->Print("2 : Frequency[160Hz], Amplitude[dynamic]");
        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 225);
        pTextWriter->Print("3 : Frequency[alternately 160Hz and 320Hz ], Amplitude[0.50]");
        pTextWriter->SetCursor(OffsetX + 20, OffsetY + 250);
        pTextWriter->Print("4 or more : Frequency[dynamic], Amplitude[0.50]");
    }

    void UpdateGraphics(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        const nn::util::Unorm8x4 TextColor = Color::White;

        WriteCommonGuide(pTextWriter);
        WriteVibrationDemoGuide(pTextWriter);
        DrawRect(pTextWriter, 630, 330, 75, 38);

        for (int i = 0; i < VibrationNpadIdCountMax; i++)
        {
            int vibrationDeviceCount = (GetVibrationDevice(i).GetDeviceConnected() ?
                GetVibrationDevice(i).GetVibrationDeviceCount() : 0);

            float baseX = 30.0f + 600.0f * (i / 4);
            float baseY = 50.0f + 160.0f * (i % 4);

            pTextWriter->SetTextColor(vibrationDeviceCount != 0 ? TextColor : Color::Gray);
            pTextWriter->SetScale(1.0f, 1.0f);
            pTextWriter->SetCursor(baseX, baseY);
            if(GetVibrationDevice(i).GetNpadId() == nn::hid::NpadId::Handheld)
            {
                pTextWriter->Print("NpadHandheld");
            }
            else
            {
                pTextWriter->Print("NpadID:No.%d, NpadStyle:%s",
                    i, GetVibrationDevice(i).GetName());
            }

            for (int deviceId = 0; deviceId < vibrationDeviceCount; deviceId++)
            {
                const VibrationState& State = GetVibrationDevice(i).GetVibrationState(deviceId);

                float x = baseX + 20.0f + 280.0f * deviceId;
                float y = baseY + 25.0f;

                pTextWriter->SetTextColor(TextColor);
                pTextWriter->SetScale(0.8f, 0.8f);
                pTextWriter->SetCursor(x, y);
                switch(State.deviceInfo.position)
                {
                case nn::hid::VibrationDevicePosition_Left:
                    pTextWriter->Print("Left");
                    break;
                case nn::hid::VibrationDevicePosition_Right:
                    pTextWriter->Print("Right");
                    break;
                default:
                    pTextWriter->Print("Unknown");
                    break;
                }
                pTextWriter->Print(" (Pattern=%d)", State.vibrationPatternId);

                VibrationValueDrawer vibrationValueDrawer(pTextWriter);
                vibrationValueDrawer.SetPosition(x, y + 15);
                vibrationValueDrawer.SetFontScale(0.6f);
                vibrationValueDrawer.DrawVibrationValue(State.currentVibration, true);
            }
        }
    }

} // namespace

VibrationDemoScene::VibrationDemoScene() NN_NOEXCEPT
{
}

VibrationDemoScene::~VibrationDemoScene() NN_NOEXCEPT
{
}

void VibrationDemoScene::InitializeScene(ApplicationHeap* pAppAllocator,
                        GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    NN_UNUSED(pAppAllocator);
    NN_UNUSED(pGraphicsSystem);

    m_IsRunnableAlways = false;
    m_StatusName = "VIB";

    if(!m_IsRunning)
    {
        for (int i = 0; i < VibrationNpadIdCountMax; i++)
        {
            GetVibrationDevice(i).Initialize(NpadIds[i]);
        }
        m_IsRunning = true;
    }
}

void VibrationDemoScene::FinalizeScene(ApplicationHeap* pAppAllocator) NN_NOEXCEPT
{
    NN_UNUSED(pAppAllocator);

    if(m_IsRunning)
    {
        m_IsRunning = false;
    }
}

void VibrationDemoScene::StopScene() NN_NOEXCEPT
{
}

void VibrationDemoScene::RestartScene() NN_NOEXCEPT
{
}

void VibrationDemoScene::SwitchScene() NN_NOEXCEPT
{
    // 無振動の振動値を各コントローラに送信
    for (int i = 0; i < VibrationNpadIdCountMax; i++)
    {
        for(int j = 0; j < GetVibrationDevice(i).GetVibrationDeviceCount() ; j++)
        {
            const VibrationState& State = GetVibrationDevice(i).GetVibrationState(j);

            nn::hid::VibrationValue vibrationValue = nn::hid::VibrationValue::Make();
            // 振動値を送信します
            nn::hid::SendVibrationValue(State.deviceHandle, vibrationValue);
        }
    }
}

void VibrationDemoScene::ToggleProcess(const nn::hid::NpadButtonSet& Buttons,
                        ApplicationHeap* pAppAllocator,
                        GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    NN_UNUSED(Buttons);
    NN_UNUSED(pAppAllocator);
    NN_UNUSED(pGraphicsSystem);
}

void VibrationDemoScene::RunScene(nn::gfx::util::DebugFontTextWriter& textWriter,
                GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    NN_UNUSED(pGraphicsSystem);

    for (int i = 0; i < VibrationNpadIdCountMax; i++)
    {
        GetVibrationDevice(i).Update();
    }

    UpdateGraphics(&textWriter);
    g_LoopCount++;
}
