﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_NpadJoy.h>

#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SettingApi.h"
#include "HidNpadIntegrate_SettingApiPluginBase.h"

class NpadDeviceType : public ReadWriteBase
{
    NN_DISALLOW_COPY(NpadDeviceType);
    NN_DISALLOW_MOVE(NpadDeviceType);

public:
    NN_IMPLICIT NpadDeviceType(nn::hid::NpadJoyDeviceType* pDeviceType) NN_NOEXCEPT
    : m_pDeviceType(pDeviceType) {}

    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (*m_pDeviceType == nn::hid::NpadJoyDeviceType_Left ? "Left" : "Right");
    }

    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        *m_pDeviceType = static_cast<nn::hid::NpadJoyDeviceType>(1 - *m_pDeviceType);
    }

private:
    nn::hid::NpadJoyDeviceType* m_pDeviceType;
};

class SetNpadJoyAssignmentModeSingleApiState : public SettingApiPluginBase
{
    NN_DISALLOW_COPY(SetNpadJoyAssignmentModeSingleApiState);
    NN_DISALLOW_MOVE(SetNpadJoyAssignmentModeSingleApiState);

public:
    SetNpadJoyAssignmentModeSingleApiState() NN_NOEXCEPT {}

    ~SetNpadJoyAssignmentModeSingleApiState() NN_NOEXCEPT NN_OVERRIDE {}

    void Initialize(int* pNextProcessor, int* pMenuSelection) NN_NOEXCEPT NN_OVERRIDE
    {
        SettingApiPluginBase::Initialize(pNextProcessor, pMenuSelection);
        m_NpadId = 0;
        m_DeviceType = nn::hid::NpadJoyDeviceType_Left;

        m_pNpadIdType = new NpadIdType(&m_NpadId);
        MenuItem npadIdTypeMenuItem("  NpadID    ", m_pNpadIdType);
        m_ReadWriteMenu.push_back(npadIdTypeMenuItem);

        m_pNpadDeviceType = new NpadDeviceType(&m_DeviceType);
        MenuItem npadDeviceTypeMenuItem("  DeviceType", m_pNpadDeviceType);
        m_ReadWriteMenu.push_back(npadDeviceTypeMenuItem);
    }

    void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pNpadIdType;
        delete m_pNpadDeviceType;
        SettingApiPluginBase::Finalize();
    }

    void Start() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(m_NpadId, m_DeviceType);
        m_ResultMsg = "Success";
    }

    void Explain(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);

        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->
            Print("This function sets AssignmentMode of Joy-Con to \"Single Mode\", and assigns");
        pTextWriter->SetCursor(OffsetX, OffsetY + 30);
        pTextWriter->
            Print("each Joy-Con to specified NpadIDs.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 60);
        pTextWriter->
            Print("So different NpadID and PlayerNo are assigned for each Joy-Con(R) and");
        pTextWriter->SetCursor(OffsetX, OffsetY + 90);
        pTextWriter->
            Print("Joy-Con(L).");
        pTextWriter->SetCursor(OffsetX, OffsetY + 120);
        pTextWriter->
            Print("One NpadID and DeviceType(L or R) are needed to execute this function.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 150);
        pTextWriter->
            Print("Specified Joy-Con by DeviceType keeps connecting.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 180);
        pTextWriter->
            Print("Another Joy-Con will be disconnected from Npad, and connected to Npad");
        pTextWriter->SetCursor(OffsetX, OffsetY + 210);
        pTextWriter->
            Print("with no connection.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 240);
        pTextWriter->
            Print("If only one Joy-Con is connected to the Npad, the value of DeviceType is");
        pTextWriter->SetCursor(OffsetX, OffsetY + 270);
        pTextWriter->
            Print("ignored.");
    }

    bool DrawAssignment(
        int index, nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("AssignmentMode: %s",
            nn::hid::GetNpadJoyAssignment(NpadIds[index]) == nn::hid::NpadJoyAssignmentMode_Dual ?
            "Dual" : "Single");

        return true;
    }

private:
    nn::hid::NpadIdType m_NpadId;
    nn::hid::NpadJoyDeviceType m_DeviceType;
    ReadWriteBase* m_pNpadIdType;
    ReadWriteBase* m_pNpadDeviceType;
};
namespace {
SET_PLUGIN(SetNpadJoyAssignmentModeSingleApiState, "SetNpadJoyAssignmentModeSingle");
}
