﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ResultController.h>

#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SettingApi.h"
#include "HidNpadIntegrate_SettingApiPluginBase.h"

class MergeSingleJoyAsDualJoyApiState : public SettingApiPluginBase
{
    NN_DISALLOW_COPY(MergeSingleJoyAsDualJoyApiState);
    NN_DISALLOW_MOVE(MergeSingleJoyAsDualJoyApiState);

public:
    MergeSingleJoyAsDualJoyApiState() NN_NOEXCEPT {}

    ~MergeSingleJoyAsDualJoyApiState() NN_NOEXCEPT NN_OVERRIDE {}

    void Initialize(int* pNextProcessor, int* pMenuSelection) NN_NOEXCEPT NN_OVERRIDE
    {
        SettingApiPluginBase::Initialize(pNextProcessor, pMenuSelection);
        m_NpadId1 = 0;
        m_NpadId2 = 0;

        m_pNpadIdType1 = new NpadIdType(&m_NpadId1);
        MenuItem npadIdType1MenuItem("  NpadID1", m_pNpadIdType1);
        m_ReadWriteMenu.push_back(npadIdType1MenuItem);

        m_pNpadIdType2 = new NpadIdType(&m_NpadId2);
        MenuItem npadIdType2MenuItem("  NpadID2", m_pNpadIdType2);
        m_ReadWriteMenu.push_back(npadIdType2MenuItem);
    }

    void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pNpadIdType1;
        delete m_pNpadIdType2;
        SettingApiPluginBase::Finalize();
    }

    void Start() NN_NOEXCEPT NN_OVERRIDE
    {
        std::stringstream stringStream;
        nn::Result result;

        result = nn::hid::MergeSingleJoyAsDualJoy(m_NpadId1, m_NpadId2);
        if(result.IsSuccess())
        {
            m_ResultMsg = "Success";
        }
        else
        {
            const int Module = result.GetModule();
            const int Description = result.GetDescription();

            stringStream << "module:" << Module << " desc:" << Description;
            m_ResultNum = stringStream.str();

            if((Module == nn::hid::ResultNpadDualConnected().GetModule()) &&
               (Description == nn::hid::ResultNpadDualConnected().GetDescription()))
            {
                m_ResultMsg = "ResultNpadDualConnected";
            }
            else if((Module == nn::hid::ResultNpadSameJoyTypeConnected().GetModule()) &&
               (Description == nn::hid::ResultNpadSameJoyTypeConnected().GetDescription()))
            {
                m_ResultMsg = "ResultNpadSameJoyTypeConnected";
            }
            else
            {
                m_ResultMsg = "Unknown";
            }
        }
    }

    void Explain(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);

        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->
            Print("This function merges two Npads into one to form pair.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 30);
        pTextWriter->
            Print("To execute this function, it is necessary to specify two NpadIDs,");
        pTextWriter->SetCursor(OffsetX, OffsetY + 60);
        pTextWriter->
            Print("and each Npads should be connected to one Joy-Con(R) and one Joy-Con(L).");
        pTextWriter->SetCursor(OffsetX, OffsetY + 90);
        pTextWriter->
            Print("In case of Npad's connection like below, execution will fail.");
        pTextWriter->SetCursor(OffsetX, OffsetY + 120);
        pTextWriter->
            Print("* One Npad with two Joy-Con");
        pTextWriter->SetCursor(OffsetX, OffsetY + 150);
        pTextWriter->
            Print("* Both Npad with Joy-Con(R)");
    }

    bool DrawAssignment(
        int index, nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX, const float OffsetY) NN_NOEXCEPT NN_OVERRIDE
    {
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("AssignmentMode: %s",
            nn::hid::GetNpadJoyAssignment(NpadIds[index]) == nn::hid::NpadJoyAssignmentMode_Dual ?
            "Dual" : "Single");

        return true;
    }

private:
    nn::hid::NpadIdType m_NpadId1;
    nn::hid::NpadIdType m_NpadId2;
    ReadWriteBase* m_pNpadIdType1;
    ReadWriteBase* m_pNpadIdType2;
};

namespace {
SET_PLUGIN(MergeSingleJoyAsDualJoyApiState, "MergeSingleJoyAsDualJoy");
}
