﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <sstream>

#include <nn/nn_Assert.h>

#include "HidNpadIntegrate_IrSensorModeState.h"

namespace
{

    const char* const AmbientNoiseLevel[] =
    {
        "Low",
        "Middle",
        "High",
        "Unknown",
    };

} // namespace

const float IrSensorModeStatistics::ExpectedMomentModeFramerate     = 70.0f;
const float IrSensorModeStatistics::ExpectedClusteringModeFramerate = 70.0f;
const float IrSensorModeStatistics::ExpectedHandAnalysisFramerate   = 66.6f;

class CurrentMode : public ReadWriteBase
{
    NN_DISALLOW_COPY(CurrentMode);
    NN_DISALLOW_MOVE(CurrentMode);

public:
    CurrentMode(IrSensorMode* pNextProcessor, int* pMenuSelection) NN_NOEXCEPT
        : m_pNextProcessor(pNextProcessor)
        , m_pMenuSelection(pMenuSelection)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        switch (*m_pNextProcessor)
        {
        case IrSensorMode_Moment:
            sstr << "Moment";
            break;
        case IrSensorMode_Clustering:
            sstr << "Clustering";
            break;
        case IrSensorMode_ImageTransfer:
            sstr << "Image Transfer";
            break;
        case IrSensorMode_HandAnalysis:
            sstr << "Hand Analysis";
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t mode = static_cast<int8_t>(*m_pNextProcessor) + delta;
        *m_pNextProcessor = static_cast<IrSensorMode>(
            std::min(
                std::max(mode, static_cast<int8_t>(IrSensorMode_Moment)),
                static_cast<int8_t>(IrSensorMode_Count - 1)
            )
        );
        *m_pMenuSelection = 0;
    }

private:
    IrSensorMode* m_pNextProcessor;
    int* m_pMenuSelection;
};

class ExposureTime : public ReadWriteBase
{
    NN_DISALLOW_COPY(ExposureTime);
    NN_DISALLOW_MOVE(ExposureTime);

public:
    ExposureTime(
        ::nn::irsensor::IrCameraConfig* pConfig,
        ::nn::TimeSpanType exposureMin,
        ::nn::TimeSpanType exposureMax) NN_NOEXCEPT
        : m_pConfig(pConfig)
        , m_ExposureMin(exposureMin)
        , m_ExposureMax(exposureMax)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pConfig->exposureTime.GetMicroSeconds();
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pConfig->exposureTime += ::nn::TimeSpanType::FromMicroSeconds(delta * 10);
        m_pConfig->exposureTime =
            std::min(std::max(m_pConfig->exposureTime, m_ExposureMin), m_ExposureMax);
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
    ::nn::TimeSpanType m_ExposureMin;
    ::nn::TimeSpanType m_ExposureMax;
};

class LightTarget : public ReadWriteBase
{
    NN_DISALLOW_COPY(LightTarget);
    NN_DISALLOW_MOVE(LightTarget);

public:
    LightTarget(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_pConfig->lightTarget)
        {
        case nn::irsensor::IrCameraLightTarget_AllObjects:
            sstr << "AllObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_FarObjects:
            sstr << "FarObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_NearObjects:
            sstr << "NearObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_None:
            sstr << "None";
            break;
        default:
            break;
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t lightTarget =
            static_cast<int8_t>(m_pConfig->lightTarget) + static_cast<int8_t>(delta);
        lightTarget = std::max(
            std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
            static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects)
        );
        m_pConfig->lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

class DigitalGain : public ReadWriteBase
{
    NN_DISALLOW_COPY(DigitalGain);
    NN_DISALLOW_MOVE(DigitalGain);

public:
    DigitalGain(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pConfig->gain;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pConfig->gain += delta;
        m_pConfig->gain =
            std::min(std::max(m_pConfig->gain, nn::irsensor::IrCameraGainMin),
                nn::irsensor::IrCameraGainMax);
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

class NegativeImage : public ReadWriteBase
{
    NN_DISALLOW_COPY(NegativeImage);
    NN_DISALLOW_MOVE(NegativeImage);

public:
    NegativeImage(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (m_pConfig->isNegativeImageUsed ? "ON" : "OFF");
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        m_pConfig->isNegativeImageUsed = !m_pConfig->isNegativeImageUsed;
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

class SamplingNumber : public ReadWriteBase
{
    NN_DISALLOW_COPY(SamplingNumber);
    NN_DISALLOW_MOVE(SamplingNumber);

public:
    SamplingNumber(
        int64_t* pSamplingNumber) NN_NOEXCEPT
        : m_pSamplingNumber(pSamplingNumber)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << *m_pSamplingNumber;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    int64_t* m_pSamplingNumber;
};

class NoiseLevel : public ReadWriteBase
{
    NN_DISALLOW_COPY(NoiseLevel);
    NN_DISALLOW_MOVE(NoiseLevel);

public:
    NoiseLevel(
        nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel) NN_NOEXCEPT
        : m_pAmbientNoiseLevel(pAmbientNoiseLevel)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << AmbientNoiseLevel[*m_pAmbientNoiseLevel];
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    nn::irsensor::IrCameraAmbientNoiseLevel* m_pAmbientNoiseLevel;
};

class PacketLoss : public ReadWriteBase
{
    NN_DISALLOW_COPY(PacketLoss);
    NN_DISALLOW_MOVE(PacketLoss);

public:
    NN_IMPLICIT PacketLoss(float* pPacketLoss) NN_NOEXCEPT
        : m_pPacketLoss(pPacketLoss)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (*m_pPacketLoss) * 100.0f << "%";
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    float* m_pPacketLoss;
};

void IrSensorModeStatistics::Update(const int64_t* pCurrentSamplingNumber,
    const float ExpectedFrameRate) NN_NOEXCEPT
{
    // 3 秒ごとに Packet Loss 向けデータを初期化します
    if ( ( m_FramerateCounter % ResetIntervalsInFrame ) == 0)
    {
        m_FramerateFirstTick = nn::os::GetSystemTick();
        m_FramerateFirstSample = *pCurrentSamplingNumber;
    }
    nn::os::Tick currentTick = nn::os::GetSystemTick() - m_FramerateFirstTick;
    int64_t currentSample = *pCurrentSamplingNumber - m_FramerateFirstSample;

    // UpdateIntervalsInFrame 周期ごとに、Packet Loss を更新します
    if ( m_FramerateCounter % UpdateIntervalsInFrame == ( UpdateIntervalsInFrame - 1 ) )
    {
        m_FramerateComputation =
            1000.0f * currentSample / nn::os::ConvertToTimeSpan(currentTick).GetMilliSeconds();
        m_PacketDropPercentage = 1.0f - m_FramerateComputation / ExpectedFrameRate;
        m_PacketDropPercentage = std::min(std::max(m_PacketDropPercentage, 0.0f), 1.0f);
    }
    m_FramerateCounter++;
}

IrSensorModeState::IrSensorModeState(IrSensorMode* pNextProcessor,
    int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle) NN_NOEXCEPT
    : m_IrCameraHandle(irCameraHandle)
    , m_pCurrentMode(NULL)
    , m_pExposureTime(NULL)
    , m_pLightTarget(NULL)
    , m_pDigitalGain(NULL)
    , m_pNegativeImage(NULL)
    , m_pSamplingNumber(NULL)
    , m_pNoiseLevel(NULL)
    , m_pPacketLoss(NULL)
{
    m_pCurrentMode = new CurrentMode(pNextProcessor, pMenuSelection);
    MenuItem currentModeMenuItem("Current Mode",
        m_pCurrentMode
    );
    m_ReadWriteMenu.push_back(currentModeMenuItem);
}

IrSensorModeState::~IrSensorModeState() NN_NOEXCEPT

{
    delete m_pCurrentMode;
    delete m_pExposureTime;
    delete m_pLightTarget;
    delete m_pDigitalGain;
    delete m_pNegativeImage;
    delete m_pSamplingNumber;
    delete m_pNoiseLevel;
    delete m_pPacketLoss;
}

void IrSensorModeState::AddCommonReadWriteMenu(std::vector<MenuItem>* pMenu,
    ::nn::irsensor::IrCameraConfig* pConfig,
    ::nn::TimeSpanType exposureMin,
    ::nn::TimeSpanType exposureMax) NN_NOEXCEPT
{
    m_pExposureTime = new ExposureTime(pConfig, exposureMin, exposureMax);
    MenuItem exposureTimeMenuItem("Exposure Time",
        m_pExposureTime
    );
    pMenu->push_back(exposureTimeMenuItem);

    m_pLightTarget = new LightTarget(pConfig);
    MenuItem lightTargetMenuItem("Light Target",
        m_pLightTarget
    );
    pMenu->push_back(lightTargetMenuItem);

    m_pDigitalGain = new DigitalGain(pConfig);
    MenuItem digitalGainMenuItem("Digital Gain",
        m_pDigitalGain
    );
    pMenu->push_back(digitalGainMenuItem);

    m_pNegativeImage = new NegativeImage(pConfig);
    MenuItem negativeImageMenuItem("Negative Image",
        m_pNegativeImage
    );
    pMenu->push_back(negativeImageMenuItem);
}

void IrSensorModeState::AddCommonReadOnlyMenu(std::vector<MenuItem>* pMenu,
    int64_t* pSamplingNumber,
    nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel) NN_NOEXCEPT
{
    m_pSamplingNumber = new SamplingNumber(pSamplingNumber);
    MenuItem samplingNumberMenuItem("Sampling Number",
        m_pSamplingNumber
    );
    pMenu->push_back(samplingNumberMenuItem);

    m_pNoiseLevel = new NoiseLevel(pAmbientNoiseLevel);
    MenuItem noiseLevelMenuItem("Ambient Noise Level",
        m_pNoiseLevel
    );
    pMenu->push_back(noiseLevelMenuItem);
}

void IrSensorModeState::AddStatisticsMenu(std::vector<MenuItem>* pMenu,
    float* pPacketLoss) NN_NOEXCEPT
{
    m_pPacketLoss = new PacketLoss(pPacketLoss);
    MenuItem packetLossMenuItem("Packet Loss",
        m_pPacketLoss
    );
    pMenu->push_back(packetLossMenuItem);
}

const std::vector<MenuItem>& IrSensorModeState::GetReadWriteMenu() NN_NOEXCEPT
{
    return m_ReadWriteMenu;
}

const std::vector<MenuItem>& IrSensorModeState::GetReadOnlyMenu() NN_NOEXCEPT
{
    return m_ReadOnlyMenu;
}

void IrSensorModeState::Stop() NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessor(m_IrCameraHandle);
}
