﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include "HidNpadIntegrate_Color.h"
#include "HidNpadIntegrate_IrSensor.h"
#include "HidNpadIntegrate_IrSensorClusteringModeState.h"
#include "HidNpadIntegrate_IrSensorHandAnalysisModeState.h"
#include "HidNpadIntegrate_IrSensorImageTransferModeState.h"
#include "HidNpadIntegrate_IrSensorMomentModeState.h"
#include "HidNpadIntegrate_PluginManager.h"

namespace
{

    SET_PLUGIN(IrSensorDemoScene, "IrSensor Demo");

    const int IrSensorNpadIdCountMax = 5;

    IrSensorDemo& GetDemoStates(int i) NN_NOEXCEPT
    {
        static IrSensorDemo s_DemoStates[IrSensorNpadIdCountMax];
        return s_DemoStates[i % IrSensorNpadIdCountMax];
    }

    void* g_WorkMemoryBuffers[IrSensorNpadIdCountMax];

    void WriteIrCameraStatus(nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX,
        const float OffsetY) NN_NOEXCEPT
    {
        pTextWriter->SetScale(2.0f, 2.0f);
        pTextWriter->SetTextColor(Color::White);

        pTextWriter->SetCursor(OffsetX + 480, OffsetY + 100);
        if (OffsetY < 360)
        {
            pTextWriter->Print("1st IrCamera Status");
        }
        else
        {
            pTextWriter->Print("2nd IrCamera Status");
        }

        pTextWriter->SetCursor(OffsetX + 420, OffsetY + 170);
        pTextWriter->Print("IrCameraStatus_Unconnected");
    }

    void WriteProcessorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const float OffsetX,
        const float OffsetY) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        DrawRect(pTextWriter, OffsetX + 315, OffsetY + 185, 35, 14);
        pTextWriter->SetScale(1.0f, 1.0f);

        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 200);
        pTextWriter->Print("Change Component");
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 215);
        pTextWriter->Print("    Press X ( Up ) / B ( Down )");
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 235);
        pTextWriter->Print("Change Value");
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 250);
        pTextWriter->Print("    Press Y ( Left ) / A ( Right )");
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 270);
        pTextWriter->Print("Reflect User Setting");
        pTextWriter->SetCursor(OffsetX + 330, OffsetY + 285);
        pTextWriter->Print("    Press R ( L )");
    }

} // namespace

void CreateSampler(GraphicsSystem* pGraphicsSystem,
    nn::gfx::Sampler* pSampler, nn::gfx::DescriptorSlot* pDescriptorSlot) NN_NOEXCEPT
{
    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->Initialize(&pGraphicsSystem->GetDevice(), samplerInfo);

    pGraphicsSystem->RegisterSamplerSlot(pDescriptorSlot, *pSampler);
}

void CreateTexture(GraphicsSystem* pGraphicsSystem,
    int width, int height, nn::gfx::ImageFormat format, nn::gfx::Texture* pTexture,
    nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pDescriptorSlot) NN_NOEXCEPT
{
    {
        nn::gfx::Texture::InfoType textureInfo;

        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetMipCount(1);
        textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
        textureInfo.SetWidth(width);
        textureInfo.SetHeight(height);
        textureInfo.SetImageFormat(format);

        pGraphicsSystem->AllocateTexture(pTexture, &textureInfo);
    }

    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetImageFormat(format);
        viewInfo.SetTexturePtr(pTexture);
        viewInfo.SetChannelMapping(nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red,
            nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Red);

        pTextureView->Initialize(&pGraphicsSystem->GetDevice(), viewInfo);
    }

    {
        pGraphicsSystem->RegisterTextureViewSlot(pDescriptorSlot, *pTextureView);
    }
}

IrSensorDemo::IrSensorDemo() NN_NOEXCEPT
    : m_CurrentProcessor()
    , m_NextProcessor()
    , m_IrCameraHandle()
    , m_MenuSelection()
    , m_PreviousButtonState()
{
}

void IrSensorDemo::Initialize(nn::hid::NpadIdType id,
    GraphicsSystem* pGraphicsSystem, void* pImageTransferWorkeMemory) NN_NOEXCEPT
{
    m_NpadId = id;

    // IR カメラのハンドルの取得
    m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);

    // IR カメラの初期化
    nn::irsensor::Initialize(m_IrCameraHandle);

    // IrSensor を Moment として初期化
    m_ModeStates[IrSensorMode_Moment] =
        new MomentModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);
    m_ModeStates[IrSensorMode_Clustering] =
        new ClusteringModeState(&m_NextProcessor, &m_MenuSelection, m_IrCameraHandle);
    m_ModeStates[IrSensorMode_ImageTransfer] =
        new ImageTransferModeState(&m_NextProcessor, &m_MenuSelection,
            m_IrCameraHandle, pImageTransferWorkeMemory, pGraphicsSystem);
    m_ModeStates[IrSensorMode_HandAnalysis] =
        new HandAnalysisModeState(&m_NextProcessor, &m_MenuSelection,
            m_IrCameraHandle, pGraphicsSystem);

    m_CurrentProcessor = m_NextProcessor = IrSensorMode_Moment;
    m_ModeStates[m_CurrentProcessor]->Start();
}

void IrSensorDemo::Finalize() NN_NOEXCEPT
{
    m_ModeStates[m_CurrentProcessor]->Stop();

    delete m_ModeStates[IrSensorMode_Moment];
    delete m_ModeStates[IrSensorMode_Clustering];
    delete m_ModeStates[IrSensorMode_ImageTransfer];
    delete m_ModeStates[IrSensorMode_HandAnalysis];

    nn::irsensor::Finalize(m_IrCameraHandle);
}

void IrSensorDemo::Run(int i, GraphicsSystem* pGraphicsSystem,
    nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    CheckNpadButtons();

    m_ModeStates[m_CurrentProcessor]->Update();

    {
        const std::vector<MenuItem>& ReadWriteMenu =
            m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
        const std::vector<MenuItem>& ReadOnlyMenu  =
            m_ModeStates[m_NextProcessor]->GetReadOnlyMenu();

        int selectIndex = GetMenuIndex();

        WriteProcessorState(pTextWriter, 660.0f, 55.0f + 350.0f * i);

        DrawMenu(ReadWriteMenu, 700.0f, 100.0f + 350.0f * i,
            selectIndex, pTextWriter);
        DrawMenu(ReadOnlyMenu, 970.0f, 100.0f + 350.0f * i,
            -1, pTextWriter);

    }

    m_ModeStates[m_CurrentProcessor]->Render(&pGraphicsSystem->GetPrimitiveRenderer(),
        &pGraphicsSystem->GetCommandBuffer(), i);
}

void IrSensorDemo::CheckNpadButtons() NN_NOEXCEPT
{
    nn::hid::NpadButtonSet currentButtonState;
    nn::hid::NpadButtonSet buttonDownSet;

    // 押されたボタンを検出
    currentButtonState = GetNpadButtonSet(m_NpadId).buttons;
    buttonDownSet = currentButtonState & ~m_PreviousButtonState;
    m_PreviousButtonState = currentButtonState;

    // Menu Item を選ぶ
    if ((buttonDownSet & nn::hid::NpadButton::B::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn())
    {
        m_MenuSelection += 1;
    };

    if ((buttonDownSet & nn::hid::NpadButton::X::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::Up::Mask).IsAnyOn())
    {
        m_MenuSelection -= 1;
    };

    // Menu の設定値を変更する
    if ((buttonDownSet & nn::hid::NpadButton::Y::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::Left::Mask).IsAnyOn())
    {
        int8_t delta = ((buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn() ||
                        (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn()) ? 1 : -1;

        const std::vector<MenuItem>& Menu =
            this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
        const int MenuIndex = GetMenuIndex();
        if (MenuIndex >= 0)
        {
            Menu[MenuIndex].Increment(delta);
        }
    }

    // イベントトリガー
    if ((buttonDownSet & nn::hid::NpadButton::R::Mask).IsAnyOn() ||
        (buttonDownSet & nn::hid::NpadButton::L::Mask).IsAnyOn())
    {
        m_ModeStates[m_CurrentProcessor]->Stop();
        m_ModeStates[m_NextProcessor]->Start();
        m_CurrentProcessor = m_NextProcessor;
    }

}

void IrSensorDemo::StartProcessor() NN_NOEXCEPT
{
    m_ModeStates[m_CurrentProcessor]->Start();
}

void IrSensorDemo::StopProcessor() NN_NOEXCEPT
{
    m_ModeStates[m_CurrentProcessor]->Stop();
}

nn::irsensor::IrCameraHandle IrSensorDemo::GetIrCameraHandle() NN_NOEXCEPT
{
    return m_IrCameraHandle;
}

int IrSensorDemo::GetMenuIndex() const NN_NOEXCEPT
{
    const std::vector<MenuItem>& Menu = this->m_ModeStates[m_NextProcessor]->GetReadWriteMenu();
    int size = static_cast<int>(Menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}

IrSensorDemoScene::IrSensorDemoScene() NN_NOEXCEPT
{
}

IrSensorDemoScene::~IrSensorDemoScene() NN_NOEXCEPT
{
}

void IrSensorDemoScene::InitializeScene(ApplicationHeap* pAppAllocator,
                        GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    m_SceneNum = GetPluginManager().GetCurrentSceneNum();
    m_IsRunnableAlways = true;
    m_StatusName       = "IR";
    m_ToggleOperation  = "Left + L Stick ( Y + R Stick )";

    if(!m_IsRunning)
    {
        for (int i = 0; i < IrSensorNpadIdCountMax; i++)
        {
            g_WorkMemoryBuffers[i] = pAppAllocator->Allocate(ImageTransferModeState::WorkMemorySize,
                                                      ImageTransferModeState::WorkMemoryAlignement);
            GetDemoStates(i).Initialize(NpadIds[i], pGraphicsSystem, g_WorkMemoryBuffers[i]);
        }
        m_IsRunning = true;
    }
}

void IrSensorDemoScene::FinalizeScene(ApplicationHeap* pAppAllocator) NN_NOEXCEPT
{
    if(m_IsRunning)
    {
        for (int i = IrSensorNpadIdCountMax - 1; i >= 0; i--)
        {
            GetDemoStates(i).Finalize();
            pAppAllocator->Deallocate(g_WorkMemoryBuffers[i]);
        }
        m_IsRunning = false;
    }
}

void IrSensorDemoScene::StopScene() NN_NOEXCEPT
{
    if(m_IsRunning)
    {
        for (int i = 0; i < IrSensorNpadIdCountMax; i++)
        {
            GetDemoStates(i).StopProcessor();
        }
    }
}

void IrSensorDemoScene::RestartScene() NN_NOEXCEPT
{
    if(m_IsRunning)
    {
        for (int i = 0; i < IrSensorNpadIdCountMax; i++)
        {
            GetDemoStates(i).StartProcessor();
        }
    }
}

void IrSensorDemoScene::SwitchScene() NN_NOEXCEPT
{
}

void IrSensorDemoScene::ToggleProcess(const nn::hid::NpadButtonSet& Buttons,
                        ApplicationHeap* pAppAllocator,
                        GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    const int ChengeWaitTime = 500;

    // IR センサによる計算処理のON/OFFを切り替える
    if ((Buttons.Test<nn::hid::NpadButton::Y>() &&
         Buttons.Test<nn::hid::NpadButton::StickR>()) ||
        (Buttons.Test<nn::hid::NpadButton::Left>() &&
         Buttons.Test<nn::hid::NpadButton::StickL>()))
    {
        if( m_SceneNum != GetPluginManager().GetCurrentSceneNum())
        {
            if(m_IsRunning)
            {
                FinalizeScene(pAppAllocator);
            }
            else
            {
                InitializeScene(pAppAllocator, pGraphicsSystem);
            }
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ChengeWaitTime));
        }
    }
}

void IrSensorDemoScene::RunScene(nn::gfx::util::DebugFontTextWriter& textWriter,
                GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    const int CameraStateViewMax = 2;
    int viewNo = 0;
    for (int i = 0; i < IrSensorNpadIdCountMax; ++i)
    {
        const nn::irsensor::IrCameraStatus IrCameraStatus =
            nn::irsensor::GetIrCameraStatus(GetDemoStates(i).GetIrCameraHandle());

        if (nn::irsensor::IrCameraStatus_Available == IrCameraStatus)
        {
            GetDemoStates(i).Run(viewNo, pGraphicsSystem, &textWriter);

            textWriter.SetTextColor(Color::White);
            textWriter.SetScale(1.2f, 1.2f);
            textWriter.SetCursor(680, 65 + 350.0f * viewNo);
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);
            NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
            if(pNpad == NULL)
            {
                continue;
            }

            textWriter.Print("NpadID:No.%d, NpadStyle:%s", i, pNpad->GetName());
            viewNo++;
        }

        if(viewNo >= CameraStateViewMax)
        {
            break;
        }
    }

    for (int i = viewNo; i < CameraStateViewMax; i++)
    {
        WriteIrCameraStatus(&textWriter, 0, 40.f + 350.0f * i);
    }
    WriteCommonGuide(&textWriter);
}
