﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <list>
#include <vector>

#include <nn/hid.h>
#include <nn/nn_Log.h>

#include <nn/hid/hid_Gesture.h>

enum SwipeType
{
    SwipeType_None,
    SwipeType_Right,
    SwipeType_Left,
};

class GestureManager
{
    NN_DISALLOW_COPY(GestureManager);
    NN_DISALLOW_MOVE(GestureManager);

private:
    std::vector<nn::hid::GesturePoint> m_Points;
    nn::hid::GestureState* m_States;
    SwipeType m_SwipeType;
    int64_t m_EventNumber;

public:
    GestureManager() NN_NOEXCEPT
        : m_SwipeType(SwipeType_None)
        , m_EventNumber(-1)
    {
        m_States = new nn::hid::GestureState[nn::hid::GestureStateCountMax]();
    }

    ~GestureManager() NN_NOEXCEPT
    {
        delete[] m_States;
    }

    const std::vector<nn::hid::GesturePoint>& GetPoints() const NN_NOEXCEPT
    {
        return m_Points;
    }

    void Update() NN_NOEXCEPT
    {
        const int count = nn::hid::GetGestureStates(
            m_States, nn::hid::GestureStateCountMax);

        for (int i = count - 1; i >= 0; --i)
        {
            if (m_EventNumber < m_States[i].eventNumber)
            {
                this->CheckGestureState(m_States[i]);
            }
        }

        m_EventNumber = m_States[0].eventNumber;
    }

    SwipeType GetSwipeType() const NN_NOEXCEPT
    {
        return m_SwipeType;
    }

    void ClearSwipeType() NN_NOEXCEPT
    {
        m_SwipeType = SwipeType_None;
    }

private:
    void CheckGestureState(const nn::hid::GestureState& state) NN_NOEXCEPT
    {
        m_Points.clear();

        for (int i = 0; i < state.pointCount; ++i)
        {
            m_Points.push_back(state.points[i]);
        }

        if(state.GetGestureType() == nn::hid::GestureType_Swipe)
        {
            if(state.GetGestureDirection() == nn::hid::GestureDirection_Left)
            {
                m_SwipeType = SwipeType_Left;
            }
            else if(state.GetGestureDirection() == nn::hid::GestureDirection_Right)
            {
                m_SwipeType = SwipeType_Right;
            }
        }
    }

};
