﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidControllerSequence_Start.cpp,PageSampleHidControllerSequence_Start}

    @brief
    サンプルアプリケーションのスタート画面でのコントローラーサポートアプレットの呼び出し方
*/

/**
    @page PageSampleHidControllerSequence_Start スタート画面でのコントローラーサポートアプレットの呼び出し方
    @tableofcontents

    @brief
    サンプルアプリケーションのスタート画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Start_SectionBrief 概要
    サンプルアプリケーションのスタート画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Start_SectionFileStructure ファイル構成
    本サンプルプログラムは
    @link ../../../Samples/Sources/Applications/HidControllerSequence
    Samples/Sources/Applications/HidControllerSequence @endlink 以下にあります。

    @section PageSampleHidControllerSequence_Start_SectionNecessaryEnvironment 必要な環境
    事前に実機とコントローラーをペアリングしてください。

    @section PageSampleHidControllerSequence_Start_SectionHowToOperate 操作方法
    コントローラーが接続されていない場合、起動直後にコントローラーサポートアプレットが呼び出されます。@n
    コントローラーが接続されている場合、スタート画面が表示されます。@n
    接続中のいずれかのコントローラーのボタンを押すことでタイトル画面に切り替わります

    @section PageSampleHidControllerSequence_Start_SectionPrecaution 注意事項
    コントローラーは十分に充電した状態でお使いください。

    @section PageSampleHidControllerSequence_Start_SectionHowToExecute 実行手順
    サンプルアプリケーションをビルドし、実行してください。

    @section PageSampleHidControllerSequence_Start_SectionDetail 解説

    @image html Applications\HidControllerSequence\HidControllerSequence_Start.png

    スタート画面での処理の流れは以下の通りです。
    - コントローラーの接続確認
    - 接続されているボタンの入力確認
    - コントローラーの入れ替え
    - タイトル画面に移動

    スタート画面では以下の場合、コントローラーサポートアプレットが呼び出されます。
    - nn::hid::NpadId::Handheld と nn::hid::NpadId::No1 がどちらも接続されていない

    nn::hid::NpadId::No1 以外の無線コントローラーでボタン入力を行った場合、@n
    当該のコントローラーを nn::hid::NpadId::No1 と入れ替え、以降マスタープレイヤーとして扱います。

    @image html Applications\HidControllerSequence\HidControllerSequence_Single.png

*/

#include "./HidControllerSequence_Start.h"

StartSceneThread::StartSceneThread() : ThreadState()
{

}

StartSceneThread::StartSceneThread(const char* name, nn::os::ThreadFunction func) : ThreadState(name, func)
{
}

void StartSceneThread::Initialize()
{
    m_pRenderer         = &g_pGraphicsSystem->GetPrimitiveRenderer();
    m_pTextWriter       = &g_pGraphicsSystem->GetDebugFont();
    m_pCommand          = &g_pGraphicsSystem->GetCommandBuffer();

    m_IsExit = false;
    m_FrameCount = 0;
    m_StartText = "- PRESS BUTTON -";
    m_StartTextPos = nn::util::MakeFloat2(
        (1280.f - m_pTextWriter->CalculateStringWidth(m_StartText.c_str())) / 2.f,
        (720.f - m_pTextWriter->CalculateStringHeight(m_StartText.c_str())) / 2.f + 72.f
    );
    m_TitleLogoPos = nn::util::MakeFloat2(
        (1280.f - 640.f) / 2.f,
        (720.f - 64.f) / 2.f - 64.f
    );
    m_WaitButtons.Reset();
    m_WaitButtons =
        nns::hid::Button::A::Mask |
        nns::hid::Button::B::Mask |
        nns::hid::Button::X::Mask |
        nns::hid::Button::Y::Mask |
        nns::hid::Button::L::Mask |
        nns::hid::Button::R::Mask |
        nns::hid::Button::ZL::Mask |
        nns::hid::Button::ZR::Mask |
        nns::hid::Button::Up::Mask |
        nns::hid::Button::Left::Mask |
        nns::hid::Button::Right::Mask |
        nns::hid::Button::Down::Mask;
}

void StartSceneThread::Update()
{
    ++m_FrameCount;
    m_pTextWriter->SetScale(1.f, 1.f);
    //==============================================
    // コントローラー の処理
    //==============================================
    nn::os::LockMutex(&g_Mutex);
    {
        g_pControllerManager->Update();
    }
    nn::os::UnlockMutex(&g_Mutex);

    // コントローラーの接続確認
    CheckPatternSet check;
    check.Reset();
    check.Set<CheckPattern::Single>();
    check.Set<CheckPattern::CallInterval>();
    check.Set<CheckPattern::Resume>();
    if (CheckConnectController(check))
    {
        // コントローラーサポートアプレットの呼び出しが必要な場合
        CallControllerSupportApplet(nullptr, 1);
    }
    else
    {
        // 何らかのボタンが押下げられているコントローラーを1Pとします
        for (
            std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
            it != g_pControllerManager->GetControllerList().end();
            ++it)
        {
            if (
                (static_cast<size_t>(it - g_pControllerManager->GetControllerList().begin()) < NN_ARRAY_SIZE(ApplicationState::NpadIds)) &&
                (*it)->IsConnected())
            {
                if (((*it)->GetButtonsDown() & m_WaitButtons).IsAnyOn())
                {
                    const nn::hid::NpadIdType id = ApplicationState::NpadIds[it - g_pControllerManager->GetControllerList().begin()];
                    ThreadState* pThreadState = nullptr;

                    nn::os::LockMutex(&g_Mutex);
                    {
                        ApplicationState::EnableHandheldMode(id == nn::hid::NpadId::Handheld);
                        NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneStart, &pThreadState));
                        g_ThreadManager.SetEndThread(pThreadState->GetThreadType());
                        NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneTitle, &pThreadState));
                        g_ThreadManager.SetNextThread(pThreadState->GetThreadType());
                    }
                    nn::os::UnlockMutex(&g_Mutex);

                    if (!ApplicationState::IsHandheldMode())
                    {
                        /*
                        * 本体装着コントローラー以外のコントローラーを使用している場合
                        * そのコントローラーを NpadId::No1 に変更します
                        */
                        nn::hid::SwapNpadAssignment(nn::hid::NpadId::No1, id);
                    }
                    NN_LOG("Master Player -> %s\n", ApplicationState::IsHandheldMode() ? "Handheld Controller" : "Wireless Controller");
                    m_IsExit = true;
                    break;
                }
            }
        }
    }
    //==============================================
    // テキスト の処理
    //==============================================
    uint8_t color = ((m_FrameCount % 255) > (UINT8_MAX / 2)) ? ((UINT8_MAX / 2) - ((m_FrameCount % 255) - (UINT8_MAX / 2))) * 2 : (m_FrameCount % 255) * 2;
    nn::os::LockMutex(&g_Mutex);
    {
        m_pTextWriter->SetScale(1.f, 1.f);
        m_pTextWriter->SetTextColor(nn::util::Color4u8(color, color, color, 255));
        m_pTextWriter->SetCursor(m_StartTextPos.x, m_StartTextPos.y);
        m_pTextWriter->Print(m_StartText.c_str());
    }
    nn::os::UnlockMutex(&g_Mutex);
}

void StartSceneThread::Draw()
{
    //==============================================
    // 描画処理
    //==============================================
    nn::os::LockMutex(&g_Mutex);
    {
        g_pGraphicsSystem->BeginDraw();
        m_pRenderer->SetColor(nn::util::Color4u8::White());
        m_pRenderer->Draw2DRect(
            m_pCommand, m_TitleLogoPos.x, m_TitleLogoPos.y,
            640, 64, g_TextureDescriptor[ResourceType_Title], g_SamplerDescriptor);

        m_pTextWriter->Draw(m_pCommand);
        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }
    nn::os::UnlockMutex(&g_Mutex);
}

void StartSceneThread::Finalize()
{

}
