﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidControllerSequence_Main.cpp,PageSampleHidControllerSequence}

    @brief
    コントローラーサポートアプレットの呼び出し方の一例となるサンプルアプリケーション
*/

/**
    @page PageSampleHidControllerSequence コントローラーサポートアプレットの呼び出し方
    @tableofcontents

    @brief
    コントローラーサポートアプレットの呼び出し方のサンプルアプリケーションの解説です

    @image html Applications\HidControllerSequence\HidControllerSequence.png

    @section PageSampleHidControllerSequence_SectionBrief 概要
    コントローラーサポートアプレットの呼び出し方の一例となるサンプルアプリケーションの解説です。

    @section PageSampleHidControllerSequence_SectionFileStructure ファイル構成
    本サンプルプログラムは
    @link ../../../Samples/Sources/Applications/HidControllerSequence
    Samples/Sources/Applications/HidControllerSequence @endlink 以下にあります。

    @section PageSampleHidControllerSequence_SectionNecessaryEnvironment 必要な環境
    事前に実機とコントローラーをペアリングしてください。

    @section PageSampleHidControllerSequence_SectionHowToOperate 操作方法
    サンプルアプリケーションを実行してください。

    コントローラーが接続されていない場合、起動直後にコントローラーサポートアプレットが呼び出されます。@n
    コントローラーが接続されている場合、スタート画面が表示されます。@n
    接続中のいずれかのコントローラーでボタン入力をするとタイトル画面に移動します。

    タイトル画面では上下の方向ボタン、もしくはスティック操作で項目を移動する事が出来ます。@n
    Aボタンを押すと、』カーソルがあっているボタンに表示されたプレイヤー人数でゲームを開始します。@n
    その際、プレイヤー人数に対してコントローラーの接続数が不足する場合、コントローラーサポートアプレットが呼び出されます。

    ゲーム画面では左右の方向ボタンもしくはスティック操作でプレイヤーを操作する事が出来ます。@n
    また、プレイヤーのコントローラーで([Plus] or [Minus])を押すと中断ダイアログが表示されます。@n
    中断ダイアログで「Return to Title」を選択するとタイトル画面に戻ります。

    @section PageSampleHidControllerSequence_SectionPrecaution 注意事項
    コントローラーは十分に充電した状態でお使いください。

    @section PageSampleHidControllerSequence_SectionHowToExecute 実行手順
    サンプルアプリケーションをビルドし、実行してください。

    @section PageSampleHidControllerSequence_Star_SectionDetail 解説
    サンプルアプリケーションは以下のシーンから構成されます。
    - スタート画面
    - プレイ人数の選択を行うタイトル画面
    - ゲーム画面
    - ダイアログで表示される画面

    @subpage PageSampleHidControllerSequence_Start @n
    @subpage PageSampleHidControllerSequence_Title @n
    @subpage PageSampleHidControllerSequence_Game @n@n

    アプリケーションは、コントローラーサポートアプレットを使わない場合でも、@n
    起動時、復帰時および動作中にコントローラーが不足した際、@n
    独自にコントローラーの準備を案内することを推奨します。@n@n

    サンプルアプリケーションの全体のフロー図です。

    @image html Applications\HidControllerSequence\HidControllerSequence_Full.png
*/

#include "./HidControllerSequence_Main.h"
#include "./HidControllerSequence_Start.h"
#include "./HidControllerSequence_Title.h"
#include "./HidControllerSequence_Game.h"

nns::hid::ControllerManager*    g_pControllerManager;
::GraphicsSystem*               g_pGraphicsSystem;

nn::gfx::ResTextureFile*        g_pResTextureFile;
int32_t                         g_TextureIndex[ResourceType_Num];
nn::gfx::DescriptorSlot         g_TextureDescriptor[ResourceType_Num];
nn::gfx::DescriptorSlot         g_ScanScreenDescriptor;
nn::gfx::DescriptorSlot         g_SamplerDescriptor;
nn::gfx::Sampler                g_Sampler;

const size_t ApplicationState::MaxPlayerCount;
const nn::hid::NpadIdType ApplicationState::NpadIds[5] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::Handheld
};
nn::hid::NpadAttributesSet ApplicationState::InitialConnectedController[(MaxPlayerCount + 1)];

size_t ApplicationState::s_PlayerCount = ApplicationState::MaxPlayerCount;
bool ApplicationState::s_IsHandheldMode = false;

const int32_t           g_FrameRate = 60;
nn::os::MutexType       g_Mutex;

extern "C" void nnMain()
{
    nn::os::InitializeMutex(&g_Mutex, false, 0);
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // 復帰時のメッセージを受け取れるようにします
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_SuspendAndNotify);
    nn::oe::SetResumeNotificationEnabled(true);

    //==============================================
    // Controller
    //==============================================
    /*
     *  コントローラーの初期化を行います
     *
     *  有効化されるNpadId
     *
     *  ::nn::hid::NpadId::No1 ～ ::nn::hid::NpadId::No4
     *  ::nn::hid::NpadId::Handheld
     *
     *  有効化されるNpadStyle
     *
     *  nn::hid::NpadStyleFullKey
     *  nn::hid::NpadStyleHandheld
     *  nn::hid::NpadStyleJoyDual
     *  nn::hid::NpadStyleJoyLeft
     *  nn::hid::NpadStyleJoyRight
     */
    nns::hid::ControllerManager controllerManager;
    nns::hid::GamePadNxAsset::Initialize();

    // 有効な操作形態を設定します
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleHandheld::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleJoyLeft::Mask |
        nn::hid::NpadStyleJoyRight::Mask
    );

    // Joy-Con を横持ちに設定します
    nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType::NpadJoyHoldType_Horizontal);

    // GamePadNx を管理対象に追加します。
    controllerManager.GetDeviceAssetList().push_back(new nns::hid::GamePadNxAsset(&controllerManager));

    // コントローラーを管理対象に追加します。
    for (size_t i = 0; i < ApplicationState::MaxPlayerCount + 1; ++i)
    {
        nns::hid::GamePadNx* pGamePadNx = new nns::hid::GamePadNx(&controllerManager);
        pGamePadNx->SetControllerNumber(i);
        controllerManager.GetControllerList().push_back(pGamePadNx);
    }

    g_pControllerManager = &controllerManager;

    //==============================================
    // Memory
    //==============================================

    nn::mem::StandardAllocator appAllocator;
    const size_t appMemorySize = 128 * 1024 * 1024;
    nn::Bit8* pAppMemory = new nn::Bit8[appMemorySize];

    appAllocator.Initialize(pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    const size_t graphicsMemorySize = 8 * 1024 * 1024;
    void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
    nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
    nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif

    //==============================================
    // Graphics
    //==============================================

    GraphicsSystem* pGraphicsSystem = new ::GraphicsSystem();
    g_pGraphicsSystem = pGraphicsSystem;
    pGraphicsSystem->SetApplicationHeap(&appAllocator);
    pGraphicsSystem->Initialize();

    // テクスチャを読み込みます
    InitializeResource(pGraphicsSystem, &appAllocator);
    // サンプラを初期化します
    InitializeSampler(pGraphicsSystem);

    //==============================================
    // Thread
    //==============================================
    g_ThreadManager.Add(new StartSceneThread("PressButton", ThreadFunc));
    g_ThreadManager.Add(new TitleSceneThread("Title", ThreadFunc));
    g_ThreadManager.Add(new GameSceneThread("Game", ThreadFunc));

    g_ThreadManager.CreateThread();

    //==============================================
    // Main Loop
    //==============================================

    g_pControllerManager->Update();

    /*
     * 起動直後に本体装着コントローラーと無線コントローラーのNo1がともに接続されていない場合
     * 一人用コントローラーサポートアプレットの呼び出しを行います
     */
    if (CheckConnectController(CheckPattern::Single::Mask))
    {
        CallControllerSupportApplet(nullptr, 1);
    }

    bool exit = false;

    ThreadState* pThreadState = nullptr;
    NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneStart, &pThreadState));
    pThreadState->Start();

    while (!exit)
    {
        nn::os::LockMutex(&g_Mutex);
        {
            if (ThreadManager::pPrevThread != nullptr)
            {
                pThreadState = nullptr;
                nn::os::WaitThread(ThreadManager::pPrevThread);
                NN_ASSERT(g_ThreadManager.GetThreadState(ThreadManager::pPrevThread, &pThreadState));
                pThreadState->Destroy();
                NN_ASSERT(g_ThreadManager.GetThreadState(ThreadManager::pPrevThread, &pThreadState));
                pThreadState->Create(9);
                if (ThreadManager::pNextThread == nullptr)
                {
                    exit = true;
                }
                else
                {
                    nn::os::StartThread(ThreadManager::pNextThread);
                }
                ThreadManager::pPrevThread = ThreadManager::pNextThread = nullptr;
            }
        }
        nn::os::UnlockMutex(&g_Mutex);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    //==============================================
    // Finalization
    //==============================================

    g_pResTextureFile->Finalize(&pGraphicsSystem->GetDevice());

    // Finalize Graphics
    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    // Finalize Memory
    appAllocator.Finalize();
    delete[] pAppMemory;
}

void ThreadFunc(void* arg)
{
    NN_ASSERT_NOT_NULL(arg);

    ThreadState* pState = static_cast<ThreadState*>(arg);
    NN_LOG("(Thread) %s Start\n", nn::os::GetThreadNamePointer(pState->GetThreadType()));

    pState->Initialize();

    //==============================================
    // Main Loop
    //==============================================
    while (!pState->IsExit())
    {
        pState->Update();
        pState->Draw();
    }

    pState->Finalize();
    FeedOut(arg);
}

void* ReadResource(nn::mem::StandardAllocator* memory, const char* filename)
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile(&hFile, filename, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    result = nn::fs::GetFileSize(&fileSize, hFile);
    NN_ASSERT(result.IsSuccess());

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, hFile, 0, &fileHeader, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == sizeof(nn::util::BinaryFileHeader));
    size_t alignment = fileHeader.GetAlignment();

    void* pBuffer = memory->Allocate(static_cast<size_t>(fileSize), static_cast<size_t>(alignment));
    result = nn::fs::ReadFile(&readSize, hFile, 0, pBuffer, static_cast< size_t >(fileSize));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == static_cast< size_t >(fileSize));

    nn::fs::CloseFile(hFile);

    return pBuffer;
}

void InitializeResource(GraphicsSystem* system, nn::mem::StandardAllocator* memory)
{
    void* g_MountRomCacheBuffer = NULL;
    size_t cacheSize = 0;
    nn::Result result;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    g_MountRomCacheBuffer = malloc(cacheSize);
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    // タイトル画像を読み込みます

    void* pTextureResource = ReadResource(memory, "Contents:/Texture.bntx");
    g_pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTextureResource);
    g_pResTextureFile->Initialize(&system->GetDevice());
    for (int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture)
    {
        g_pResTextureFile->GetResTexture(idxTexture)->Initialize(&system->GetDevice());

    }
    const char* textureFileNames[] = {
        "Title", "StyleIcon", "ButtonIcon"
    };
    for (size_t i = 0; i < NN_ARRAY_SIZE(textureFileNames); ++i)
    {
        ResourceType type = (ResourceType)i;
        g_TextureIndex[type] = g_pResTextureFile->GetTextureDic()->FindIndex(textureFileNames[i]);
        NN_ASSERT(g_TextureIndex[type] != -1);
        system->RegisterTextureViewSlot(&g_TextureDescriptor[type], *static_cast<nn::gfx::TextureView*>(g_pResTextureFile->GetResTexture(g_TextureIndex[type])->GetTextureView()));
    }

    // 直前の画面

    system->RegisterTextureViewSlot(&g_ScanScreenDescriptor, *system->GetScanBufferTextureView());

    nn::fs::Unmount("Contents");

    free(g_MountRomCacheBuffer);
    g_MountRomCacheBuffer = nullptr;
}

void InitializeSampler(GraphicsSystem* system)
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear);
    info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);

    g_Sampler.Initialize(&system->GetDevice(), info);
    system->RegisterSamplerSlot(&::g_SamplerDescriptor, g_Sampler);
}

void DrawStyleIcon(
    const nn::util::Float2& pos,
    const nn::util::Float2& size,
    int32_t iconIndex)
{
    static const nn::util::Float2 textureSize = NN_UTIL_FLOAT_2_INITIALIZER(1024, 256.f);
    static const nn::util::Float2 iconSize = NN_UTIL_FLOAT_2_INITIALIZER(128.f, 128.f);

    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = &(g_pGraphicsSystem->GetPrimitiveRenderer());
    nns::gfx::PrimitiveRenderer::PrimitiveMesh mesh;

    static const nn::util::Float2 ScreenSize = NN_UTIL_FLOAT_2_INITIALIZER(1280, 720);

    static const uint32_t index[6] =
    {
        0, 1, 2, 0, 2, 3
    };

    if (mesh.Initialize(pRenderer->GetGpuBuffer(), 4, 6,
        (nns::gfx::PrimitiveRenderer::VertexFormat)(nns::gfx::PrimitiveRenderer::VertexFormat_Pos | nns::gfx::PrimitiveRenderer::VertexFormat_Color | nns::gfx::PrimitiveRenderer::VertexFormat_Uv)) == false)
    {
        return;
    }

    uint32_t* pIndexData = mesh.GetIndexBufferCpuAddress();

    for (size_t i = 0; i < NN_ARRAY_SIZE(index); ++i)
    {
        pIndexData[i] = index[i];
    }


    nn::util::Float3* pPos = static_cast<nn::util::Float3*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos));
    pPos[0] = nn::util::MakeFloat3(
        -1.f + (2.f * (pos.x)) / ScreenSize.x,
        1.f - (2.f * (pos.y)) / ScreenSize.y,
        0);
    pPos[1] = nn::util::MakeFloat3(
        -1.f + (2.f * (pos.x + size.x)) / ScreenSize.x,
        1.f - (2.f * (pos.y)) / ScreenSize.y,
        0);
    pPos[2] = nn::util::MakeFloat3(
        -1.f + (2.f * (pos.x + size.x)) / ScreenSize.x,
        1.f - (2.f * (pos.y + size.y)) / ScreenSize.y,
        0);
    pPos[3] = nn::util::MakeFloat3(
        -1.f + (2.f * (pos.x)) / ScreenSize.x,
        1.f - (2.f * (pos.y + size.y)) / ScreenSize.y,
        0);

    nn::util::Float4* pColor = static_cast<nn::util::Float4*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Color));
    for (size_t i = 0; i < NN_ARRAY_SIZE(index); ++i)
    {
        pColor[i] = NN_UTIL_FLOAT_4_INITIALIZER(1.f, 1.f, 1.f, 1.f);
    }
    nn::util::Float4 uv = NN_UTIL_FLOAT_4_INITIALIZER
    (
        static_cast<float>(iconIndex * iconSize.x) / textureSize.x,
        nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical ? 0.f : 0.5f,
        static_cast<float>((iconIndex + 1) * iconSize.x) / textureSize.x,
        nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical ? 0.5f : 1.f
    );

    nn::util::Float2* pUv = static_cast<nn::util::Float2*>(mesh.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Uv));
    pUv[0] = NN_UTIL_FLOAT_2_INITIALIZER(uv.x, uv.y);
    pUv[1] = NN_UTIL_FLOAT_2_INITIALIZER(uv.z, uv.y);
    pUv[2] = NN_UTIL_FLOAT_2_INITIALIZER(uv.z, uv.w);
    pUv[3] = NN_UTIL_FLOAT_2_INITIALIZER(uv.x, uv.w);

    pRenderer->DrawUserMesh(
        &(g_pGraphicsSystem->GetCommandBuffer()),
        nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
        &mesh,
        g_TextureDescriptor[ResourceType_StyleIcon], g_SamplerDescriptor
    );
}

void FeedOut(void* arg)
{
    NN_UNUSED(arg);

    for (int i = 0; i < 20; ++i)
    {
        g_pGraphicsSystem->BeginDraw(false);
        {
            g_pGraphicsSystem->GetPrimitiveRenderer().SetColor(nn::util::Color4u8(0, 0, 0, 255 / 5));
            g_pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&g_pGraphicsSystem->GetCommandBuffer(), 0, 0, 1280, 720);
        }
        g_pGraphicsSystem->EndDraw();

        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }
}
