﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// #define VCC
// #define RENDER_6CH

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/init.h>
#include <nn/tma/tma.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_ThreadApi.h>

#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nne/vcc/vcc_API.h>
#include <nne/audio/audio.h>

#define TRACEF(...)
#define TRACEV(...)
#include <audio/include/trace_t.h>

using namespace nn;
using namespace nne::audio;

namespace PlayTest {

//***************************************************************************
//***************************************************************************
//
// Heap
//
//***************************************************************************
//***************************************************************************

namespace {
    const int heapSize = 512 * 1024;
    uint8_t heapBuffer[heapSize];
    lmem::HeapHandle heap;

#ifdef NN_SDK_BUILD_RELEASE
    const bool adspStartLogs = false;
#else
    const bool adspStartLogs = true;
#endif

    os::SemaphoreType audioRendererDoneSemaphore;
    gmix::Session* audioRenderer = nullptr;

    os::SemaphoreType processSemaphore;
    NN_ALIGNAS(4096) char processThreadStack[8192];
    volatile bool processThreadExit = false;

    NN_ALIGNAS(4096) char fileBuffer[8000 * 1024];
    size_t fileSize = 0;
}

//***************************************************************************
// heapAllocate
//***************************************************************************
void* heapAllocate(size_t size)
{
    return lmem::AllocateFromExpHeap(heap, size);
}

//***************************************************************************
// heapDeallocate
//***************************************************************************
void heapDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    lmem::FreeToExpHeap(heap, p);
}


//***************************************************************************
// process
//***************************************************************************
void process(void* arg)
{
    TRACEF("PlayTest", __func__);

    uint32_t bufferSize = audioRenderer->GetBufferSize();
    uint8_t* buffer = (uint8_t*)audioRenderer->GetBufferAddress();
    uint32_t offsetSize = bufferSize / 2;
    uint32_t remainder = 0;
    uint32_t lastOffset = 0;
    uint32_t currentOffset = 0;
    uint32_t amountToCopy = 0;
    int64_t fileOffset = 0;
    int processCallsRemaining = 1;
    bool fileDone = false;
    bool done = false;

    amountToCopy = fileSize - fileOffset < bufferSize ? fileSize : bufferSize;
    fileOffset += amountToCopy;

    TimeSpan startTime, timeSpan;
    startTime = os::ConvertToTimeSpan(os::GetSystemTick());
    memcpy(buffer, fileBuffer, amountToCopy);
    timeSpan = os::ConvertToTimeSpan(os::GetSystemTick());
    TRACEV("[%s] %lldus - memcpy\n", __func__, timeSpan.GetMicroSeconds() - startTime.GetMicroSeconds());

    if (amountToCopy < bufferSize)
    {
        memset(buffer + amountToCopy, 0, bufferSize - amountToCopy);
        if (amountToCopy < offsetSize)
        {
            processCallsRemaining = 0;
        }

        fileDone = true;
    }

    while (!processThreadExit)
    {
        os::AcquireSemaphore(&processSemaphore);
        if (processThreadExit)
        {
            break;
        }

        if (!processCallsRemaining)
        {
            if (!done)
            {
                done = true;
                os::ReleaseSemaphore(&audioRendererDoneSemaphore);
            }
        }

        else if (fileDone)
        {
            processCallsRemaining--;
        }

        if (done)
        {
            continue;
        }

        currentOffset = audioRenderer->GetReadPosition();
        if (currentOffset < lastOffset)
        {
            // wrap
            amountToCopy = bufferSize - lastOffset;
            remainder = fileSize - fileOffset;
            if (amountToCopy < remainder)
            {
                memcpy(buffer + lastOffset, fileBuffer + fileOffset, amountToCopy);
                fileOffset += amountToCopy;
            }

            else
            {
                if (remainder)
                {
                    memcpy(buffer + lastOffset, fileBuffer + fileOffset, remainder);
                    fileOffset += remainder;
                    fileDone = true;
                }

                memset(buffer + lastOffset + remainder, 0, amountToCopy - remainder);
            }

            lastOffset = 0;
        }

        if (lastOffset < currentOffset)
        {
            amountToCopy = currentOffset - lastOffset;
            remainder = fileSize - fileOffset;
            if (amountToCopy < remainder)
            {
                memcpy(buffer + lastOffset, fileBuffer + fileOffset, amountToCopy);
                fileOffset += amountToCopy;
            }

            else
            {
                if (remainder)
                {
                    memcpy(buffer + lastOffset, fileBuffer + fileOffset, remainder);
                    fileOffset += remainder;
                    fileDone = true;
                }

                memset(buffer + lastOffset + remainder, 0, amountToCopy - remainder);
                TRACEV("zfill #1\n");
            }

            lastOffset = currentOffset;
        }
    }
}


//***************************************************************************
// loadFile
//***************************************************************************
void loadFile()
{
    TRACEF("PlayTest", __func__);

//    #define FILE_NAME   "c:/HosHostFs/Media/sweep_48kHz_16bit_stereo.snd"
#if defined(RENDER_6CH)
    const char* FILE_NAME = "c:/HosHostFs/Media/6chpcmtest.snd";
#else
    const char* FILE_NAME = "c:/HosHostFs/Media/who_let_the_dogs_out.snd";
#endif

    TRACEV("[%s] Opening file %s\n", __func__, FILE_NAME);
    fs::FileHandle audioFilehandle;

    Result result = fs::MountHostRoot();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    TimeSpan startTime, timeSpan;
    startTime = os::ConvertToTimeSpan(os::GetSystemTick());
    result = fs::OpenFile(&audioFilehandle, FILE_NAME, fs::OpenMode_Read);
    timeSpan = os::ConvertToTimeSpan(os::GetSystemTick());
    TRACEV("[%s] fs::OpenFile took %lldus - \n", __func__, timeSpan.GetMicroSeconds() - startTime.GetMicroSeconds());
    NN_ASSERT(result.IsSuccess());

    startTime = os::ConvertToTimeSpan(os::GetSystemTick());
    result = fs::ReadFile(&fileSize, audioFilehandle, 0, fileBuffer, sizeof(fileBuffer));
    timeSpan = os::ConvertToTimeSpan(os::GetSystemTick());
    TRACEV("[%s] fs::ReadFile took %lldus - \n", __func__, timeSpan.GetMicroSeconds() - startTime.GetMicroSeconds());
    NN_ASSERT(result.IsSuccess());

    fs::CloseFile(audioFilehandle);

    fs::UnmountHostRoot();
}


//***************************************************************************
//***************************************************************************
//
// External Entry Points
//
//***************************************************************************
//***************************************************************************

const size_t HeapSize = 256 * 1024 * 1024;

extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 16 * 1024 * 1024;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(HeapSize);
    if (!result.IsSuccess())
    {
        NN_LOG("SetMemoryHeapSize failed.\n");
        return;
    }

    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void *>(address), MallocMemorySize);
}

//***************************************************************************
// nnMain
//***************************************************************************

extern "C" void nnMain()
{
    os::ThreadType processThread;

    tma::Initialize();

    TRACEF("HdaPlayTest", __func__);

    int argc = os::GetHostArgc();
    NN_UNUSED(argc);

    char** argv = os::GetHostArgv();
    NN_UNUSED(argv);

    heap = lmem::CreateExpHeap(heapBuffer, heapSize, lmem::CreationOption_NoOption);
    fs::SetAllocator(heapAllocate, heapDeallocate);

    uintptr_t adspBin, vectorBin;
    size_t adspSize, vectorSize;
    firmware::Initialize(&adspBin, &adspSize, &vectorBin, &vectorSize);

#if defined(VCC)
    nne::vcc::Initialize();
#endif
    adsp::Initialize(adspBin, adspSize, vectorBin, vectorSize, adspStartLogs);

    TimeSpan startTime, timeSpan;
    startTime = os::ConvertToTimeSpan(os::GetSystemTick());

    loadFile();

    os::InitializeSemaphore(&audioRendererDoneSemaphore, 0, 1);

    // Could have been two in a no print world...
//    os::InitializeSemaphore(&processSemaphore, 0, 2);
    os::InitializeSemaphore(&processSemaphore, 0, 1000);

    gmix::Initialize(&processSemaphore);
#if defined(RENDER_6CH)
    gmix::OpenSession(&audioRenderer, gmix::Session::Name::AudioRenderer, 0, gmix::Session::Format::Surround);
#else
    gmix::OpenSession(&audioRenderer, gmix::Session::Name::AudioRenderer, 0, gmix::Session::Format::Stereo);
#endif
    auto result = os::CreateThread(&processThread, process,
                                   nullptr,
                                   processThreadStack,
                                   sizeof(processThreadStack),
                                   os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess());
    os::StartThread(&processThread);

    audioRenderer->Start();

    device::Session* play = nullptr;
    ahub::Initialize();
    ahub::OpenSession(&play, device::Play);

    device::Session *playHda = nullptr;
#if defined(RENDER_6CH)
    hda::Initialize(hda::AUDIO_HDA_FORMAT_LPCM, hda::AUDIO_HDA_SAMPLERATE_48000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_6);
#else
    hda::Initialize(hda::AUDIO_HDA_FORMAT_LPCM, hda::AUDIO_HDA_SAMPLERATE_48000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_2);
#endif
    hda::OpenSession(&playHda, device::Play);

    playHda->Start();
    os::SleepThread(TimeSpan::FromSeconds(2));
    play->Start();

    gmix::StartProcessTimer();

    while (1)
    {
        if (os::TryAcquireSemaphore(&audioRendererDoneSemaphore))
        {
            break;
        }

        os::SleepThread(TimeSpan::FromMilliSeconds(20));
    }

    audioRenderer->Stop();
    gmix::CloseSession(audioRenderer);

    playHda->Stop();
    play->Stop();

    gmix::StopProcessTimer();

    ahub::CloseSession(play);
    ahub::Finalize();

    hda::CloseSession(playHda);
    hda::Finalize();

    processThreadExit = true;
    os::ReleaseSemaphore(&processSemaphore);
    os::WaitThread(&processThread);
    os::DestroyThread(&processThread);

    gmix::Stats();
    gmix::Finalize();

    os::FinalizeSemaphore(&processSemaphore);
    os::FinalizeSemaphore(&audioRendererDoneSemaphore);

    adsp::Finalize();
    firmware::Finalize();

    lmem::DestroyExpHeap(heap);

    timeSpan = os::ConvertToTimeSpan(os::GetSystemTick());
    TRACEV("[%s] %lldms - Total application time\n", __func__, timeSpan.GetMilliSeconds() - startTime.GetMilliSeconds());

    tma::Finalize();
}

}
